/******************************************************************************
 * mod_uploader / upload-item.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not bcktuired.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: upload-item.cpp 631 2005-09-07 12:50:24Z svn $
 *****************************************************************************/

#define __STDC_CONSTANT_MACROS

#include "upload-item.h"
#include "Auxiliary.h"

#include "apr_pools.h"
#include "apr_time.h"

static VALUE list_total_file_size(VALUE self);
static VALUE reader_alloc(VALUE klass);
static void reader_free(reader_impl *impl);
static VALUE reader_initialize(VALUE self, VALUE file_dir);
static VALUE reader_read(VALUE self, VALUE file_name);
static VALUE item_file_size(VALUE self);
static VALUE item_time(VALUE self);

#define define_method(klass, name, func, argc)                      \
    rb_define_method(rb_eval_string(klass), name,                   \
                     reinterpret_cast<VALUE (*)(...)>(func), argc);


/******************************************************************************
 * 
 *****************************************************************************/
void Init_UploadItem(void)
{
    apr_initialize();

    rb_define_alloc_func(rb_eval_string("UploadItemReader"), reader_alloc);

    rb_define_private_method(rb_eval_string("UploadItemReader"), "initialize",
                             reinterpret_cast<VALUE (*)(...)>(reader_initialize), 1);

    define_method("UploadItemList", "total_file_size", list_total_file_size,  0);
    define_method("UploadItemReader", "read", reader_read, 1);

    define_method("UploadItem", "file_size", item_file_size, 0);
    define_method("UploadItem", "time", item_time, 0);
}


/******************************************************************************
 * UploadItemList
 *****************************************************************************/
VALUE list_total_file_size(VALUE self)
{
    VALUE total_size;
    apr_pool_t *pool;

    total_size = rb_funcall(self, rb_intern("total_file_size_impl"), 0);

    if (apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        exit(EXIT_FAILURE); // :-<
    }

    total_size = rb_str_new2(size_str(pool, NUM2UINT(total_size)));

    apr_pool_clear(pool);

    return total_size;
}


/******************************************************************************
 * UploadItemReader
 *****************************************************************************/
VALUE reader_alloc(VALUE klass)
{
    reader_impl *impl = ALLOC(reader_impl);

    impl->instance  = NULL;

    return Data_Wrap_Struct(klass, 0, reader_free, impl);
}

void reader_free(reader_impl *impl)
{
    delete impl->instance;

    free(impl);
}

VALUE reader_initialize(VALUE self, VALUE file_dir)
{
    reader_impl *impl;

    Data_Get_Struct(self, reader_impl, impl);

    impl->instance = new UploadItemReader(NULL, StringValueCStr(file_dir));

    return Qnil;
}

VALUE reader_read(VALUE self, VALUE file_path)
{
    reader_impl *impl;
    apr_file_t *file;
    UploadItem::header *header;
    VALUE item;
    apr_pool_t *pool;

    Data_Get_Struct(self, reader_impl, impl);

    if (apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        exit(EXIT_FAILURE); // :-<
    }

    try {
        header = impl->instance->read(StringValueCStr(file_path), &file, pool);
    } catch(const char *message) {
        apr_pool_clear(pool);
        rb_raise(rb_eval_string("Exception"), "%s", message);
    }

    item = rb_funcall(rb_eval_string("UploadItem"), rb_intern("new"), 8,
                      file_path,
                      rb_str_new2(header->file_name),
                      rb_str_new2(header->file_mime),
                      UINT2NUM(header->file_size),
                      UINT2NUM(header->time/APR_USEC_PER_SEC),
                      rb_str_new2(header->comment),
                      rb_str_new2(header->remove_pass),
                      rb_str_new2(header->download_pass));

    apr_pool_clear(pool);

    return item;
}


/******************************************************************************
 * UploadItem
 *****************************************************************************/
VALUE item_file_size(VALUE self)
{
    VALUE file_size;
    apr_pool_t *pool;

    if (apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        exit(EXIT_FAILURE); // :-<
    }

    file_size = rb_str_new2(size_str(pool,
                                     NUM2UINT(rb_ivar_get(self, rb_intern("@file_size")))));

    apr_pool_clear(pool);

    return file_size;
}

VALUE item_time(VALUE self)
{
    VALUE time;
    apr_pool_t *pool;
    apr_time_exp_t time_exp;
    char *time_str;
    apr_size_t size;

    if (apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        exit(EXIT_FAILURE); // :-<
    }

    time_str = reinterpret_cast<char *>(apr_palloc(pool,
                                                   sizeof(char)*UITEM_MAX_TIME_LENGTH));
    if (time_str == NULL) {
        apr_pool_clear(pool);
        rb_raise(rb_eval_string("Exception"),
                 "%s", "ɽΤΥݤ˼Ԥޤ");
    }

    apr_time_exp_tz(&time_exp,
                    NUM2UINT(rb_ivar_get(self, rb_intern("@time")))*APR_USEC_PER_SEC,
                    UITEM_TIME_ZONE_OFFSET);
    apr_strftime(time_str, &size, UITEM_MAX_TIME_LENGTH, UITEM_TIME_FORMAT, &time_exp);
    time = rb_str_new2(time_str);

    apr_pool_clear(pool);

    return time;
}

/*
 * Local Variables:
 * mode: c++
 * buffer-file-coding-system: euc-japan-dos
 * End:
 */
