/* mg-work-layout.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-work-layout.h"
#include "mg-work-widget.h"
#include "mg-custom-layout.h"
#include "mg-referer.h"
#include "mg-target.h"
#include "mg-db-table.h"
#include "mg-work-grid.h"
#include "mg-work-form.h"
#include "mg-work-matrix.h"
#include "mg-qfield.h"

static void mg_work_layout_class_init (MgWorkLayoutClass * class);
static void mg_work_layout_init (MgWorkLayout * wid);
static void mg_work_layout_dispose (GObject   * object);

static void mg_work_layout_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void mg_work_layout_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void mg_work_layout_initialize (MgWorkLayout *form);

static void nullified_custom_layout_cb (MgCustomLayout *layout, MgWorkLayout *form);

/* MgWorkWidget interface */
static void            mg_work_layout_widget_init         (MgWorkWidgetIface *iface);
static void            mg_work_layout_run                 (MgWorkWidget *iface, guint mode);
static void            mg_work_layout_show_global_actions (MgWorkWidget *iface, gboolean show_actions);
static MgParameter    *mg_work_layout_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, 
							   gboolean in_exec_context);
static MgContext      *mg_work_layout_get_exec_context    (MgWorkWidget *iface);

struct _MgWorkLayoutPrivate {
	MgCustomLayout  *layout;

	/* global context */
	MgContext       *exec_context; /* garbage of parameters not dependant on each others */

	/* sub layouts */
	GSList          *work_widgets; /* list of MgWorkLayout widgets */

	/* glade layout */
	GladeXML        *glade_instance;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* properties */
enum
{
        PROP_0,
};

guint
mg_work_layout_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgWorkLayoutClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_work_layout_class_init,
			NULL,
			NULL,
			sizeof (MgWorkLayout),
			0,
			(GInstanceInitFunc) mg_work_layout_init
		};		
		
		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) mg_work_layout_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "MgWorkLayout", &info, 0);
		g_type_add_interface_static (type, MG_WORK_WIDGET_TYPE, &work_widget_info);
	}

	return type;
}

static void
mg_work_layout_widget_init (MgWorkWidgetIface *iface)
{
	iface->run = mg_work_layout_run;
	iface->set_mode = NULL;
	iface->set_entry_editable = NULL;
	iface->show_entry_actions = NULL;
	iface->show_global_actions = mg_work_layout_show_global_actions;
	iface->get_param_for_field = mg_work_layout_get_param_for_field;
        iface->has_been_changed = NULL;
	iface->get_exec_context = mg_work_layout_get_exec_context;
	iface->get_actions_group = NULL;
}


static void
mg_work_layout_class_init (MgWorkLayoutClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = mg_work_layout_dispose;
	
	/* Properties */
        object_class->set_property = mg_work_layout_set_property;
        object_class->get_property = mg_work_layout_get_property;
}

static void
mg_work_layout_init (MgWorkLayout * wid)
{
	wid->priv = g_new0 (MgWorkLayoutPrivate, 1);
}

/**
 * mg_work_layout_new
 * @layout: a #MgWorkLayout object
 *
 * Creates a new #MgWorkLayout widget.
 *
 * Returns: the new widget
 */
GtkWidget *
mg_work_layout_new (MgCustomLayout *layout)
{
	GObject *obj;
	MgWorkLayout *wl;

	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);

	obj = g_object_new (MG_WORK_LAYOUT_TYPE, NULL);
	wl = MG_WORK_LAYOUT (obj);

	wl->priv->layout = layout;
	g_object_ref (layout);
	g_signal_connect (G_OBJECT (wl->priv->layout), "nullified",
			  G_CALLBACK (nullified_custom_layout_cb), wl);

	mg_work_layout_initialize (wl);

	return GTK_WIDGET (obj);
}

static void
nullified_custom_layout_cb (MgCustomLayout *layout, MgWorkLayout *wl)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
					      G_CALLBACK (nullified_custom_layout_cb), wl);
	
	g_object_unref (G_OBJECT (wl->priv->layout));
	wl->priv->layout = NULL;
}

static void
mg_work_layout_dispose (GObject *object)
{
	MgWorkLayout *wl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_WORK_LAYOUT (object));
	wl = MG_WORK_LAYOUT (object);

	if (wl->priv) {
		/* core */
		if (wl->priv->layout)
			nullified_custom_layout_cb (wl->priv->layout, wl);

		if (wl->priv->exec_context)
			g_object_unref (wl->priv->exec_context);

		g_slist_free (wl->priv->work_widgets);

		if (wl->priv->glade_instance)
			g_object_unref (wl->priv->glade_instance);

		/* the private area itself */
		g_free (wl->priv);
		wl->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
mg_work_layout_set_property (GObject              *object,
			     guint                 param_id,
			     const GValue         *value,
			     GParamSpec           *pspec)
{
	MgWorkLayout *wl;

        wl = MG_WORK_LAYOUT (object);
        if (wl->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
mg_work_layout_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	MgWorkLayout *wl;

        wl = MG_WORK_LAYOUT (object);
        if (wl->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}



/*
 * Real initialization
 */
static void init_work_form_grid (MgWorkLayout *wl);
static void init_work_matrix (MgWorkLayout *wl);
static void init_super_layout (MgWorkLayout *wl);
static void 
mg_work_layout_initialize (MgWorkLayout *wl)
{
	if (! mg_referer_activate (MG_REFERER (wl->priv->layout))) {
		GtkWidget *wid = gtk_label_new (_("missing required objects!"));
		gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
	}
	else {
		switch (mg_custom_layout_get_layout_type (wl->priv->layout)) {
		case MG_CUSTOM_LAYOUT_LAYOUT:
			init_super_layout (wl);
			break;
		case MG_CUSTOM_LAYOUT_GRID:
		case MG_CUSTOM_LAYOUT_FORM:
			init_work_form_grid (wl);
			break;
		case MG_CUSTOM_LAYOUT_MATRIX:
			init_work_matrix (wl);
			break;
		}
	}
}

static void
init_work_form_grid (MgWorkLayout *wl)
{
	GtkWidget *wid, *root;
	MgCustomLayout *cl = wl->priv->layout;
	GladeXML *glade;
	GHashTable *hash = NULL;
	GError *error = NULL;
	MgCustomLayoutData *cldata;
	MgTarget *target = NULL;
	gboolean err = FALSE;

	glade = mg_custom_layout_get_glade_instance (cl, &root, &hash, &error);
	wl->priv->glade_instance = glade;
	if (error) {
		g_warning ("Error building GladeXML file: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	cldata = mg_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using MgCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	/* testing */
	if (!cldata->contents.work_iface.query ||
	    ! mg_query_is_select_query (cldata->contents.work_iface.query))
		err = TRUE;

	if (!err) {
		target = (MgTarget *) cldata->contents.work_iface.modified;
		if (!IS_MG_TARGET (target) ||
		    (mg_target_get_query (target) != cldata->contents.work_iface.query)) 
			target = NULL;
	}

	if (! err) {
		/* actual widget creation */
		if (cldata->type == MG_CUSTOM_LAYOUT_GRID) 
			wid = mg_work_grid_new (cldata->contents.work_iface.query, target);
		else 
			wid = mg_work_form_new_in_layout (cldata->contents.work_iface.query, target, root, hash);
		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		mg_work_widget_set_mode (MG_WORK_WIDGET (wid), cldata->contents.work_iface.mode);
		g_object_set (G_OBJECT (wid), "title_visible", FALSE, NULL);
	}
	else 
		wid = gtk_label_new (_("Error creating widget"));
	
	gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	if (hash)
		g_hash_table_destroy (hash);
	g_free (cldata);
}

static void
init_work_matrix (MgWorkLayout *wl)
{
	GtkWidget *wid;
	MgConf *conf = mg_base_get_conf (MG_BASE (wl->priv->layout));
	MgCustomLayout *cl = wl->priv->layout;
	MgCustomLayoutData *cldata;
	MgTarget *target;
	MgDbTable *table;
	gboolean err = FALSE;
	GError *error = NULL;
	
	cldata = mg_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using MgCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}

	
	/* tests */
	table = (MgDbTable *) cldata->contents.work_iface.modified;
	if (!IS_MG_DB_TABLE (table))
		err = TRUE;

	if (!cldata->contents.work_iface.query ||
	    ! mg_query_is_select_query (cldata->contents.work_iface.query))
		err = TRUE;

	if (!cldata->contents.work_iface.query_extra ||
	    ! mg_query_is_select_query (cldata->contents.work_iface.query_extra))
		err = TRUE;


	if (!err) {
		target = cldata->contents.work_iface.rows_target;
		if (!IS_MG_TARGET (target) ||
		    (mg_target_get_query (target) != cldata->contents.work_iface.query)) 
			err = TRUE;
	}

	if (!err) {
		target = cldata->contents.work_iface.cols_target;
		if (!IS_MG_TARGET (target) ||
		    (mg_target_get_query (target) != cldata->contents.work_iface.query_extra)) 
			err = TRUE;
	}
	    
	/* actual widget creation */
	if (!err) {
		wid = mg_work_matrix_new (conf,
					  cldata->contents.work_iface.query,
					  cldata->contents.work_iface.rows_target,
					  cldata->contents.work_iface.query_extra,
					  cldata->contents.work_iface.cols_target,
					  (MgDbTable *) cldata->contents.work_iface.modified,
					  NULL);
		mg_work_matrix_set_view_type (MG_WORK_MATRIX (wid), cldata->contents.work_iface.view_type);
		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		mg_work_widget_set_mode (MG_WORK_WIDGET (wid), cldata->contents.work_iface.mode);
		g_object_set (G_OBJECT (wid), "title_visible", FALSE, NULL);
	}
	else
		wid = gtk_label_new (_("Error creating widget"));

	gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	g_free (cldata);
}

static MgWorkLayout *
find_by_custom_layout (MgWorkLayout *super_layout, MgCustomLayout *cl)
{
	MgWorkLayout *retval = NULL;
	GSList *list = super_layout->priv->work_widgets;
	while (list && !retval) {
		if (MG_WORK_LAYOUT (list->data)->priv->layout == cl)
			retval = MG_WORK_LAYOUT (list->data);
		else
			list = g_slist_next (list);
	}

	return retval;
}

static void
init_super_layout (MgWorkLayout *wl)
{
	GtkWidget *wid, *box, *root;
	MgCustomLayout *cl = wl->priv->layout;
	GladeXML *glade;
	GHashTable *hash = NULL;
	GError *error = NULL;
	MgCustomLayoutData *cldata;
	GSList *list;

	glade = mg_custom_layout_get_glade_instance (cl, &root, &hash, &error);
	wl->priv->glade_instance = glade;
	if (error) {
		g_warning ("Error building GladeXML file: %s", error->message);
		g_error_free (error);
		error = NULL;
	}

	if (!root) {
		/* If there is no Glade layout, or if there was an error, then we use
		   a table layout */
		TO_IMPLEMENT;
	}
	
	cldata = mg_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using MgCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	/* actual widget creation */
	list = cldata->contents.layout.children;
	while (list) {
		wid = mg_work_layout_new (MG_CUSTOM_LAYOUT (list->data));

		if (hash) {
			box = g_hash_table_lookup (hash, list->data);
			gtk_box_pack_start (GTK_BOX (box), wid, TRUE, TRUE, 0);
			gtk_widget_show (wid);
			if (! g_object_get_data (G_OBJECT (box), "show_actions")) 
				mg_work_widget_alldata_show_actions (MG_WORK_WIDGET (wid), FALSE);
		}
		else {
			TO_IMPLEMENT;
		}

		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		list = g_slist_next (list);
	}

	list = cldata->contents.layout.connects;
	while (list) {
		MgCustomLayoutConnect *lc = (MgCustomLayoutConnect*) list->data;
		MgWorkWidget *src, *dest;
		MgParameter *param_src, *param_dest;
		MgWorkLayout *tmp;

		tmp = find_by_custom_layout (wl, lc->src_layout);
		g_assert (tmp->priv && (g_slist_length (tmp->priv->work_widgets) == 1));
		src = MG_WORK_WIDGET (tmp->priv->work_widgets->data);

		tmp = find_by_custom_layout (wl, lc->dest_layout);
		g_assert (tmp->priv && (g_slist_length (tmp->priv->work_widgets) == 1));
		dest = MG_WORK_WIDGET (tmp->priv->work_widgets->data);

		param_src = mg_work_widget_get_param_for_field_data (src, MG_QFIELD (lc->src_field));
		if (!param_src) 
			g_warning (_("Cannot find a parameter for source field connection"));
		param_dest = mg_work_widget_get_param_for_field_exec (dest, MG_QFIELD (lc->dest_field));
		if (!param_dest) 
			g_warning (_("Cannot find a parameter for destination field connection"));
		
		if (param_src && param_dest)
			mg_parameter_bind_to_param (param_dest, param_src);
		
		list = g_slist_next (list);
	}
	
	gtk_box_pack_start (GTK_BOX (wl), root, TRUE, TRUE, 0);
	gtk_widget_show (root);
	if (hash)
		g_hash_table_destroy (hash);
	g_free (cldata);
}

/**
 * mg_work_layout_lookup_widget
 * @layout: a #MgWorkLayout widget
 * @widget_name: a string
 *
 * Retreives the #GtkWidget which name is @widget_name, if @layout
 * uses a Glade file as layout.
 *
 * Returns: the requested #GtkWidget, or %NULL if not found
 */
GtkWidget *
mg_work_layout_lookup_widget (MgWorkLayout *layout, const gchar *widget_name)
{
	GtkWidget *retval = NULL;

	g_return_val_if_fail (layout && IS_MG_WORK_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	if (layout->priv->glade_instance)
		retval = glade_xml_get_widget (layout->priv->glade_instance, widget_name);

	return retval;
}

/*
 * MgWorkWidget interface implementation
 */
static void
mg_work_layout_run (MgWorkWidget *iface, guint mode)
{
	GSList *list;
	MgWorkLayout *wl;

	g_return_if_fail (iface && IS_MG_WORK_LAYOUT (iface));
	wl = MG_WORK_LAYOUT (iface);
	g_return_if_fail (wl->priv);

	list = wl->priv->work_widgets;
	while (list) {
		mg_work_widget_run (MG_WORK_WIDGET (list->data), 0); /* don't change previous mode */
		list = g_slist_next (list);
	}
}

static void
mg_work_layout_show_global_actions (MgWorkWidget *iface, gboolean show_actions)
{
	GSList *list;
	MgWorkLayout *wl;

	g_return_if_fail (iface && IS_MG_WORK_LAYOUT (iface));
	wl = MG_WORK_LAYOUT (iface);
	g_return_if_fail (wl->priv);

	list = wl->priv->work_widgets;
	while (list) {
		mg_work_widget_alldata_show_actions (MG_WORK_WIDGET (list->data), show_actions);
		list = g_slist_next (list);
	}
}

static MgParameter *
mg_work_layout_get_param_for_field (MgWorkWidget *iface, MgQfield *field, const gchar *field_name, gboolean in_exec_context)
{
	MgWorkLayout *wl;
	MgParameter *param = NULL;

	g_return_val_if_fail (iface && IS_MG_WORK_LAYOUT (iface), NULL);
	wl = MG_WORK_LAYOUT (iface);
	g_return_val_if_fail (wl->priv, NULL);
	g_return_val_if_fail (field || (field_name && *field_name), NULL);
	
	TO_IMPLEMENT;
	return param;
}


static MgContext *
mg_work_layout_get_exec_context (MgWorkWidget *iface)
{
	MgWorkLayout *wl;

	g_return_val_if_fail (iface && IS_MG_WORK_LAYOUT (iface), NULL);
	wl = MG_WORK_LAYOUT (iface);
	g_return_val_if_fail (wl->priv, NULL);
	
	return wl->priv->exec_context;
}
