/*
 * Copyright (C) 1997 and 1998 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: m17n.c,v 1.5 2004/08/19 16:28:57 nishida Exp $ 
 */

#ifdef USE_M17N
#include "mgp.h"
#define M17N_MAX_CACHE 100

static void M17N_format_line();

static MDrawControl control;
static MText *mt;
static MFace *face_size;
static MFace *face_color;
static MFace *face_other;
static MSymbol lang;
/* M-text containing a single space.  It is used to get normal
   ascent/descent of an ASCII font selected for the current set of
   faces.  */
static MText *space;
static struct render_state *current_state;
static int current_linewidth; 
static int current_indent; 

static int kinsoku_range [][2] = {
	0x3041, 0x30FF,		/* Kana */
	0x2E80, 0x2FDF,		/* radical */
	0x3400, 0x4DB5,		/* CJK Ideograph Extention A */
	0x4E00, 0x9FAF,		/* CJK Ideograph */  
	0XF900, 0xFAFF,		/* CJK COMPATIBILITY IDEOGRAPH */
	0X20000, 0x2A6D6,	/* CJK Ideograph Extention B */
	0x2F800, 0x2FA1D	/* CJK COMPATIBILITY IDEOGRAPH */		
};

static int kinsoku_list [] = {
/* UNICODE */
0x2019, 0x201A, 0x201D, 0x201E, 0x2032, 0x2033, 0x2034, 0x203A,
0x203C, 0x203D, 0x2046, 0x2047, 0x2048, 0x2049, 0x204D, 0x3000,
0x3001, 0x3002, 0x3009, 0x300B, 0x300D, 0x300F, 0x3011, 0x3015,
0x3017, 0x3018, 0x309B, 0x309C, 0x309D, 0x309E, 0x30FD, 0x30FD,
0xFF01, 0xFF09, 0xFF0C, 0xFF0E, 0xFF1A, 0xFF1B, 0xFF1F, 0xFF3D,
0xFF5D, 0xFF60, 0xFF61, 0xFF63, 0xFF64,
/* JISX208 */
0x00A7, 0x00B0, 0x2018, 0x201C, 0x2032, 0x2033, 0x2103, 0x3008,
0x300A, 0x300C, 0x300E, 0x3010, 0x3014, 0x3016, 0xFF02, 0xFF08,
0xFF20, 0xFF3B, 0xFF5B
};

static MCharTable *kinsoku_table;
static MSymbol Mkinsoku;

static int
m17n_line_break (mt, pos, from, to, line, y)
     MText *mt;
     int pos, from, to;
     int line, y;
{
	int c, i = 0, opos = pos;
	c = mtext_ref_char(mt, pos);
	if (mchartable_lookup(kinsoku_table, c) == Mt) return pos;	
	while(pos != from){
		if (i ++ > 10) return opos; /* for stupidly long line */	
		pos --;
		c = mtext_ref_char(mt, pos);
		if (c < 256 && isspace(c)) return pos +1;	
		if (mchartable_lookup(kinsoku_table, c) == Mt) return pos;	
	}

	return pos;
}

void
M17N_init()
{
	int i;
	M17N_INIT();

	space = mtext_from_data (" ", 1, MTEXT_FORMAT_US_ASCII);
#if 0
	face_other = mface ();
	// default fontset is not truetype
//	mface_put_prop (face_other, Mfontset, mfontset ("truetype"));
	mface_put_prop (face_other, Mfontset, mfontset ("default"));
#endif
	memset (&control, 0, sizeof control);
	control.two_dimensional = 1;
	control.enable_bidi = 1;
	control.ignore_formatting_char = 1;
	control.disable_caching = 0;
	control.max_line_width = 0;
	control.anti_alias = 1;
	control.format = M17N_format_line; 
//	control.line_break = mdraw_default_line_break; 
	control.line_break = m17n_line_break; 

	/* Generate kinsoku_char_table.  */
	Mkinsoku = msymbol ("kinsoku");
	kinsoku_table = mchartable (Msymbol, Mnil);
	for (i = 0; i < (sizeof(kinsoku_range) / sizeof(kinsoku_range[0])); i++)
		mchartable_set_range (kinsoku_table, kinsoku_range[i][0],
			kinsoku_range[i][1], Mt);
	for (i = 0; i < (sizeof kinsoku_list / sizeof (int)); i++)
		mchartable_set (kinsoku_table, kinsoku_list[i], Mkinsoku);
}

#define MAXCACHEDMTEXT 40
MText *
M17N_gen_mtext (p)
	u_char *p;
{
	static MText *cached_mt[MAXCACHEDMTEXT]; 
	static u_char *cached_txt[MAXCACHEDMTEXT][BUFSIZ]; 
	static int cached_txt_len[MAXCACHEDMTEXT]; 
	static int cached_count;
	MSymbol coding;
	MText *mt = 0;
	int i, len = strlen (p);

	if (len == 0) return;
	for (i = 0; i < cached_count; i ++)
		if ((len == cached_txt_len[i]) && 
				!memcmp((void *)p, (void *)cached_txt[i], cached_txt_len[i])){
			m17n_object_ref(cached_mt[i]);
		   	return cached_mt[i];
		}

	if (strchr (p, '\033')) {
		/* Try ISO-2022 encoding.  */
		coding = mconv_resolve_coding (msymbol ("iso-2022-7bit"));
		if (coding) mt = mconv_decode_buffer (coding, p, len);
	}
	if (!mt) {
		/* Try UTF-8.  */
		coding = mconv_resolve_coding (msymbol ("utf-8"));
		if (coding) mt = mconv_decode_buffer (coding, p, len);
	}
	if (!mt){
		/* Try the encoding of the current locale.  */
		mt = mconv_decode_buffer (Mnil, p, len);
	}

	bzero(cached_txt[cached_count], BUFSIZ);
	if (cached_mt[cached_count]) 
		m17n_object_unref(cached_mt[cached_count]);
	strcpy((char *)cached_txt[cached_count], p); 
	cached_txt_len[cached_count] = strlen(p);
	cached_mt[cached_count] = mt;   	
	cached_count = (cached_count + 1) % MAXCACHEDMTEXT;	
	m17n_object_ref(mt);

	return mt;
}

void
M17N_set_size(size)
	int size;
{
	static MFace *msize[M17N_MAX_CACHE];
	static int msize_num = 0;
	int i, targetsize;
	int delta, mindelta, mindelta_i;

	targetsize = size * 10;

	mindelta = -1;
	for (i = 0; i < msize_num; i ++){
		if (msize[i]){
			delta = abs((int)mface_get_prop(msize[i], Msize) - targetsize);
			if (!delta){
				face_size = msize[i];
				return;
			} else {
				if (mindelta < 0){ mindelta = delta; mindelta_i = i; }
				else if (mindelta > delta) { mindelta = delta; mindelta_i = i; }
			}
		}
	}

	/* If msize_num exceeds limit, choose the closest size */
	if (msize_num == M17N_MAX_CACHE) {
		face_size = msize[mindelta_i];
		return;
	}

	/* create new face */
	msize[msize_num] = mface();
	mface_put_prop (msize[msize_num], Msize, (void *) targetsize);
	face_size = msize[msize_num];
	msize_num ++;
	return;
}

void
M17N_set_color(color)
	u_long color;
{
	static u_long mcolor_val[M17N_MAX_CACHE];
	static MFace *mcolor[M17N_MAX_CACHE];
	static int mcolor_num = 0;
	int i, delta, mindelta, mindelta_i;
	char color_string[20];
	XColor xcolor;

	for (i = 0; i < mcolor_num; i ++){
		delta = abs(mcolor_val[i] - color);
		if (!delta){
			face_color = mcolor[i];
			return;
		} else {
			if (mindelta < 0){ mindelta = delta; mindelta_i = i; }
			else if (mindelta > delta) { mindelta = delta; mindelta_i = i; }
		}
	}

	/* If mcolor_num exceeds limit, choose the closest color */
	if (mcolor_num == M17N_MAX_CACHE) {
		face_color = mcolor[mindelta_i];
		return;
	}

	/* create new face */
	xcolor.pixel = color;  
	XQueryColor(display, colormap, &xcolor);
	sprintf(color_string, "#%04X%04X%04X\0", xcolor.red, xcolor.green, xcolor.blue); 
	mcolor[mcolor_num] = mface();
	mcolor_val[mcolor_num] = color;
	mface_put_prop(mcolor[mcolor_num], Mforeground, msymbol(color_string));
	face_color = mcolor[mcolor_num];
	mcolor_num ++;
	return;
}

char *
M17N_draw_fragment(state, p, len)
	struct render_state *state;
	u_char *p;
	u_int len;
{
	static MFrame *drawframe;
	static MFace  *drawface;
	static MPlist *plist;
	MFace *faces[256];
	MText *mt;
	MDrawGlyphInfo info;
	MDrawMetric rect;
	int ascent = 0, descent = 0, i, n, flheight = 0;
	char tmptext[10000];
	Bool flag_s, flag_c, flag_o;

	if (!drawface) drawface = mface();
	if (!plist){
		plist = mplist ();
		mplist_add(plist, Mdisplay, display);
		drawframe = mframe (plist);
	}
	M17N_set_size(char_size[caching]);
	M17N_set_color(fore_color[caching]); 

	if (len >= 10000) len = 9999; 
	strncpy(tmptext, p, len);
	tmptext[len] = 0;
	mt = M17N_gen_mtext(tmptext);

	/* check if the faces attached Mface are already cached */
	flag_s = flag_c = flag_o = True;
	n = mtext_get_prop_values (mt, 0, Mface, (void **) faces, 256);
	for (i = 0; i < n; i ++){
		if (faces[i] == face_size)  flag_s = False;
		if (faces[i] == face_color) flag_c = False;
		if (faces[i] == face_other) flag_o = False;
	}

	/* if faces are not cached, push them to Mface */
	if (flag_s) mtext_push_prop (mt, 0, mtext_len(mt), Mface, face_size);
	if (flag_c) mtext_push_prop (mt, 0, mtext_len(mt), Mface, face_color);
	if (flag_o) mtext_push_prop (mt, 0, mtext_len(mt), Mface, face_other);
	if (lang) mtext_push_prop (mt, 0, mtext_len(mt), Mlanguage, lang);

	current_linewidth = state->width - state->leftfillpos / 2 - state->linewidth;
	current_indent = state->leftfillpos;

	/* Compare the normal ascent/descent and the physical
	   ascent/descent of this M-text, and use the bigger ones.  */
	mdraw_text_extents(drawframe, space, 0, 1,
			   &control, NULL, &rect, NULL);
	ascent = - rect.y, descent = rect.height + rect.y;
	mdraw_text_extents(drawframe, mt, 0, 1, &control, &rect, NULL, NULL);
	flheight = rect.height + rect.y;
	mdraw_text_extents(drawframe, mt, 0, mtext_len(mt), 
			   &control, &rect, NULL, NULL);
	if (ascent < - rect.y)
	  ascent = - rect.y;
	if (descent < rect.height + rect.y)
	  descent = rect.height + rect.y;
	if (rect.width > 0)
	  state->brankline = 0;

	draw_line_itemsize(state, ascent, descent, flheight);
	if (obj_new_mtext(state, state->linewidth, state->charoff, mt, 
				drawframe, ascent, descent)){
		state->linewidth += rect.width;
		return p + len;
	} else
		return NULL;
}

void
M17N_draw_object(obj, target, x, y)
	struct render_object *obj;
	Drawable target;	
	int x, y;
{
	MFace *faces[256];
	int i, n;			

	mdraw_text_with_control (
		obj->data.m17ntext.drawframe, 
		(MDrawWindow) target, x, y,
		obj->data.m17ntext.mt, 0, 
		obj->data.m17ntext.len, &control);

	/* detach extra properties */
	n = mtext_get_prop_values (obj->data.m17ntext.mt, 0, Mface, (void **) faces, 256);
	for (i = n; i > 3; i --){
		mtext_pop_prop (obj->data.m17ntext.mt, 0, 
			obj->data.m17ntext.len, Mface);
	}
}

void
M17N_draw_string(state, data)
	struct render_state *state;
	char *data;
{
	char *p = data;
	char *registry = NULL;
	int charset16 = 0;

	if (strlen(p))
		M17N_draw_fragment(state, p, strlen(p));
}

static void
M17N_format_line(line, y, indent, width)
	int line, y, *indent, *width;
{
	*indent = 0; // we don't need to set this.
	*width = current_linewidth;
}


#define M17NMAXKEY 3
#define M17NMAXVAL 128
void
M17N_process_direc(key, value)
	char *key, *value;
{
	static u_char *mfontset_val[M17N_MAX_CACHE];
	static u_char *mfamily_val[M17N_MAX_CACHE];
	static MFace *mfontset_face[M17N_MAX_CACHE];
	static MFace *mfamily_face[M17N_MAX_CACHE];
	static int mfontset_num = 0;
	static int mfamily_num = 0;
	static MFace *last_used; 

	char *keywords[M17NMAXKEY] = {"fontset", "family", "language"};
	int i;

	for (i = 0; i < M17NMAXKEY; i ++) {
		if (!strcmp(key, keywords[i])) {
			break;
		}
	}
	switch(i){
	case 0: 
		for (i = 0; i < mfontset_num; i ++){
			if (strlen(mfontset_val[i]) == strlen(value) &&
				!(strcmp(mfontset_val[i], value))){
				face_other = mfontset_face[i];
				goto done;
			}
		}
		if (mfontset_num == M17N_MAX_CACHE){
			face_other = mfontset_face[0]; // adhoc
			goto done;
		} 
		mfontset_face[mfontset_num] = mface();
		mface_put_prop (mfontset_face[mfontset_num], Mfontset, mfontset (value));
		face_other = mfontset_face[mfontset_num];
		mfontset_val[mfontset_num] = strdup(value);
		mfontset_num ++;
		goto done;
		break;
	case 1: 
		for (i = 0; i < mfamily_num; i ++){
			if (strlen(mfamily_val[i]) == strlen(value) &&
				!(strcmp(mfamily_val[i], value))){
				face_other = mfamily_face[i];
				goto done;
			}
		}
		if (mfamily_num == M17N_MAX_CACHE){
			face_other = mfamily_face[0]; // adhoc
			goto done;
		} 
		mfamily_face[mfamily_num] = mface();
		mface_put_prop (mfamily_face[mfamily_num], Mfamily, msymbol (value));
		face_other = mfamily_face[mfamily_num];
		mfamily_val[mfamily_num] = strdup(value);
		mfamily_num ++;
		goto done;
		break;
	case 2: 
		lang = msymbol(value);
		break;
	default:
		fprintf(stderr, "unknown keyword:%s for m17n-lib\n", key);	
	}
done:
	last_used = face_other;
	return;
}
#endif
