/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include "kerryapp.h"
#include "searchdlg.h"
#include "hitwidget.h"
#include "kwidgetlistbox.h"
#include "kerrylabel.h"

#include <qlayout.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kapplication.h>
#include <kfileitem.h>
#include <kiconloader.h>
#include <klocale.h>
#include <klineedit.h>
#include <kurllabel.h>
#include <krun.h>
#include <krfcdate.h>
#include <qtable.h>
#include <qdir.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <kio/job.h>
#include <kprotocolinfo.h>
#include <ktrader.h>
#include <kprocess.h>
#include <qregexp.h>
#include <kaction.h>
#include <kpopupmenu.h>
#include <dcopclient.h>
#include <qcheckbox.h>
#include <kdesktopfile.h>

#include <kwin.h>
#include <kwinmodule.h>

SearchDlg::SearchDlg(QWidget *parent, const char *name)
 : DCOPObject("search"), HitsLayout(parent, name)
{
  connect(& beagle_search, SIGNAL(found(BeagleSearch::BeagleResultList&)),
          this, SLOT(searchHasOutput(BeagleSearch::BeagleResultList&)));
  connect(& beagle_search, SIGNAL(vanished(BeagleSearch::BeagleVanishedURIList&)),
          this, SLOT(searchLostOutput(BeagleSearch::BeagleVanishedURIList&)));
  connect(& beagle_search, SIGNAL(finished()),
          this, SLOT(searchFinished()));
  connect(& beagle_search, SIGNAL(oops_error(const QString&)),
          this, SLOT(searchError(const QString&)));
  connect(editSearch, SIGNAL(returnPressed()), SLOT(search()));
  connect(editSearch, SIGNAL(textChanged(const QString &)), SLOT(searchChanged(const QString &)));
  connect(comboShow, SIGNAL(activated(int)), SLOT(sortFilterResults()));
  connect(buttonFind, SIGNAL(clicked()), SLOT(search()));
  connect(buttonClear, SIGNAL(clicked()), SLOT(slotButtonClear()));
  connect(buttonPrevious, SIGNAL(clicked()), SLOT(slotPrevious()));
  connect(buttonNext, SIGNAL(clicked()), SLOT(slotNext()));
  connect(tableHits, SIGNAL(contextMenuRequested (int, int, const QPoint &)), SLOT(slotContextMenu(int, int, const QPoint &)));
  buttonFind->setDefault(true);
  setMouseTracking(true);
  results.setAutoDelete(true);
  results.clear();
  displayed_results.clear();
  displayAmount = 5;
  displayOffset = 0;
  labelStatus->setAlignment(Qt::SingleLine);
  pPreviewJob = 0;
  pPreviewMimeTypes = 0;
  previewItems.clear();
  previewItems.setAutoDelete(true);
  pending_showQuickTips=true;
  updateStatus();
  defaultSortOrder = Relevance;
  currentSortOrder = defaultSortOrder;
  kapp->dcopClient()->setDefaultObject( objId() );
  beagleJustStarted = false;
}

SearchDlg::~SearchDlg()
{
}

void SearchDlg::showQuickTips()
{
  tableHits->clear();

  HitWidget* item = new HitWidget(QString::null, QString::null);
  QLabel *headerLabel = new QLabel(item);
  headerLabel->setText(i18n("Quick Tips"));
  item->insertHeaderWidget(0,headerLabel);

  item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("messagebox_info", KIcon::NoGroup, KIcon::SizeLarge));
  item->setDescriptionText("<qt>"+i18n("- You can use upper and lower case; search is case-insensitive.<br>"
                           "- To search for optional terms, use OR.  ex: <b>George OR Ringo</b><br>"
                           "- To exclude search terms, use the minus symbol in front, such as <b>-cats</b><br>"
                           "- When searching for a phrase, add quotes. ex: <b>\"There be dragons\"</b><br>"
                           "- Add ext:type to specify a file extension, ex: <b>ext:txt</b> or <b>ext:</b> for none")+"</qt>");
  tableHits->insertItem(item);

  item = new HitWidget(QString::null, QString::null);
  headerLabel = new QLabel(item);
  headerLabel->setText(i18n("Configuration"));
  item->insertHeaderWidget(0,headerLabel);

  item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("package_settings", KIcon::NoGroup, KIcon::SizeLarge));
  item->setDescriptionText("<qt>"+i18n("- Choose what folders and resources shall be indexed - or not.<br>"
                           "- Change the sort order and the number of shown results.<br>"
                           "- Define your own shortcuts to invoke the search dialog.")+"</qt>");
  KURLLabel *buttonStart = new KURLLabel(item);
  buttonStart->setPixmap(SmallIcon("exec"));
  item->insertHitWidget(0,buttonStart);
  connect(buttonStart, SIGNAL(leftClickedURL()), SIGNAL(configure()));

  buttonStart = new KURLLabel(item);
  buttonStart->setText(i18n("Open configuration dialog"));
  item->insertHitWidget(1,buttonStart);
  connect(buttonStart, SIGNAL(leftClickedURL()), SIGNAL(configure()));

  tableHits->insertItem(item);

  labelStatus->setText("");
}

void SearchDlg::search(const QString & search)
{
  pending_showQuickTips=false;
  showSearchDialog();
  comboShow->setCurrentItem(Everywhere);
  editSearch->setText(search);
  this->search();
}

void SearchDlg::searchChanged(const QString & search)
{
  buttonFind->setEnabled(!search.isEmpty());
  if (!search.isEmpty() && displayed_results.count()==0 && tableHits->count()==1)
    tableHits->clear();
  if (search.isEmpty() && displayed_results.count()==0 && tableHits->count()==0)
    showQuickTips();
}

void SearchDlg::setDisplayAmount(int amount)
{
    if (amount<2 || displayAmount==amount)
	return;

    displayAmount = amount;
    displayOffset = 0;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::setSortOrder(int order)
{
    defaultSortOrder = order;

    if (currentSortOrder==order)
        return;

    currentSortOrder = order;
    if (displayed_results.count())
        sortFilterResults();
}

void SearchDlg::slotButtonClear()
{
    editSearch->clear();
    slotClear();
    showQuickTips();
    currentSortOrder = defaultSortOrder;
}

void SearchDlg::slotClear()
{
    displayOffset = 0;
    stopPreview();
    tableHits->clear();
    displayed_results.clear();
    results.clear();
    updateStatus();
}

void SearchDlg::slotPrevious()
{
    if (displayOffset==0)
      return;

    displayOffset-=displayAmount;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::slotNext()
{
    if (displayOffset+displayAmount>=(int)displayed_results.count())
      return;

    displayOffset+=displayAmount;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::sortFilterResults()
{
    displayOffset = 0;
    stopPreview();
    tableHits->clear();
    displayed_results.clear();
    displayResults(results);
    updateStatus();
}

void SearchDlg::fillTableHits()
{
    stopPreview();
    tableHits->clear();
    previewItems.clear();
    if (displayOffset+displayAmount<=(int)displayed_results.count()) {
      for(int i = displayOffset; i < displayOffset+displayAmount; ++i) {
	  insertResult(displayed_results.at(i),i-displayOffset);
      }
    }
    else
    for(uint i = displayOffset; i < (displayed_results.count() % displayAmount)+displayOffset; ++i) {
	  insertResult(displayed_results.at(i),i-displayOffset);
    }
    if (previewItems.count()) {
      startPreview(previewItems);
    }
}

void SearchDlg::updateStatus()
{
    buttonPrevious->setEnabled(displayOffset>0);
    buttonNext->setEnabled(displayOffset+displayAmount<(int)displayed_results.count());
    labelStatus->setAlignment(Qt::SingleLine);
    const int count = displayed_results.count();
    if (count==0)
       labelStatus->setText(i18n("<qt>No results.</qt>"));
    else if (displayOffset==0)
       labelStatus->setText(i18n("Best <b>%1 results of %2</b> shown.").arg(tableHits->count()).arg(displayed_results.count()));
    else
       labelStatus->setText(i18n("Results <b>%1 through %2 of %3</b> are shown.").arg(displayOffset+1).arg(displayOffset+tableHits->count()).arg(displayed_results.count()));
}

void SearchDlg::search()
{
    QString searchText = editSearch->text();
    if (searchText.isEmpty())
       return;

    slotClear();
    labelStatus->setText(i18n("Searching..."));
    emit searchStarted(searchText);
    if (!beagle_search.search(searchText)) {
       tableHits->clear();
       HitWidget* item = new HitWidget(QString::null, QString::null);
       QLabel *headerLabel = new QLabel(item);
       headerLabel->setText(i18n("The query for \"%1\" failed.").arg(searchText));
       item->insertHeaderWidget(0,headerLabel);

       item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("messagebox_critical", KIcon::NoGroup, KIcon::SizeLarge));
       item->setDescriptionText("<qt>"+i18n("The likely cause is that the Beagle daemon is not running.")+"</qt>");

       cb_beagleStart = new QCheckBox(i18n("Automatically start Beagle daemon at login"),item);
       item->insertTextWidget(1,cb_beagleStart);

       KURLLabel *buttonStart = new KURLLabel(item);
       buttonStart->setPixmap(SmallIcon("exec"));
       item->insertHitWidget(0,buttonStart);
       connect(buttonStart, SIGNAL(leftClickedURL()), SLOT(slotStartBeagle()));

       buttonStart = new KURLLabel(item);
       buttonStart->setText(i18n("Click to start the Beagle daemon"));
       item->insertHitWidget(1,buttonStart);
       connect(buttonStart, SIGNAL(leftClickedURL()), SLOT(slotStartBeagle()));

       tableHits->insertItem(item);
       labelStatus->setText(""); 
    }
}

QString SearchDlg::takeProperty( const QString& property, QStringList& propertyList )
{
      QString ret = QString::null;
      BeagleSearch::PropertyList::iterator it;
      for ( it = propertyList.begin(); it != propertyList.end(); ++it ) {
         const QString search = property+'=';
         if ((*it).startsWith(search)) {
           ret = (*it).remove(search);
           propertyList.erase(it);
           break;
         }
      }
      return ret;
}

QDateTime SearchDlg::datetimeFromString( const QString& s)
{
      int year( s.mid( 0, 4 ).toInt() );
      int month( s.mid( 4, 2 ).toInt() );
      int day( s.mid( 6, 2 ).toInt() );
      int hour( s.mid( 8, 2 ).toInt() );
      int min( s.mid( 10, 2 ).toInt() );
      int sec( s.mid( 12, 2 ).toInt() );
      return QDateTime(QDate(year,month,day),QTime(hour,min,sec));
}

void SearchDlg::insertResult(BeagleSearch::beagle_result_struct *result,int index)
{
      KURL url(*(result->uri));


      HitWidget* item = new HitWidget(*(result->uri),*(result->mime_type),tableHits);

      item->icon->setURL(*(result->uri));
      connect(item->icon, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      KURLLabel *buttonGo = new KerryLabel(item);
      buttonGo->setPixmap(SmallIcon( *(result->mime_type)=="application/x-desktop" ? "exec" : "fileopen") );
      buttonGo->setURL(*(result->uri));
      item->insertHitWidget(0,buttonGo);
      connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      buttonGo = new KerryLabel(item);
      buttonGo->setText( *(result->mime_type)=="application/x-desktop" ? i18n("Run") : i18n("Open") );
      buttonGo->setURL(*(result->uri));
      item->insertHitWidget(1,buttonGo);
      connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      if (result->tilegroup == BeagleSearch::Website) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("network", KIcon::NoGroup, KIcon::SizeLarge));

        QString description = "<qt>";
        if (result->hit_type!="Google") {
          QDateTime datetime;
          datetime.setTime_t(result->last_index_time);
          description = description + i18n("Last viewed: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>";
        }

        item->setDescriptionText(description +i18n("URL:")+" "+*(result->uri)+"</qt>");

        if (result->snippet)
            item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        KerryLabel *headerFileLabel = new KerryLabel(item);
        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);
        if (title.isEmpty())
           title = takeProperty("Title",_properties);

        headerFileLabel->setText(title.isEmpty() ? i18n("Untitled Page") : title);
        headerFileLabel->setAlignment(headerFileLabel->alignment() | Qt::SingleLine);
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (result->tilegroup == BeagleSearch::Feed) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("network", KIcon::NoGroup, KIcon::SizeLarge));

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+ i18n("Published: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>");

        if (result->snippet)
          item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("Weblog:"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(0,headerLabel);

        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);
        QString identifier = takeProperty("dc:identifier", _properties);
        item->setUri(identifier);
        buttonGo->setURL(identifier);

        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setText(title.isEmpty() ? i18n("Untitled Entry") : title);
        headerFileLabel->setAlignment(headerFileLabel->alignment() | Qt::SingleLine);
        headerFileLabel->setURL(identifier);
        item->insertHeaderWidget(1,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (result->tilegroup == BeagleSearch::Note) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("contents2", KIcon::NoGroup, KIcon::SizeLarge));
        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+ i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>");


        if (result->snippet)
          item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);

        KerryLabel *headerNoteLabel = new KerryLabel(item);
        headerNoteLabel->setText(title.isEmpty() ? i18n("Untitled Entry") : title);
        headerNoteLabel->setAlignment(headerNoteLabel->alignment() | Qt::SingleLine);
        headerNoteLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNoteLabel);
        connect(headerNoteLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if ((*(result->uri)).startsWith("calendar:/")) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("ximian-evolution-calendar", KIcon::NoGroup, KIcon::SizeLarge));

        QStringList _properties(result->properties);
        QString summary = takeProperty("fixme:summary",_properties);

        QString properties;
        QDateTime datetime;
        datetime = datetimeFromString(takeProperty("fixme:starttime",_properties));
        properties = properties + i18n("Start time: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)) + "<br>";
        datetime = datetimeFromString(takeProperty("fixme:endtime",_properties));
        properties = properties + i18n("End time: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)) + "<br>";

        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        item->setDescriptionText("<qt>"+properties+"</qt>");

        KerryLabel *headerSummaryLabel = new KerryLabel(item);
        headerSummaryLabel->setText(summary.isEmpty() ? i18n("No Summary Specified") : summary);
        headerSummaryLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerSummaryLabel);
        connect(headerSummaryLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotOpenEvolution(const QString&)));
      }
      else if ((*(result->uri)).startsWith("contacts:/")) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("ximian-evolution-addressbook", KIcon::NoGroup, KIcon::SizeLarge));

        QStringList _properties(result->properties);
        QString name = takeProperty("fixme:Name",_properties);
        QString email = takeProperty("fixme:Email1",_properties);

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (!properties.isEmpty())
           item->setDescriptionText("<qt>"+properties+"</qt>");

        KerryLabel *headerNameLabel = new KerryLabel(item);
        headerNameLabel->setText(name.isEmpty() ? i18n("No Name Known") : name);
        headerNameLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNameLabel);
        connect(headerNameLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotOpenEvolution(const QString&)));

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n(","));
        item->insertHeaderWidget(1,headerLabel);

        KerryLabel *headerEmailLabel = new KerryLabel(item);
        headerEmailLabel->setText(email);
        headerEmailLabel->setURL(email);
        item->insertHeaderWidget(2,headerEmailLabel);
        connect(headerEmailLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotMailTo(const QString&)));
      }
      else if (*(result->mime_type)=="message/rfc822" || (*(result->uri)).startsWith("email:/") ) {
        item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("mail_generic", KIcon::NoGroup, KIcon::SizeLarge));

        QStringList _properties(result->properties);
        QString subject = takeProperty("dc:title",_properties);
        QString from = takeProperty("fixme:from",_properties);
        QString received = takeProperty("fixme:date",_properties);

        QDateTime received_datetime;
        received_datetime = datetimeFromString(received);
        if (!received.isEmpty())
          item->setDescriptionText("<qt>"+i18n("Received: %1").arg(KGlobal::locale()->formatDateTime(received_datetime,false))+"</qt>");

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }

        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");

        KerryLabel *headerSubjectLabel = new KerryLabel(item);
        headerSubjectLabel->setText(subject.isEmpty() ? i18n("No Subject") : subject);
        item->insertHeaderWidget(0,headerSubjectLabel);
        headerSubjectLabel->setURL(*(result->uri));
        connect(headerSubjectLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("From"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(1,headerLabel);

        KerryLabel *headerFromLabel = new KerryLabel(item);
        headerFromLabel->setText(from.isEmpty() ? i18n("Unknown Person") : from);
        headerFromLabel->setURL(from);
        item->insertHeaderWidget(2,headerFromLabel);
        connect(headerFromLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotMailTo(const QString&)));
      }
      else if (*(result->mime_type)=="application/x-desktop") {
        KFileItem *fileitem=new KFileItem(*(result->uri),*(result->mime_type),KFileItem::Unknown);
        item->icon->setPixmap(fileitem->pixmap(KIcon::SizeHuge, KIcon::DefaultState));
        if (canPreview(fileitem))
            previewItems.append(fileitem);
        else
            delete fileitem;

        KDesktopFile desktopfile(url.path(),true);
        if (!desktopfile.readGenericName().isNull()) {
	    item->setDescriptionText("<qt>"+desktopfile.readGenericName()+"</qt>");
        }

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("Application:"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(0,headerLabel);

        KerryLabel *headerFileLabel = new KerryLabel(item);
        QStringList _properties(result->properties);
        QString title = takeProperty("fixme:Name",_properties);
        headerFileLabel->setText(title.isEmpty() ? desktopfile.readName() : title);
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(1,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (*(result->mime_type)=="beagle/x-kopete-log" || *(result->mime_type)=="beagle/x-gaim-log") {
        if (*(result->mime_type)=="beagle/x-kopete-log")
          item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("kopete", KIcon::NoGroup, KIcon::SizeLarge));
        else
          item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("gaim", KIcon::NoGroup, KIcon::SizeLarge));

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+i18n("Date: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)+"</qt>"));

        if (result->snippet)
            item->setPropertiesText("<qt>"+*(result->snippet)+ "</qt>");

        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setURL(*(result->uri));
        QStringList _properties(result->properties);
        QString person = takeProperty("fixme:speakingto",_properties);
        headerFileLabel->setText(i18n("Conversation With %1").arg(person.isEmpty() ? i18n("Unknown Person") : person));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else {
        KFileItem *fileitem=new KFileItem(*(result->uri),*(result->mime_type),KFileItem::Unknown);
        item->icon->setPixmap(fileitem->pixmap(KIcon::SizeHuge, KIcon::DefaultState));
        if (canPreview(fileitem))
            previewItems.append(fileitem);
        else
            delete fileitem;
    
        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setText(url.fileName());
        headerFileLabel->setTipText(url.prettyURL());
        headerFileLabel->setUseTips();
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
    
        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("In Folder"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(1,headerLabel);
    
        KerryLabel *headerDirLabel = new KerryLabel(item);
        QString directory=url.directory();
        int i = directory.findRev('/');
        if (i>0)
            directory=directory.mid(i+1);
        headerDirLabel->setText(directory);
        headerDirLabel->setURL(url.directory());
        headerDirLabel->setTipText(url.directory());
        headerDirLabel->setUseTips();
        item->insertHeaderWidget(2,headerDirLabel);
        connect(headerDirLabel, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));


        QString title;
        if (*(result->mime_type)=="inode/directory") {
            int count = QDir(url.path()).count() -2 ;
            if (count==0)
              title=i18n("Empty");
            else
              title=i18n("Contains 1 item","Contains %n items",count);
        }
        else {
            QDateTime datetime;
            datetime.setTime_t(result->last_index_time);
            title = i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false));
        }
        item->setDescriptionText("<qt>"+title+"</qt>");
    
        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = result->properties.begin(); it != result->properties.end(); ++it )
                properties=properties+(*it);

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }
   
        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");
    

        item->insertHitSpacing(2,10);
        buttonGo = new KURLLabel(item);
        buttonGo->setPixmap(SmallIcon("kfm"));
        item->insertHitWidget(3,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
    
        buttonGo = new KURLLabel(item);
        buttonGo->setText(i18n("Reveal in File Manager"));
        item->insertHitWidget(4,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
      }

      item->score->setText(i18n("<p align=\"center\">Score: %1</p>").arg(result->score,0,'f',1));
    
      tableHits->insertItem(item,index);
}

void SearchDlg::searchHasOutput(BeagleSearch::BeagleResultList &items)
{
    for (BeagleSearch::BeagleResultList::ConstIterator it = items.begin(); it != items.end(); ++it)
        results.append(*it);
    displayResults(items);
}

void SearchDlg::displayResults(BeagleSearch::BeagleResultList &items)
{
    bool rebuildDisplay = false;
    for (BeagleSearch::BeagleResultList::ConstIterator it = items.begin(); it != items.end(); ++it) {
        BeagleSearch::beagle_result_struct *result = *it;

        bool show = false;
        switch (comboShow->currentItem())
        {
	  case Everywhere:
		show = true;
		break;
	  case Applications:
                show = ( result->tilegroup == BeagleSearch::Application );
		break;
/*          case Contacts:
		show = ( result->tilegroup == BeagleSearch::Contact );
		break;*/
          case Documents:
		show = ( result->tilegroup == BeagleSearch::Documents );
		break;
          case Conversations:
		show = ( result->tilegroup == BeagleSearch::Conversations );
		break;
          case Images:
		show = ( result->tilegroup == BeagleSearch::Image );
		break;
          case Media:
		show = ( result->tilegroup == BeagleSearch::Audio || result->tilegroup == BeagleSearch::Video);
		break;
          case Website:
		show = ( result->tilegroup == BeagleSearch::Website);
		break;
          case FilePathName:
		show = (((*(result->uri)).find(editSearch->text(),0,false)) != -1);
          default:
		break;
        }

        if (!show)
           continue;

        int i = 0;
        for (BeagleSearch::BeagleResultList::ConstIterator result_it = displayed_results.begin(); result_it != displayed_results.end(); ++result_it) {
           bool foundplace = false;
           switch (currentSortOrder)
           {
		case Name:
           		if (KURL(*(result->uri)).fileName().lower() < KURL(*((*(result_it))->uri)).fileName().lower())
              			foundplace = true;
			break;
		case Modified:
           		if (result->last_index_time >= (*result_it)->last_index_time)
              			foundplace = true;
			break;
		default:
           		if (result->score >= (*result_it)->score)
              			foundplace = true;
			break;
           }
           if (foundplace)
		break;
           ++i;
        }
        if (displayed_results.count()==0)
          tableHits->clear();
        displayed_results.insert(i, result);
        if (i<=displayOffset+displayAmount)
          rebuildDisplay = true;
    }
    if (rebuildDisplay) {
      tableHits->setUpdatesEnabled(false);
      fillTableHits();
      tableHits->setUpdatesEnabled(true);
    }
    updateStatus();
}

void SearchDlg::slotOpen()
{
    HitWidget* item = static_cast<HitWidget*>(sender()->parent());
    if (item) {
      QString mimetype = item->mimetype();
      if (mimetype=="beagle/x-kopete-log" || mimetype=="beagle/x-gaim-log") {
        KProcess *proc = new KProcess;
        *proc << "beagle-imlogviewer";
        KURL kuri = KURL(item->uri());
        QString uri = kuri.path();
        if (mimetype=="beagle/x-kopete-log")
          *proc << "--client" << "kopete" << "--highlight-search" << editSearch->text() << uri;
        else
          *proc << "--client" << "gaim" << "--highlight-search" << editSearch->text() << uri;
        if (!proc->start()) {
//        KMessageBox::error(0,i18n("Could not start instant message log viewer."));
          if (mimetype=="beagle/x-kopete-log")
            KRun::runURL(uri, "text/plain", false, true);
          else
            KRun::runURL(uri, "text/html", false, true);
          return;
        }
      }
      else if (item->uri().startsWith("calendar:/") || item->uri().startsWith("contacts:/") 
            || item->uri().startsWith("email:/"))
      {
        slotOpenEvolution(item->uri());
      }
      else if (item->uri().startsWith("note:/")) {
        KProcess *proc = new KProcess;
        *proc << "tomboy";
        *proc << "--open-note" << item->uri() << "--highligh-search" << "\""+editSearch->text()+"\"";
        if (!proc->start()) {
          KMessageBox::error(0,i18n("Could not start Tomboy."));
          return;
        }
      }
      else {
        if (mimetype=="beagle/x-konq-cache")
          mimetype = "text/html";
        KRun::runURL(item->uri(), mimetype, false, true);
      }
    }
}

void SearchDlg::slotOpenDir()
{
    HitWidget* item = static_cast<HitWidget*>(sender()->parent());
    if (item)
      KRun::runURL(KURL(item->uri()).directory(), "inode/directory", false, true);
}

void SearchDlg::slotMailTo(const QString &address)
{
    kapp->invokeMailer(address, QString::null);
}

void SearchDlg::slotOpenEvolution(const QString &address)
{
    KProcess *proc = new KProcess;
    *proc << "evolution";
    *proc << address;
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start Evolution."));
      return;
    }
}

void SearchDlg::slotStartBeagle()
{
    beagleJustStarted = true;
    if (cb_beagleStart->isChecked()) {
      KConfig *config = KGlobal::config();
      config->setGroup("Beagle");
      config->writeEntry("AutoStart",true);
      config->sync();
    }

    KProcess *proc = new KProcess;
    *proc << "beagled";
    *proc << "--indexing-delay 2";
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start Beagle daemon."));
      return;
    }
    slotClear();
    QTimer::singleShot(5000, this, SLOT(search()));
}

void SearchDlg::searchLostOutput(BeagleSearch::BeagleVanishedURIList &items)
{
    bool rebuildDisplay = false;
    for (BeagleSearch::BeagleVanishedURIList::ConstIterator it = items.begin(); it != items.end(); ++it) {
      int i;
      for(i = 0; i < (int)displayed_results.count(); ++i)
      {
         BeagleSearch::beagle_result_struct *result = displayed_results.at(i);
         if (*(result->uri)==(*it)) {
           displayed_results.remove(i);
           if (displayed_results.count()==0)
              searchFinished();
           else if (i<=displayOffset+displayAmount) {
             rebuildDisplay = true;
             if (displayOffset>=(int)displayed_results.count())
               displayOffset-=displayAmount;
           }
           break;
        }
      }
      for(i = 0; i < (int)results.count(); ++i)
      {
         BeagleSearch::beagle_result_struct *result = results.at(i);
         if (*(result->uri)==(*it)) {
           results.remove(i);
           break;
        }
      }
    }

    if (rebuildDisplay) {
      tableHits->setUpdatesEnabled(false);
      fillTableHits();
      tableHits->setUpdatesEnabled(true);
    }

    updateStatus();
}

void SearchDlg::searchFinished()
{
    if (displayed_results.count())
       return;

    if (editSearch->text().isEmpty()) {
       showQuickTips();
       return;
    }

    tableHits->clear();
    HitWidget* item = new HitWidget(QString::null, QString::null);
    QLabel *headerLabel = new QLabel(item);
    headerLabel->setText(i18n("No results for \"%1\" were found.").arg(editSearch->text()));
    item->insertHeaderWidget(0,headerLabel);

    item->icon->setPixmap(KGlobal::iconLoader()->loadIcon("messagebox_warning", KIcon::NoGroup, KIcon::SizeLarge));

    QString text = "<qt>";
    if (comboShow->currentItem())
	text += i18n("- You can change the scope of your search using the \"Within\" combo box.<br>&nbsp;&nbsp;A broader search scope might produce more results.")+"<br>";
    text += i18n("- You should check the spelling of your search words to see if you accidentally misspelled any words.");
    if (beagleJustStarted) {
        text += "<br>"+i18n("- The Beagle daemon was just started. Please be patient until it finished its indexing.");
        beagleJustStarted = false;
    }
    item->setDescriptionText(text+"</qt>");
    labelStatus->setText("");

    tableHits->insertItem(item);
}

void SearchDlg::searchError(const QString& error)
{
     kdDebug() << "SearchDlg::searchError() " << error << endl;
}

bool SearchDlg::canPreview( KFileItem* item )
{
    if ( !KGlobalSettings::showFilePreview( item->url() ) )
        return false;

    if ( pPreviewMimeTypes == 0L )
        updatePreviewMimeTypes();

    return mimeTypeMatch( item->mimetype(), *( pPreviewMimeTypes ) );
}

void SearchDlg::updatePreviewMimeTypes()
{
    if ( pPreviewMimeTypes == 0L )
        pPreviewMimeTypes = new QStringList;
    else
        pPreviewMimeTypes->clear();

    // Load the list of plugins to determine which mimetypes are supported
    KTrader::OfferList plugins = KTrader::self()->query("ThumbCreator");
    KTrader::OfferList::ConstIterator it;

    for ( it = plugins.begin(); it != plugins.end(); ++it ) {
        QStringList mimeTypes = (*it)->property("MimeTypes").toStringList();
        for (QStringList::ConstIterator mt = mimeTypes.begin(); mt != mimeTypes.end(); ++mt)
           pPreviewMimeTypes->append(*mt);
    }
}

bool SearchDlg::mimeTypeMatch( const QString& mimeType, const QStringList& mimeList ) const
{
    for (QStringList::ConstIterator mt = mimeList.begin(); mt != mimeList.end(); ++mt)
    {
        if ( mimeType == *mt )
            return true;
        // Support for *mt == "image/*"
        QString tmp( mimeType );
        if ( (*mt).endsWith("*") && tmp.replace(QRegExp("/.*"), "/*") == (*mt) )
            return true;
    }
    return false;
}

void SearchDlg::startPreview( const KFileItemList& items )
{
    stopPreview(); // just in case
    int iconSize = KGlobal::iconLoader()->currentSize( KIcon::Desktop );

    pPreviewJob = KIO::filePreview( items, KIcon::SizeHuge, KIcon::SizeHuge, iconSize,
        true /*m_pSettings->textPreviewIconTransparency()*/, true /* scale */,
        true /* save */, 0); //&(d->previewSettings) );
    connect( pPreviewJob, SIGNAL( gotPreview( const KFileItem *, const QPixmap & ) ),
             this, SLOT( slotPreview( const KFileItem *, const QPixmap & ) ) );
    connect( pPreviewJob, SIGNAL( result( KIO::Job * ) ),
             this, SLOT( slotPreviewResult() ) );
}


void SearchDlg::stopPreview()
{
    if (pPreviewJob)
    {
        pPreviewJob->kill();
        pPreviewJob = 0;
    }
}

void SearchDlg::slotPreview(const KFileItem *item, const QPixmap &pix)
{
  for(uint i=0; i < tableHits->count(); ++i)
  {
    HitWidget* w = static_cast<HitWidget*>(tableHits->item(i));
    if (w && w->uri() == item->url().prettyURL() ) {
      w->icon->setPixmap(pix);
      break;
    }
  }
}

void SearchDlg::slotPreviewResult()
{
    pPreviewJob = 0;
    previewItems.clear();
}

void SearchDlg::keyPressEvent(QKeyEvent *e)
{
    if (e->key()==Key_PageDown) {
      if (e->state()==ControlButton) {
        if (displayOffset+displayAmount>=(int)displayed_results.count())
          return;

        displayOffset=((displayed_results.count() -1) / displayAmount) * displayAmount;
        tableHits->setUpdatesEnabled(false);
        fillTableHits();
        tableHits->setUpdatesEnabled(true);
        updateStatus();
      }
      else
        slotNext();
    }
    else if (e->key()==Key_PageUp) {
      if (e->state()==ControlButton) {
        if (displayOffset==0)
          return;

        displayOffset=0;
        tableHits->setUpdatesEnabled(false);
        fillTableHits();
        tableHits->setUpdatesEnabled(true);
        updateStatus();
      }
      else
        slotPrevious();
    }
    else
      HitsLayout::keyPressEvent(e);
}

void SearchDlg::showEvent(QShowEvent *e )
{
    HitsLayout::showEvent( e );
    if (pending_showQuickTips) {
      showQuickTips();
      pending_showQuickTips=false;
    }
}

void SearchDlg::slotContextMenu( int /*row*/, int /*col*/, const QPoint & pos )
{
    KPopupMenu *popup = new KPopupMenu(this);
    popup->insertTitle(i18n("Sort By"));
    popup->insertItem(i18n("Relevance"),Relevance);
    popup->insertItem(i18n("Name"),Name);
    popup->insertItem(i18n("Date Modified"),Modified);
    popup->setItemChecked(currentSortOrder,true);
    int selected = popup->exec(pos);
    if (selected>=Relevance && selected<=Modified && currentSortOrder!=selected) {
	currentSortOrder = selected;
        if (displayed_results.count())
          sortFilterResults();
    }
    delete popup;
}

void SearchDlg::showSearchDialog()
{
    show();
    KWin::setOnDesktop( winId(), KWin::currentDesktop() );
    kapp->updateUserTimestamp();
    KWin::forceActiveWindow( winId() );
    editSearch->setFocus();
    editSearch->selectAll();
}

#include "searchdlg.moc"
