/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: recordctl.cpp,v 1.8.20.1 2004/07/09 02:05:58 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "recordctl.h"
#include "ihxpckts.h"
#include "hxrecord.h"

#include "hxheap.h"

#ifdef _DEBUG
#undef HX_THIS_FILE             
static const char HX_THIS_FILE[] = __FILE__;
#endif

HXRecordControl::HXRecordControl(IUnknown* pUnkPlayer, IUnknown* pUnkSource) :
    m_lRefCount(0),
    m_pRecordSource(NULL),
    m_pRecordService(NULL),
    m_bCanGetPackets(FALSE)
{
    SPIHXRecordManager spRecordManager = pUnkPlayer;
    if(spRecordManager.IsValid())
	spRecordManager->GetRecordService(m_pRecordService);
    if(m_pRecordService)
	m_pRecordService->CreateRecordSource(pUnkSource, m_pRecordSource);

    if(m_pRecordSource)
    {
	if(m_pRecordSource->SetFormatResponse(this) == HXR_OK)
	    m_bCanGetPackets = TRUE;
    }
}

HXRecordControl::~HXRecordControl()
{
    Cleanup();
}

STDMETHODIMP 
HXRecordControl::QueryInterface(REFIID riid, void** ppvObj)
{
    QInterfaceList qiList[] =
        {
            { GET_IIDHANDLE(IID_IHXFormatResponse), (IHXFormatResponse*)this },
            { GET_IIDHANDLE(IID_IUnknown), (IUnknown*)(IHXFormatResponse*)this },
        };
    
    return ::QIFind(qiList, QILISTSIZE(qiList), riid, ppvObj);
}

STDMETHODIMP_(ULONG32) 
HXRecordControl::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

STDMETHODIMP_(ULONG32) 
HXRecordControl::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
	return m_lRefCount;
    }

    delete this;
    return 0;
}


void
HXRecordControl::Cleanup()
{
    for(UINT16 nStream = 0; nStream < m_PendingGetPackets.GetSize(); nStream++)
    {
	IHXPacket* pPacket = (IHXPacket*)m_PendingGetPackets.GetAt(nStream);
	m_PendingGetPackets.SetAt(nStream, NULL);
	HX_RELEASE(pPacket);
    }

    if(m_pRecordService)
	m_pRecordService->CloseRecordSource(m_pRecordSource);

    HX_RELEASE(m_pRecordService);
    HX_RELEASE(m_pRecordSource);

    while(!m_PendingPutPackets.IsEmpty())
    {
	PendingPutPacket* pPutPacket = (PendingPutPacket*)m_PendingPutPackets.RemoveHead();
	HX_ASSERT(pPutPacket);
	HX_RELEASE(pPutPacket->pPacket);
	HX_DELETE(pPutPacket);
    }
}

STDMETHODIMP	
HXRecordControl::PacketReady(HX_RESULT status, IHXPacket* pPacket)
{
    if(pPacket)
    {
	pPacket->AddRef();

	if(pPacket->GetStreamNumber() < m_PendingGetPackets.GetSize())
	    HX_ASSERT(!m_PendingGetPackets.GetAt(pPacket->GetStreamNumber()));

	m_PendingGetPackets.SetAt(pPacket->GetStreamNumber(), pPacket);
    }

    return HXR_OK;
}

HX_RESULT
HXRecordControl::Seek(ULONG32 seekTime)
{
    if(!m_pRecordSource)
	return HXR_NOT_INITIALIZED;

    HX_RESULT nResult = HXR_FAILED;
    if(m_bCanGetPackets)
    {
	nResult = m_pRecordSource->Seek(seekTime);
	if(nResult == HXR_OK)
	{
	    for(UINT16 nStream = 0; nStream < m_PendingGetPackets.GetSize(); nStream++)
	    {
		IHXPacket* pPacket = (IHXPacket*)m_PendingGetPackets.GetAt(nStream);
		m_PendingGetPackets.SetAt(nStream, NULL);
		HX_RELEASE(pPacket);
		
		// If seek succeeded the first GetPacket() for each stream should not fail. 
		// In case of metafiles though no packets should be send to renderers again.
		// Therefore GetPacket() fails in case of metafiles. SB.
		m_pRecordSource->GetPacket(nStream);
	    }
	}
    }

    if(nResult != HXR_OK)
    {
	m_pRecordSource->Flush();

	for(UINT16 nStream = 0; nStream < m_PendingGetPackets.GetSize(); nStream++)
	{
	    IHXPacket* pPacket = (IHXPacket*)m_PendingGetPackets.GetAt(nStream);
	    m_PendingGetPackets.SetAt(nStream, NULL);
	    HX_RELEASE(pPacket);
	}
    }

    return nResult;
}

HX_RESULT	
HXRecordControl::GetPacket(UINT16 usStreamNumber, IHXPacket*& pPacket)
{
    pPacket = NULL;

    if(!m_pRecordSource || !m_bCanGetPackets)
	return HXR_FAILED;

    if (m_PendingGetPackets.IsEmpty())
        return HXR_NO_DATA;
    
    HX_RESULT nResult = HXR_OK;
    pPacket = (IHXPacket*)m_PendingGetPackets.GetAt(usStreamNumber);

    if(!pPacket)
    {
	nResult = m_pRecordSource->GetPacket(usStreamNumber);
	if(nResult == HXR_OK)
	{
	    pPacket = (IHXPacket*)m_PendingGetPackets.GetAt(usStreamNumber);
	    if(!pPacket)
		nResult = HXR_WOULD_BLOCK;
	}
    }

    m_PendingGetPackets.SetAt(usStreamNumber, NULL);

    return nResult;
}

HX_RESULT
HXRecordControl::OnFileHeader(IHXValues* pValues)
{
    UINT32 nStreamCount = 0;			
    if(pValues)
	pValues->GetPropertyULONG32("StreamCount", nStreamCount);

    if(nStreamCount)
    {
	m_PendingGetPackets.SetSize(nStreamCount);
	for(UINT16 nStream = 0; nStream < nStreamCount; nStream++)
	    m_PendingGetPackets.SetAt(nStream, NULL);
    }

    HX_RESULT nResult = HXR_FAILED;
    if(m_pRecordSource)
	nResult = m_pRecordSource->OnFileHeader(pValues);

    if(nResult != HXR_OK && nResult != HXR_RECORD)
	Cleanup();

    return nResult;
}

HX_RESULT
HXRecordControl::OnStreamHeader(IHXValues* pValues)
{
    HX_RESULT nResult = HXR_FAILED;
    if(m_pRecordSource)
	nResult = m_pRecordSource->OnStreamHeader(pValues);

    if(nResult != HXR_OK && nResult != HXR_RECORD)
	Cleanup();

    return nResult;
}

HX_RESULT
HXRecordControl::OnPacket(IHXPacket* pPacket, INT32 nTimeOffset)
{
    HX_RESULT nResult = HXR_FAILED;

    if(pPacket && m_pRecordSource)
    {
	if(m_PendingPutPackets.GetCount())
	    nResult = HXR_RETRY;
	else
	    nResult = m_pRecordSource->OnPacket(pPacket, nTimeOffset);
	if(nResult == HXR_RETRY)
	{
	    PendingPutPacket* pPutPacket = new PendingPutPacket;
	    if(!pPutPacket)
		return HXR_OUTOFMEMORY;

	    pPutPacket->pPacket = pPacket;
	    pPutPacket->pPacket->AddRef();
	    pPutPacket->nTimeOffset = nTimeOffset;

	    m_PendingPutPackets.AddTail(pPutPacket);

	    nResult = HXR_OK;
	}
    }

    if(nResult != HXR_OK)
	Cleanup();

    return nResult;
}

BOOL		
HXRecordControl::CanAcceptPackets()
{
    if(!m_pRecordSource)
	return FALSE;

    while(!m_PendingPutPackets.IsEmpty())
    {
	PendingPutPacket* pPutPacket = (PendingPutPacket*)m_PendingPutPackets.GetHead();
	HX_ASSERT(pPutPacket);

	if(m_pRecordSource->OnPacket(pPutPacket->pPacket, pPutPacket->nTimeOffset) == HXR_RETRY)
	    return FALSE;

	HX_RELEASE(pPutPacket->pPacket);
	HX_DELETE(pPutPacket);
	m_PendingPutPackets.RemoveHead();
    }

    return TRUE;
}

void		
HXRecordControl::SetSource(IUnknown* pUnkSource)
{
    if(m_pRecordSource)
	m_pRecordSource->SetSource(pUnkSource);
}

void		
HXRecordControl::OnEndOfPackets()
{
    if(m_pRecordSource)
	m_pRecordSource->OnEndOfPackets();
}

