/***************************************************************************
 *            cxp-file-list-store.c
 *
 *  土  9月  3 23:50:52 2005
 *  Copyright  2005  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <cxp.h>
#include <string.h>
#include "cxp-file-list-store.h"

static void cxp_file_list_store_class_init (CxpFileListStoreClass * klass);
static void cxp_file_list_store_model_init (GtkTreeModelIface * iface);
static void cxp_file_list_store_init (CxpFileListStore * sp);
static void cxp_file_list_store_finalize (GObject * object);

static GtkTreeModelFlags cxp_file_list_store_get_flags (GtkTreeModel *
							tree_model);
static gint cxp_file_list_store_get_n_columns (GtkTreeModel * tree_model);
static GType cxp_file_list_store_get_column_type (GtkTreeModel * tree_model,
						  gint index);
static gboolean cxp_file_list_store_get_iter (GtkTreeModel * tree_model,
					      GtkTreeIter * iter,
					      GtkTreePath * path);
static GtkTreePath *cxp_file_list_store_get_path (GtkTreeModel * tree_model,
						  GtkTreeIter * iter);
static void cxp_file_list_store_get_value (GtkTreeModel * tree_model,
					   GtkTreeIter * iter,
					   gint column, GValue * value);
static gboolean cxp_file_list_store_iter_next (GtkTreeModel * tree_model,
					       GtkTreeIter * iter);
static gboolean cxp_file_list_store_iter_children (GtkTreeModel * tree_model,
						   GtkTreeIter * iter,
						   GtkTreeIter * parent);
static gboolean cxp_file_list_store_iter_has_child (GtkTreeModel * tree_model,
						    GtkTreeIter * iter);
static gint cxp_file_list_store_iter_n_children (GtkTreeModel * tree_model,
						 GtkTreeIter * iter);
static gboolean cxp_file_list_store_iter_nth_child (GtkTreeModel * tree_model,
						    GtkTreeIter * iter,
						    GtkTreeIter * parent,
						    gint n);
static gboolean cxp_file_list_store_iter_parent (GtkTreeModel * tree_model,
						 GtkTreeIter * iter,
						 GtkTreeIter * child);

/* -- GtkTreeSortable interface functions -- */

static gboolean cxp_file_list_store_sortable_get_sort_column_id (GtkTreeSortable *
							 sortable,
							 gint * sort_col_id,
							 GtkSortType * order);

static void cxp_file_list_store_sortable_set_sort_column_id (GtkTreeSortable * sortable,
						     gint sort_col_id,
						     GtkSortType order);

static void cxp_file_list_store_sortable_set_sort_func (GtkTreeSortable * sortable,
						gint sort_col_id,
						GtkTreeIterCompareFunc
						sort_func, gpointer user_data,
						GtkDestroyNotify destroy_func);

static void cxp_file_list_store_sortable_set_default_sort_func (GtkTreeSortable *
							sortable,
							GtkTreeIterCompareFunc
							sort_func,
							gpointer user_data,
							GtkDestroyNotify
							destroy_func);

static gboolean cxp_file_list_store_sortable_has_default_sort_func (GtkTreeSortable *
							    sortable);

/*
static void cxp_file_list_store_resort (CxpFileListStore * cxp_file_list_store);
*/
static void cxp_file_list_store_sortable_init (GtkTreeSortableIface * iface);


struct CxpFileListStorePrivate
{
	/* Place Private Members Here */
	GType column_types[FILE_LIST_N_COLUMNS];
	GPtrArray *store;
	gint sort_id;
	GtkSortType sort_order;
	gint stamp;
};

typedef enum
{
	/* Place Signal Types Here */
	SIGNAL_TYPE_EXAMPLE,
	LAST_SIGNAL
} CxpFileListStoreSignalType;

typedef struct
{
	CxpFileListStore *object;
} CxpFileListStoreSignal;

static guint cxp_file_list_store_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType cxp_file_list_store_get_type ()
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo our_info = {
			sizeof (CxpFileListStoreClass),
			NULL,
			NULL,
			(GClassInitFunc) cxp_file_list_store_class_init,
			NULL,
			NULL,
			sizeof (CxpFileListStore),
			0,
			(GInstanceInitFunc) cxp_file_list_store_init,
		};

		type = g_type_register_static (G_TYPE_OBJECT,
					       "CxpFileListStore", &our_info,
					       0);

		static const GInterfaceInfo tree_model_info = {
			(GInterfaceInitFunc) cxp_file_list_store_model_init,
			NULL,
			NULL
		};

		g_type_add_interface_static (type, GTK_TYPE_TREE_MODEL,
					     &tree_model_info);

		static const GInterfaceInfo tree_sortable_info = {
			(GInterfaceInitFunc) cxp_file_list_store_sortable_init,
			NULL,
			NULL
		};

		g_type_add_interface_static (type,
					     GTK_TYPE_TREE_SORTABLE,
					     &tree_sortable_info);
	}

	return type;
}

static void cxp_file_list_store_class_init (CxpFileListStoreClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = cxp_file_list_store_finalize;

	/* Create signals here:
	 * cxp_file_list_store_signals[SIGNAL_TYPE_EXAMPLE] = g_signal_new(...)
	 */
}

static void cxp_file_list_store_model_init (GtkTreeModelIface * iface)
{
	iface->get_flags = cxp_file_list_store_get_flags;
	iface->get_n_columns = cxp_file_list_store_get_n_columns;
	iface->get_column_type = cxp_file_list_store_get_column_type;
	iface->get_iter = cxp_file_list_store_get_iter;
	iface->get_path = cxp_file_list_store_get_path;
	iface->get_value = cxp_file_list_store_get_value;
	iface->iter_next = cxp_file_list_store_iter_next;
	iface->iter_children = cxp_file_list_store_iter_children;
	iface->iter_has_child = cxp_file_list_store_iter_has_child;
	iface->iter_n_children = cxp_file_list_store_iter_n_children;
	iface->iter_nth_child = cxp_file_list_store_iter_nth_child;
	iface->iter_parent = cxp_file_list_store_iter_parent;
}

static void cxp_file_list_store_sortable_init (GtkTreeSortableIface * iface)
{
	iface->get_sort_column_id = cxp_file_list_store_sortable_get_sort_column_id;
	iface->set_sort_column_id = cxp_file_list_store_sortable_set_sort_column_id;
	iface->set_sort_func = cxp_file_list_store_sortable_set_sort_func;	/* NOT SUPPORTED */
	iface->set_default_sort_func = cxp_file_list_store_sortable_set_default_sort_func;	/* NOT SUPPORTED */
	iface->has_default_sort_func = cxp_file_list_store_sortable_has_default_sort_func;	/* NOT SUPPORTED */
}

static void cxp_file_list_store_init (CxpFileListStore * obj)
{
	obj->priv = g_new0 (CxpFileListStorePrivate, 1);
	/* Initialize private members, etc. */
	obj->priv->store = g_ptr_array_new ();
	obj->priv->column_types[0] = G_TYPE_POINTER;
	obj->priv->column_types[1] = G_TYPE_STRING;
	obj->priv->column_types[2] = G_TYPE_STRING;
	obj->priv->column_types[3] = G_TYPE_UINT;
	obj->priv->column_types[4] = G_TYPE_UINT;
	obj->priv->column_types[5] = G_TYPE_UINT;
	obj->priv->column_types[6] = G_TYPE_BOOLEAN;

	g_assert (FILE_LIST_N_COLUMNS == 7);

	obj->priv->stamp = g_random_int ();

	obj->priv->sort_id = SORT_ID_NONE;
	obj->priv->sort_order = GTK_SORT_ASCENDING;
}

static void cxp_file_list_store_finalize (GObject * object)
{
	CxpFileListStore *cobj;

	cobj = CXP_FILE_LIST_STORE (object);

	/* Free private members, etc. */
	g_ptr_array_free (cobj->priv->store, TRUE);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

CxpFileListStore *cxp_file_list_store_new ()
{
	CxpFileListStore *obj;

	obj = CXP_FILE_LIST_STORE (g_object_new
				   (CXP_TYPE_FILE_LIST_STORE, NULL));

	return obj;
}

/*****************************************************************************
 *
 * cxp_file_list_store_get_flags: tells the rest of the world whether our tree model
 *                        has any special characteristics. In our case,
 *                        we have a list model (instead of a tree), and each
 *                        tree iter is valid as long as the row in question
 *                        exists, as it only contains a pointer to our struct.
 *
 ******************************************************************************/

static GtkTreeModelFlags cxp_file_list_store_get_flags (GtkTreeModel *
							tree_model)
{
	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model),
			      (GtkTreeModelFlags) 0);

	return (GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}


/*****************************************************************************
 *
 *  cxp_file_list_store_get_n_columns: tells the rest of the world how many data
 *                             columns we export via the tree model interface
 *
 ******************************************************************************/

static gint cxp_file_list_store_get_n_columns (GtkTreeModel * tree_model)
{
	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), 0);

	return FILE_LIST_N_COLUMNS;
}


/*****************************************************************************
 *
 * cxp_file_list_store_get_column_type: tells the rest of the world which type of
 *                             data an exported model column contains
 *
 *****************************************************************************/

static GType cxp_file_list_store_get_column_type (GtkTreeModel * tree_model,
						  gint index)
{
	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model),
			      G_TYPE_INVALID);
	g_return_val_if_fail (index < FILE_LIST_N_COLUMNS
			      && index >= 0, G_TYPE_INVALID);

	return CXP_FILE_LIST_STORE (tree_model)->priv->column_types[index];
}


/*****************************************************************************
 *
 * cxp_file_list_store_get_iter: converts a tree path (physical position) into a
 *                      tree iter structure (the content of the iter
 *                     fields will only be used internally by our model).
 *                      We simply store a pointer to our CustomRecord
 *                       structure that represents that row in the tree iter.
 *
 ******************************************************************************/

static gboolean cxp_file_list_store_get_iter (GtkTreeModel * tree_model,
					      GtkTreeIter * iter,
					      GtkTreePath * path)
{
	CxpFileListStore *obj;
	CxpFileInfo *record;
	gint *indices, n, depth;

	g_assert (CXP_IS_FILE_LIST_STORE (tree_model));
	g_assert (path != NULL);

	obj = CXP_FILE_LIST_STORE (tree_model);

	indices = gtk_tree_path_get_indices (path);
	depth = gtk_tree_path_get_depth (path);

	/* we do not allow children */
	g_assert (depth == 1);	/* depth 1 = top level; a list only has top level nodes and no children */

	n = indices[0];		/* the n-th top level row */

	if (n >= obj->priv->store->len || n < 0)
		return FALSE;

	record = g_ptr_array_index (obj->priv->store, n);
	g_assert (record != NULL);
	record->pos = n;

	/* We simply store a pointer to our custom record in the iter */
	iter->stamp = obj->priv->stamp;
	iter->user_data = record;
	iter->user_data2 = NULL;	/* unused */
	iter->user_data3 = NULL;	/* unused */

	return TRUE;
}


/*****************************************************************************
 *
 * cxp_file_list_store_get_path: converts a tree iter into a tree path (ie. the
 *                        physical position of that row in the list).
 *
 ******************************************************************************/
static GtkTreePath *cxp_file_list_store_get_path (GtkTreeModel * tree_model,
						  GtkTreeIter * iter)
{
	GtkTreePath *path;
	CxpFileInfo *record;
	CxpFileListStorePrivate *priv;

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), NULL);
	g_return_val_if_fail (iter != NULL, NULL);
	g_return_val_if_fail (iter->user_data != NULL, NULL);

	priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	record = (CxpFileInfo *) iter->user_data;

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, record->pos);

	return path;
}

/*****************************************************************************
 *
 * cxp_file_list_store_get_value: Returns a row's exported data columns
 *                         (_get_value is what gtk_tree_model_get uses)
 *
 ******************************************************************************/

static void cxp_file_list_store_get_value (GtkTreeModel * tree_model,
					   GtkTreeIter * iter, gint column,
					   GValue * value)
{
	CxpFileInfo *record;
	CxpFileListStorePrivate *priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	g_return_if_fail (CXP_IS_FILE_LIST_STORE (tree_model));
	g_return_if_fail (iter != NULL);
	g_return_if_fail (column < FILE_LIST_N_COLUMNS);

	g_value_init (value, priv->column_types[column]);

	record = (CxpFileInfo *) iter->user_data;

	g_return_if_fail (record != NULL);

	if (record->pos >= priv->store->len)
		g_return_if_reached ();

	switch (column)
	{
	case FILE_LIST_COL_RECORD:
		g_value_set_pointer (value, record);
		break;
	case FILE_LIST_COL_FULLPATH:
		g_value_set_string (value, record->fullpath);
		break;
	case FILE_LIST_COL_MIMETYPE:
		g_value_set_string (value, record->mime_type);
		break;
	case FILE_LIST_COL_FILESIZE:
		g_value_set_uint (value, record->file_size);
		break;
	case FILE_LIST_COL_FILEMODE:
		g_value_set_uint (value, record->file_mode);
		break;
	case FILE_LIST_COL_FILETIME:
		g_value_set_uint (value, record->file_mtime);
		break;
	case FILE_LIST_COL_EDITABLE:
		g_value_set_boolean (value, record->editable);
		break;
	}
}


/*****************************************************************************
 *
 * cxp_file_list_store_iter_next: Takes an iter structure and sets it to point
 *                         to the next row.
 *
 ******************************************************************************/
static gboolean cxp_file_list_store_iter_next (GtkTreeModel * tree_model,
					       GtkTreeIter * iter)
{
	CxpFileInfo *record, *nextrecord;
	CxpFileListStorePrivate *priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), FALSE);

	if (iter == NULL || iter->user_data == NULL)
		return FALSE;

	record = (CxpFileInfo *) iter->user_data;

	/* Is this the last record in the list? */
	if ((record->pos + 1) >= priv->store->len)
		return FALSE;

	nextrecord = g_ptr_array_index (priv->store, record->pos + 1);

	g_assert (nextrecord != NULL);
	/* g_assert ( nextrecord->pos == (record->pos + 1) ); */
	nextrecord->pos = record->pos + 1;

	iter->stamp = priv->stamp;
	iter->user_data = nextrecord;

	return TRUE;
}


/*****************************************************************************
 *
 * cxp_file_list_store_iter_children: Returns TRUE or FALSE depending on whether
 *                             the row specified by 'parent' has any children.
 *                            If it has children, then 'iter' is set to
 *                             point to the first child. Special case: if
 *                            'parent' is NULL, then the first top-level
 *                             row should be returned if it exists.
 *
 *****************************************************************************/
static gboolean cxp_file_list_store_iter_children (GtkTreeModel * tree_model,
						   GtkTreeIter * iter,
						   GtkTreeIter * parent)
{
	CxpFileListStorePrivate *priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	g_return_val_if_fail (parent == NULL
			      || parent->user_data != NULL, FALSE);

	/* this is a list, nodes have no children */
	if (parent)
		return FALSE;

	/* parent == NULL is a special case; we need to return the first top-level row */

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), FALSE);

	/* No rows => no first row */
	if (priv->store->len == 0)
		return FALSE;

	/* Set iter to first item in list */
	iter->stamp = priv->stamp;
	iter->user_data = g_ptr_array_index (priv->store, 0);

	return TRUE;
}


/*****************************************************************************
 *
 * cxp_file_list_store_iter_has_child: Returns TRUE or FALSE depending on whether
 *                              the row specified by 'iter' has any children.
 *                              We only have a list and thus no children.
 *
 ******************************************************************************/
static gboolean cxp_file_list_store_iter_has_child (GtkTreeModel * tree_model,
						    GtkTreeIter * iter)
{
	return FALSE;
}


/*****************************************************************************
 *
 * cxp_file_list_store_iter_n_children: Returns the number of children the row
 *                               specified by 'iter' has. This is usually 0,
 *                               as we only have a list and thus do not have
 *                               any children to any rows. A special case is
 *                               when 'iter' is NULL, in which case we need
 *                               to return the number of top-level nodes,
 *                               ie. the number of rows in our list.
 *
 ******************************************************************************/

static gint cxp_file_list_store_iter_n_children (GtkTreeModel * tree_model,
						 GtkTreeIter * iter)
{
	CxpFileListStorePrivate *priv;

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), -1);
	g_return_val_if_fail (iter == NULL || iter->user_data != NULL, FALSE);

	priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	/* special case: if iter == NULL, return number of top-level rows */
	if (!iter)
		return priv->store->len;

	return 0;		/* otherwise, this is easy again for a list */
}


/*****************************************************************************
 *
 *  cxp_file_list_store_iter_nth_child: If the row specified by 'parent' has any
 *                              children, set 'iter' to the n-th child and
 *                              return TRUE if it exists, otherwise FALSE.
 *                              A special case is when 'parent' is NULL, in
 *                              which case we need to set 'iter' to the n-th
 *                              row if it exists.
 *
 ******************************************************************************/
static gboolean cxp_file_list_store_iter_nth_child (GtkTreeModel * tree_model,
						    GtkTreeIter * iter,
						    GtkTreeIter * parent,
						    gint n)
{
	CxpFileInfo *record;
	CxpFileListStorePrivate *priv;

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (tree_model), FALSE);

	priv = CXP_FILE_LIST_STORE (tree_model)->priv;

	/* a list has only top-level rows */
	if (parent)
		return FALSE;

	/* special case: if parent == NULL, set iter to n-th top-level row */

	if (n >= priv->store->len)
		return FALSE;

	record = g_ptr_array_index (priv->store, n);

	g_assert (record != NULL);
	record->pos = n;

	iter->stamp = priv->stamp;
	iter->user_data = record;

	return TRUE;
}


/*****************************************************************************
 *
 * cxp_file_list_store_iter_parent: Point 'iter' to the parent node of 'child'. As
 *                           we have a list and thus no children and no
 *                           parents of children, we can just return FALSE.
 *
 ******************************************************************************/
static gboolean cxp_file_list_store_iter_parent (GtkTreeModel * tree_model,
						 GtkTreeIter * iter,
						 GtkTreeIter * child)
{
	return FALSE;
}




/*****************************************************************************
 *
 * cxp_file_list_store_append_file:  Empty lists are boring. This function can
 *                              be used in your own code to add rows to the
 *                              list. Note how we emit the "row-inserted"
 *                              signal after we have appended the row
 *                              internally, so the tree view and other
 *                              interested objects know about the new row.
 *
 ******************************************************************************/
void cxp_file_list_store_append_file (CxpFileListStore * obj,
				      const gchar * fullpath,
				      GtkTreeIter * iter)
{
	GtkTreePath *path;
	CxpFileInfo *newrecord;
	CxpFileListStorePrivate *priv;
	guint pos;
	struct stat status;

	g_return_if_fail (CXP_IS_FILE_LIST_STORE (obj));
	g_return_if_fail (fullpath != NULL);

	priv = obj->priv;

	pos = priv->store->len;

	newrecord = g_new0 (CxpFileInfo, 1);

	newrecord->fullpath = g_strdup (fullpath);
	newrecord->mime_type = cxp_get_mime_type_for_file (fullpath);
	lstat (fullpath, &status);
	newrecord->file_size = status.st_size;
	newrecord->file_mode = status.st_mode;
	newrecord->file_mtime = status.st_mtime;
	newrecord->pos = pos;
	newrecord->editable = FALSE;

	g_ptr_array_add (priv->store, newrecord);

	/* inform the tree view and other interested objects
	 *  (e.g. tree row references) that we have inserted
	 *  a new row, and where it was inserted */

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, newrecord->pos);

	cxp_file_list_store_get_iter (GTK_TREE_MODEL (obj), iter, path);

	gtk_tree_model_row_inserted (GTK_TREE_MODEL (obj), path, iter);

	gtk_tree_path_free (path);
}

void cxp_file_list_store_clear (CxpFileListStore * obj)
{
	CxpFileListStorePrivate *priv;

	g_return_if_fail (CXP_IS_FILE_LIST_STORE (obj));

	priv = obj->priv;
	if (priv->store->len != 0)
	{
		g_ptr_array_remove_range (priv->store, 0, priv->store->len);
	}
}


void cxp_file_list_store_rename_file (CxpFileListStore * obj,
				      GtkTreeIter * iter,
				      const gchar * fullpath)
{
	CxpFileInfo *record;

	g_return_if_fail (CXP_IS_FILE_LIST_STORE (obj));
	g_return_if_fail (iter != NULL);

	record = (CxpFileInfo *) iter->user_data;
	if (record->fullpath != NULL)
	{
		g_free (record->fullpath);
	}
	record->fullpath = g_strdup (fullpath);
}

void cxp_file_list_store_set_editable (CxpFileListStore * obj,
				       GtkTreeIter * iter, gboolean editable)
{
	CxpFileInfo *record;

	g_return_if_fail (CXP_IS_FILE_LIST_STORE (obj));
	g_return_if_fail (iter != NULL);

	record = (CxpFileInfo *) iter->user_data;
	record->editable = editable;
}

gboolean cxp_file_list_store_remove (CxpFileListStore * obj, GtkTreeIter * iter)
{
	GtkTreePath *path;
	CxpFileInfo *ptr, *next;
	CxpFileListStorePrivate *priv;
	guint pos;

	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (obj), FALSE);
	g_return_val_if_fail (iter != NULL, FALSE);

	path = cxp_file_list_store_get_path (GTK_TREE_MODEL (obj), iter);
	priv = obj->priv;

	ptr = iter->user_data;
	pos = ptr->pos;
	next = g_ptr_array_index (priv->store, pos + 1);

	g_ptr_array_remove (priv->store, ptr);
	//ptr = (CxpFileInfo *) g_ptr_array_remove (priv->store, ptr);
	gtk_tree_model_row_deleted (GTK_TREE_MODEL (obj), path);
	gtk_tree_path_free (path);
	g_free (ptr->fullpath);
	g_free (ptr->mime_type);
	g_free (ptr);

	if (pos >= priv->store->len)
	{
		iter->stamp = 0;
		return FALSE;
	}
	else
	{
		iter->stamp = priv->stamp;
		next->pos = pos;
		iter->user_data = next;
		return TRUE;
	}
}

/*
 * ToDo:ʲ׽ 
 */
static gboolean cxp_file_list_store_sortable_get_sort_column_id (GtkTreeSortable * sortable, gint * sort_col_id, GtkSortType * order)
{
	CxpFileListStore *obj;

	g_return_val_if_fail (sortable != NULL, FALSE);
	g_return_val_if_fail (CXP_IS_FILE_LIST_STORE (sortable), FALSE);

	obj = CXP_FILE_LIST_STORE (sortable);

	if (sort_col_id)
		*sort_col_id = obj->priv->sort_id;

	if (order)
		*order = obj->priv->sort_order;

	return TRUE;
}


static void
cxp_file_list_store_sortable_set_sort_column_id (GtkTreeSortable * sortable, gint sort_col_id, GtkSortType order)
{
	CxpFileListStore *obj;

	g_return_if_fail (sortable != NULL);
	g_return_if_fail (CXP_IS_FILE_LIST_STORE (sortable));

	obj = CXP_FILE_LIST_STORE (sortable);

	if (obj->priv->sort_id == sort_col_id
	    && obj->priv->sort_order == order)
		return;

	obj->priv->sort_id = sort_col_id;
	obj->priv->sort_order = order;

	cxp_file_list_store_resort (obj);

	/* emit "sort-column-changed" signal to tell any tree views
	 *  that the sort column has changed (so the little arrow
	 *  in the column header of the sort column is drawn
	 *  in the right column)                                     */

	gtk_tree_sortable_sort_column_changed (sortable);
}


static void
cxp_file_list_store_sortable_set_sort_func (GtkTreeSortable * sortable,
				    gint sort_col_id,
				    GtkTreeIterCompareFunc sort_func,
				    gpointer user_data,
				    GtkDestroyNotify destroy_func)
{
	g_warning ("%s is not supported by the CxpFileListStore model.\n",
		   __FUNCTION__);
}


static void
cxp_file_list_store_sortable_set_default_sort_func (GtkTreeSortable * sortable,
					    GtkTreeIterCompareFunc sort_func,
					    gpointer user_data,
					    GtkDestroyNotify destroy_func)
{
	g_warning ("%s is not supported by the CxpFileListStore model.\n",
		   __FUNCTION__);
}


static gboolean
cxp_file_list_store_sortable_has_default_sort_func (GtkTreeSortable * sortable)
{
	return FALSE;
}


static gint cxp_file_list_store_compare_records (gint sort_id, CxpFileInfo * a, CxpFileInfo * b)
{
	switch (sort_id)
	{
	case SORT_ID_NONE:
		return 0;

	case SORT_ID_NAME:
		if ((a->fullpath) && (b->fullpath))
			return strcmp (a->fullpath, b->fullpath);

		if (a->fullpath == b->fullpath)
			return 0;	/* both are NULL */
		else
			return (a->fullpath == NULL) ? -1 : 1;
	case SORT_ID_SIZE:
		return	a->file_size - b->file_size;
	case SORT_ID_TYPE:
		if ((a->mime_type) && (b->mime_type))
			return strcmp (a->mime_type, b->mime_type);

		if (a->mime_type == b->mime_type)
			return 0;	/* both are NULL */
		else
			return (a->mime_type == NULL) ? -1 : 1;
	case SORT_ID_TIME:
		return	a->file_mtime - b->file_mtime;
	}


	g_return_val_if_reached (0);
}


static gint cxp_file_list_store_qsort_compare_func (CxpFileInfo ** a, CxpFileInfo ** b, CxpFileListStore * obj)
{
	gint ret;

	g_assert ((a) && (b) && (obj));

	ret = cxp_file_list_store_compare_records (obj->priv->sort_id, *a, *b);

	/* Swap -1 and 1 if sort order is reverse */
	if (ret != 0 && obj->priv->sort_order == GTK_SORT_DESCENDING)
		ret = (ret < 0) ? 1 : -1;

	return ret;
}


/* static */
void cxp_file_list_store_resort (CxpFileListStore * obj)
{
	GtkTreePath *path;
	gint *neworder, i;
	CxpFileInfo *info;

	g_return_if_fail (obj != NULL);
	g_return_if_fail (CXP_IS_FILE_LIST_STORE (obj));

	if (obj->priv->sort_id == SORT_ID_NONE)
		return;

	if (obj->priv->store->len == 0)
		return;

	/* resort */
	g_ptr_array_sort_with_data (obj->priv->store, (GCompareDataFunc)cxp_file_list_store_qsort_compare_func, obj);

	/* let other objects know about the new order */
	neworder = g_new0 (gint, obj->priv->store->len);

	for (i = 0; i < obj->priv->store->len; ++i)
	{
		/* Note that the API reference might be wrong about
		 * this, see bug number 124790 on bugs.gnome.org.
		 * Both will work, but one will give you 'jumpy'
		 * selections after row reordering. */
		/* neworder[(cxp_file_list_store->rows[i])->pos] = i; */
		info = g_ptr_array_index(obj->priv->store, i);
		neworder[i] = info->pos;
		info->pos = i;
	}

	path = gtk_tree_path_new ();

	gtk_tree_model_rows_reordered (GTK_TREE_MODEL (obj), path, NULL,
				       neworder);

	gtk_tree_path_free (path);
	g_free (neworder);
}

