//
// FileNameFilter.cs
//
// Copyright (C) 2004, 2005 Novell, Inc.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Text.RegularExpressions;
using System.IO;

using Beagle.Util;

namespace Beagle.Daemon.FileSystemQueryable {

	public class FileNameFilter {

		private FileSystemQueryable queryable;

		private static bool Debug = false;
		
		// All user defined excludes, used for determining deltas
		// when the configuration is reloaded.
		private ArrayList excludes = new ArrayList ();

		// User defined paths to exclude
		private ArrayList exclude_paths = new ArrayList ();
		
		// User defined exclude patterns
		private ArrayList exclude_patterns = new ArrayList ();

		// Our default exclude patterns
		private ArrayList exclude_patterns_default = new ArrayList ();

		/////////////////////////////////////////////////////////////

		// Setup our default exclude patterns.

		private void SetupDefaultPatternsToIgnore ()
		{
			// FIXME: This probably shouldn't be hard-wired.  Or should it?
			AddDefaultPatternToIgnore (new string [] {
				                   ".*",
						   "*~",
						   "#*#",
						   "*.o",
						   "*.a",
						   "*.S",
						   "*.la",
						   "*.lo",
						   "*.loT",
						   "*.so",
						   "*.exe",
						   "*.dll",
						   "*.mdb",
						   "*.com",
						   "*.csproj",
						   "*.dsp",
						   "*.dsw",
						   "*.m4",
						   "*.pc",
						   "*.pc.in",
						   "*.in.in",
						   "*.omf",
						   "*.aux",
						   "*.tmp",
						   "autom4te.cache",
						   "po",
						   "aclocal",
						   "Makefile",
						   "Makefile.am",
						   "Makefile.in",
						   "CVS",
						   "SCCS",
						   // Garbage generated by the autotools
						   "conftest",
						   "confdefs.h",
						   "conftest.*",
						   "confstat*",
						   "/conf[0-9]+.sh/",
						   "/conf[0-9]+.file/"
			});
		}

		private void AddDefaultPatternToIgnore (IEnumerable patterns)
		{
			foreach (string pattern in patterns)
				exclude_patterns_default.Add (new ExcludeItem (ExcludeType.Pattern, pattern));
		}
		
		/////////////////////////////////////////////////////////////

		private void AddExclude (ExcludeItem exclude)
		{
			if (Debug)
				Logger.Log.Debug ("FileNameFilter: Adding ExcludeItem (value={0}, type={1})", exclude.Value, exclude.Type);

			switch (exclude.Type) {
			case ExcludeType.Path:
				exclude_paths.Add (exclude);
				queryable.RemoveDirectory (exclude.Value);
				break;
			case ExcludeType.Pattern:
				exclude_patterns.Add (exclude);
				break;
			default: 
				return;
			}

			excludes.Add (exclude);
		}

		private bool RemoveExclude (ExcludeItem exclude)
		{
			if (Debug)
				Logger.Log.Debug ("FileNameFilter: Removing ExcludeItem (value={0}, type={1})", exclude.Value, exclude.Type);

			switch (exclude.Type) {
			case ExcludeType.Path:
				exclude_paths.Remove (exclude);
				break;
			case ExcludeType.Pattern:
				exclude_patterns.Remove (exclude);
				break;
			default: 
				return false;
			}
			
			excludes.Remove (exclude);

			return true;
		}
		
		/////////////////////////////////////////////////////////////

		public FileNameFilter (FileSystemQueryable queryable)
		{
			this.queryable = queryable;

			SetupDefaultPatternsToIgnore ();
			LoadConfiguration ();
		}

		/////////////////////////////////////////////////////////////

		// Load data from configuration. Intersect deltas to the currently active excludes and
		// implement any changes upon notification.

		private void LoadConfiguration () 
		{
			foreach (ExcludeItem exclude in Conf.Indexing.Excludes)
				AddExclude (exclude);

			Conf.Subscribe (typeof (Conf.IndexingConfig), OnConfigurationChanged);
		}

		private void OnConfigurationChanged (Conf.Section section)
		{
			ArrayList exclude_paths_removed = new ArrayList ();

			IList excludes_wanted = Conf.Indexing.Excludes;
			IList excludes_to_add, excludes_to_remove;
			bool clear_fs_state = false;

			ArrayFu.IntersectListChanges (excludes_wanted, 
						      excludes, 
						      out excludes_to_add, 
						      out excludes_to_remove);

			// Process any excludes we think we should remove
			foreach (ExcludeItem exclude in excludes_to_remove) {
				if (exclude.Type == ExcludeType.Pattern)
					clear_fs_state = true;
				else if (exclude.Type == ExcludeType.Path)
					exclude_paths_removed.Add (exclude.Value);
				RemoveExclude (exclude);
			}

			// Process any excludes we found to be new
			foreach (ExcludeItem exclude in excludes_to_add)
				AddExclude (exclude);

			// If an exclude pattern is removed, we need to recrawl everything
			// so that we can index those files which were previously ignored.
			if (clear_fs_state)
				queryable.RecrawlEverything ();

			// Make sure we re-crawl the paths we used to ignored but
			// no longer do.
			foreach (string path in exclude_paths_removed) 
				queryable.Recrawl (path);
		}

		/////////////////////////////////////////////////////////////

		// Try to match any of our current excludes to determine if 
		// we should ignore a file/directory or not.

		public bool Ignore (DirectoryModel parent, string name, bool is_directory) 
		{
			if (Debug)
				Logger.Log.Debug ("*** Ignore Check (parent={0}, name={1}, is_directory={2})", (parent != null) ? parent.FullName : null, name, is_directory);

			// If parent is null, we have a root. But it might not be
			// active anymore so we need to check if it's still in the list.
			if (parent == null && queryable.Roots.Contains (name)) {
				if (Debug)
					Logger.Log.Debug ("*** Ignore Check Passed");
				return false;
			}
			
			string path;
			if (parent != null)
				path = Path.Combine (parent.FullName, name);
			else
				path = name;
			
			// Exclude paths
			foreach (ExcludeItem exclude in exclude_paths)
				if (exclude.IsMatch (path))
					return true;
			
			// Exclude patterns
			foreach (ExcludeItem exclude in exclude_patterns)
				if (exclude.IsMatch (name))
					return true;
			
			// Default exclude patterns
			foreach (ExcludeItem exclude in exclude_patterns_default)
				if (exclude.IsMatch (name))
					return true;
			
			if (parent == null) {
				if (Debug)
					Logger.Log.Debug ("*** Parent is null (name={0}, is_directory={1}", name, is_directory);
				return false;
			}

			// This is kind of a hack, but if parent.Parent is null, we need to pass
			// the full path of the directory as second argument to Ignore to allow
			// us to do the root check.
			return Ignore (parent.Parent, (parent.Parent == null) ? parent.FullName : parent.Name, true);
		}
	}		
}
