
/*

Copyright (C) (2004 - 2005) (Venkata Ramana Enaganti) <ramana@intraperson.com>

This program is free software; you can redistribute it and/or 
modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later 
version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* mutual execlusion based on strings.
   ie locking string names and unlocking.
*/

#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <malloc.h>
#include <stdlib.h>
#include <pthread.h>
#include "miscfuncs.h"
#include "msg.h"
#include "thread.h"
#include "workon.h"

#define WORKON_HASH_SIZE (503) /*OPTIMIZE*/
#define WORKON_CACHE_SIZE (50) /*OPTIMIZE*/

typedef struct wentry {
	char name[ NAME_MAX + 1 ]; /*file/dir name can not exceed more then this*/
	unsigned int hash;
	int in_use;
	pthread_mutex_t wait;	/*hang on here if some one else locked it already*/
	struct wentry *next;
} Wentry;

/* for reuse of malloced memory */
static struct {
	Wentry *list;
	int count;
	pthread_mutex_t lock;
} wcache;

static Wentry **whash;
static pthread_mutex_t hash_lock;

#define wentry_key( s )		( s % WORKON_HASH_SIZE )

/*allocate from the unused previousely allocated memory
or else create new dynamic memory.
*/
static Wentry *wentry_malloc( void )
{
	Wentry *tmp;

	if( wcache.count > 0 )
	{
		pthread_mutex_lock( &wcache.lock );
		if( wcache.count <= 0 )
		{
			pthread_mutex_unlock( &wcache.lock );
			goto dyn_alloc;
		}
		tmp = wcache.list;
		wcache.list = wcache.list->next;
		wcache.count--;
		pthread_mutex_unlock( &wcache.lock );
		return tmp;
	}

dyn_alloc:
	return (Wentry *) malloc( sizeof(Wentry) );
}

static void wentry_free( Wentry *we )
{
	Wentry *tmp;

	if( ! we ) return;

	pthread_mutex_destroy( &we->wait );

	if( wcache.count < WORKON_CACHE_SIZE )
	{
		pthread_mutex_lock( &wcache.lock );
		tmp = wcache.list;
		wcache.list = we;
		we->next = tmp;
		wcache.count++;
		pthread_mutex_unlock( &wcache.lock );
		return;
	}
	else free( we );
}

/*not thread safe. mutual exclusion must be
  setup before calling this.
 */

#define WENTRY_LOCATE( name, hash, dptr )			\
do {								\
	dptr = &( whash[ wentry_key( hash ) ] ); 		\
	while( *dptr ) 						\
	{							\
		if( (*dptr)->hash == hash && 			\
			*name == (*dptr)->name[0] &&		\
		       	! strcmp( name, (*dptr)->name ) ) 	\
			break; 					\
		dptr = &( (*dptr)->next ); 			\
	} 							\
} while( 0 )


int workon_name( const char *name )
{
	unsigned int hash;
	Wentry **dptr, *new_ent;

	if( ! name || ! (*name) )
	{
		msglog( MSG_ERR, "workon_name: invalid name" );
		return 0;
	}

	/*PERFORMANCE:
	we allocate in advance. in almost all cases we
	need new entry to be added.*/
	new_ent = wentry_malloc();
	if( ! new_ent )
	{
		msglog( MSG_ALERT, "workon_name: " \
				"could not allocate memory" );
		return 0;
	}

	hash = string_hash( name );
	string_n_copy( new_ent->name, name, sizeof(new_ent->name) );
	new_ent->hash = hash;
	new_ent->in_use = 1;
	new_ent->next = NULL;
        thread_mutex_init(&new_ent->wait);

	/*Now the actual part*/
 	pthread_mutex_lock( &hash_lock );

	WENTRY_LOCATE( name, hash, dptr );
	if( *dptr ) /*entry exists*/
	{
		(*dptr)->in_use++;
		pthread_mutex_unlock( &hash_lock );

		/*we need to free new entry allocated above*/
		wentry_free( new_ent );
		pthread_mutex_lock( &( (*dptr)->wait ) );
		return 1;
	}

	(*dptr) = new_ent;
	pthread_mutex_unlock( &hash_lock );
	pthread_mutex_lock( &new_ent->wait );

	return 1;
}

void workon_release( const char *name )
{
	Wentry **dptr, *ent, *tmp = NULL;
	unsigned int hash;

	if( ! name || ! (*name) )
	{
		msglog( MSG_ERR, "workon_release: invalid name" );
		return;
	}

	hash = string_hash( name );

	pthread_mutex_lock( &hash_lock );
	WENTRY_LOCATE( name, hash, dptr );
	ent = *dptr;
	if( ! ent ) /*entry does not exist*/
	{
		pthread_mutex_unlock( &hash_lock );
		msglog( MSG_ALERT, "workon_release: " \
				"entry for %s does not exist", name );
		return;
	}

	( ent->in_use )--;
	if( ! ( ent->in_use ) )
	{
		(*dptr) = ent->next;
		tmp = ent;
	}
	pthread_mutex_unlock( &hash_lock );
	pthread_mutex_unlock( &ent->wait );

	if( tmp ) wentry_free( tmp );
}

static void workon_cleanup( void )
{
	int i;
	Wentry *we, *tmp;

	/*clean hash first*/
	pthread_mutex_destroy( &hash_lock );

	for( i = 0 ; i < WORKON_HASH_SIZE ; i ++ )
	{
		if( ! whash[ i ] ) continue;
		we = whash[ i ];
		while( we )
		{
			pthread_mutex_destroy( &we->wait );
			tmp = we;
			we = we->next;
			free( tmp );
		}
	}
	free( whash );

	/*clean cache*/
	pthread_mutex_destroy( &wcache.lock );
	we = wcache.list;

	while( we )
	{
		tmp = we;
		we = we->next;
		free( tmp );
	}
}

void workon_init( void )
{
	whash = ( Wentry ** ) calloc( WORKON_HASH_SIZE, sizeof(Wentry *) );

	if( ! whash )
		msglog( MSG_FATAL, "workon_init: " \
			"could not allocate hash table" );

        thread_mutex_init(&hash_lock);

	wcache.list = NULL;
	wcache.count = 0;
        thread_mutex_init(&wcache.lock);

	if( atexit( workon_cleanup ) )
	{
		workon_cleanup();
		msglog( MSG_FATAL, "workon_init: " \
			"could not reigster cleanup method" );
	}
}
