
/*

Copyright (C) (2004 - 2005) (Venkata Ramana Enaganti) <ramana@intraperson.com>

This program is free software; you can redistribute it and/or 
modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later 
version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <limits.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "msg.h"
#include "miscfuncs.h"
#include "thread.h"
#include "multipath.h"

/*All this code is to keep track of the usage count for virtual directories.
  For example, when both home directories /home/user1 and /home/.user1 accessed,
  usage count for directory user1 is 2.

  We need this tracking for deciding to start backup when usage count gets to 0.*/

#define MULTIPATH_HASH_SIZE (5003) /*OPTIMIZE*/

/*cache of used memory size*/
#define MULTIPATH_CACHE_SIZE (50) /*OPTIMIZE*/

typedef struct mentry {
	char name[ NAME_MAX + 1 ];
	int count;
	unsigned int hash;
	struct mentry *next;
} mentry;

/* for reuse of malloced memory */
static struct {
	mentry *list;
	int count;
} mcache;

static mentry **mhash;
static pthread_mutex_t hash_lock; /*exclusive access for hash table*/

#define mentry_key( s )		( s % MULTIPATH_HASH_SIZE )

/*allocate from the unused previousely allocated memory
  or else create new dynamic memory.

  mutex must be set before calling this.
*/
static mentry *mentry_malloc( void )
{
	mentry *tmp;

	if( mcache.count > 0 )
	{
		tmp = mcache.list;
		mcache.list = mcache.list->next;
		mcache.count--;
		return tmp;
	}

	return (mentry *) malloc( sizeof(mentry) );
}

/*mutex must be set before calling this.*/
static void mentry_free( mentry *me )
{
	mentry *tmp;

	if( ! me ) return;
	if( mcache.count < MULTIPATH_CACHE_SIZE )
	{
		tmp = mcache.list;
		mcache.list = me;
		me->next = tmp;
		mcache.count++;
		return;
	}
	else free( me );
}

/*not thread safe. mutual exclusion must be
  setup before calling this.
 */

#define MENTRY_LOCATE(name, hash, dptr)				\
do {								\
	dptr = &( mhash[ mentry_key( hash ) ] );		\
	while( *dptr )						\
	{							\
		if( (*dptr)->hash == hash &&			\
			*name == (*dptr)->name[0] &&		\
		       	! strcmp( name, (*dptr)->name ) )	\
			break;					\
		dptr = &( (*dptr)->next );			\
	}							\
} while( 0 );

/*public interface. Increment count for the given name.*/
int multipath_inc( const char *name )
{
	unsigned int hash;
	mentry *ent, **dptr;

	if( ! name || ! (*name) )
	{
		msglog( MSG_ERR, "multipath_inc: invalid name" );
		return 0;
	}

	hash = string_hash( name );

	pthread_mutex_lock( &hash_lock );

	MENTRY_LOCATE( name, hash, dptr );
	if( ! *dptr )
	{
		if( ! ( ent = mentry_malloc() ) )
		{
			pthread_mutex_unlock( &hash_lock );
			msglog( MSG_ALERT, "multipath_inc: " \
				"could not allocate memory" );
			return 0;
		}

		ent->count = 1;
		string_n_copy( ent->name, name, sizeof(ent->name) );
		ent->hash = hash;

		ent->next = NULL;
		*dptr = ent;

		pthread_mutex_unlock( &hash_lock );
		return 1;
	}

	(*dptr)->count++;
	pthread_mutex_unlock( &hash_lock );
	return 1;
}

/*public interface. Decrement count for the given name.*/
int multipath_dec( const char *name )
{
	unsigned int hash;
	int count;
	mentry **dptr, *ent;

	if( ! name || ! (*name) )
	{
		msglog( MSG_ERR, "lockfile_unhash: invalid name" );
		return 0;
	}

	hash = string_hash( name );

	pthread_mutex_lock( &hash_lock );

	MENTRY_LOCATE( name, hash, dptr );
	ent = *dptr;
	if( ! ent ) /*entry does not exist*/
	{
		pthread_mutex_unlock( &hash_lock );
		msglog( MSG_ALERT, "multipath_dec: " \
				"entry for %s does not exist", name );
		return -1;
	}

	count = --(ent->count);
	if( ! count )
	{
		(*dptr) = ent->next;
		mentry_free( ent );
	}
	pthread_mutex_unlock( &hash_lock );
	return count;
}

/*cleanup and initialization code*/
static void multipath_clean( void )
{
	int i;
	mentry *me, *tmp;

	/*clean hash first*/
	pthread_mutex_destroy( &hash_lock );

	if( mhash )
	{
		for( i = 0 ; i < MULTIPATH_HASH_SIZE ; i ++ )
		{
			if( ! mhash[ i ] ) continue;
			me = mhash[ i ];
			while( me )
			{
				tmp = me;
				me = me->next;
				free( tmp );
			}
		}
		free( mhash );
	}

	/*clean cache*/
	me = mcache.list;

	while( me )
	{
		tmp = me;
		me = me->next;
		free( tmp );
	}
}

void multipath_init( void )
{
	mhash = ( mentry ** ) calloc( MULTIPATH_HASH_SIZE,
			sizeof(mentry *) );

	if( ! mhash )
		msglog( MSG_FATAL, "multipath_init: " \
			"could not allocate hash table" );

        thread_mutex_init(&hash_lock);
	mcache.list = NULL;
	mcache.count = 0;
	
	if( atexit( multipath_clean ) )
	{
		multipath_clean();
		msglog( MSG_FATAL, "multipath_init: " \
			"could not register cleanup method" );
	}
}
