
/*

Copyright (C) (2004 - 2005) (Venkata Ramana Enaganti) <ramana@intraperson.com>

This program is free software; you can redistribute it and/or 
modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later 
version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/****************************************************************
 for keeping backup requests and for launching backup when timeouts
*****************************************************************/

#include <stdio.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "msg.h"
#include "barg.h"
#include "miscfuncs.h"
#include "thread.h"
#include "time_mono.h"
#include "back.h"

#define BHASH_SIZE		15013 /*OPTIMIZE*/

/*At a time this many backup programs are started*/
#define BACK_START_MAX		300 /*OPTIMIZE*/

#define DFLT_BACK_WAIT		(0)

/*backup programs will not exceed this number at any time*/
#define DFLT_MAX_BPROC		200

/*we give low priority*/
#define DFLT_BPROC_PRI		30

#define PRIORITY_MAX		1
#define PRIORITY_MIN		40

#define BACKUP_WAIT_MAX		86400 /*one day*/

/*		IMPORTANT CHANGES		*/

/* Now, waitpid always follows kill system call
  so that there is no accidental killing of
  some other process because of recyling of pids
  by operating system. For this to happen the code
  depends on pipe fd created for backup. It menas
  if the stdout or stderr closed by backup, Autodir
  assumes backup is dead. Even it is not dead, it will
  be killed by Autodir once stdout and stderr are closed.*/

typedef struct bchild {
	/*hash table info*/
	unsigned int hash;
	int key;

	char dname[NAME_MAX+1];
	char dpath[PATH_MAX+1];
	time_t estamp;	 /*time when entry added to chain*/

	pid_t pid;	/* backup pid*/
	int pipe;	/* for logging backup process output --read*/

	/*for hash double list*/
	struct bchild *next;
	struct bchild *prev;

	/*entry creation time stamp based double linked list*/
	struct bchild *next_t;
	struct bchild *prev_t;

	/* to support sending signal to
	   backup process and wait until it exits*/
	int waiting;
	pthread_cond_t wait_cond;

	int in_bchain;
	struct bchild *bchain_next;
	int back_running;
} Bchild;

static struct {
	Bchild **hash;

	int count; /*number of procs*/
	time_t wait; /*how long to wait before starting backup*/
	int maxproc; /*max backup proc limit*/
	int pri; /*backup process priority*/
	int wait4b; /*do not kill backup process but just wait for it*/
	int nokill; /*do not kill backup process*/

	/*mutex access to hash structure*/
	pthread_mutex_t lock;

	/*thread that keep track of entries
	  which are ready for backup*/
	pthread_t watch; 

	/*wait when there is no entries to work on or
	  when max backup process reached*/
	pthread_cond_t watch_wait;

	/*cleanup method wait on this conditional
	  to be informed of when there are no threads left
	  and its safe to cleanup*/
	pthread_cond_t stop_cond;
	int stop; /* cleanup started? */

	/*for time stamp based double linked list*/
	Bchild *start_t;
	Bchild *end_t;
	Bchild *cur_t;

	pthread_cond_t bchain_wait;
	pthread_mutex_t bchain_lock;
	Bchild *bchain;

	/*command line argument for backup option*/
	char *backup;
} BH;

/*Exported functions. This is the interface for backup services.*/
void (*back_add)( const char *name, const char *path );
int (*back_remove)( const char *name, int force);

/*do nothing functions for use when backup feature not requested*/
static void back_add_null( const char *name, const char *path ) { }
static int back_remove_null( const char *name, int force) { return 1; }


/***********************************************/
/* dynamic memory caching methods	       */
/***********************************************/


/*for dynamic memory cache to minimize dependence on malloc.*/

#define BCACHE_MAX 50

static struct {
	int count;
	Bchild *list;
	pthread_mutex_t lock;
} bcache;

static void bh_entry_free( Bchild *bc )
{
	Bchild *tmp;

	pthread_cond_destroy( &bc->wait_cond );
	if( bcache.count < BCACHE_MAX )
	{
		pthread_mutex_lock( &bcache.lock );
		tmp = bcache.list;
		bcache.list = bc;
		bc->next = tmp;
		bcache.count++;
		pthread_mutex_unlock( &bcache.lock );

		return;
	}
	free(bc);
}

/*use cache if available otherwise malloc*/
static Bchild *bh_entry_allocate( void )
{
	Bchild *tmp;

	if( bcache.count > 0 )
	{
		pthread_mutex_lock( &bcache.lock );
		if( bcache.count <= 0 )
		{
			pthread_mutex_unlock( &bcache.lock );
			goto dalloc;
		}

		tmp = bcache.list;
		bcache.list = bcache.list->next;
		bcache.count--;
		pthread_mutex_unlock( &bcache.lock );

		return tmp;
	}

dalloc:
	return (Bchild *) malloc( sizeof(Bchild) );
}


/***********************************************/
/* hash manipulation		               */
/***********************************************/


#define bh_key( n )	( string_hash( n ) % BHASH_SIZE )

/*Only linked lists and pointers to them are manipulated.
  Mutual exclusion should be in effect before calling this.

  Does not deal with bachin list.
*/
static void bh_entry_release( Bchild *bc )
{
	Bchild *nxt,*prv;

	prv = bc->prev_t;
	nxt = bc->next_t;
	/*update BH structure if anything points to current entry*/
	if( bc == BH.start_t ) BH.start_t = nxt;
	if( bc == BH.cur_t )   BH.cur_t = nxt;
	if( bc == BH.end_t )   BH.end_t = prv;

	/*release links in time based double link list*/
	if( nxt ) nxt->prev_t = prv;
	if( prv ) prv->next_t = nxt;

	/*release links in hash double link list*/
	prv = bc->prev;
	nxt = bc->next;

	if( nxt ) nxt->prev = prv;
	if( prv ) prv->next = nxt;

	/*first entry in hash chain?*/
	if( BH.hash[ bc->key ] == bc )
	       BH.hash[ bc->key ] = nxt;
}

/*
   Manipulate only linked lists and pointers for them. 

   Don't deal with the node data itself 
   except data structure supporting data.

   Mutual exclusion should be in effect before calling this.

   Does not deal with bchain list.
*/
static int bh_entry_add( Bchild *new )
{
	int key;
	unsigned int val;
	Bchild *tmp;

	val = new->hash = string_hash( new->dname );
	key = new->key = val % BHASH_SIZE;
	tmp = BH.hash[ key ];

	/*check to see if it already exists*/
	while( tmp )
	{
		if( val == tmp->hash && 
		    ! strcmp( new->dname, tmp->dname ) )
			return 0;

		if( ! tmp->next ) break;
		tmp = tmp->next;
	}

	/*update hash*/
	if( tmp )
	{
		new->prev = tmp;
		tmp->next = new;
	}
	else /*first entry in a hash chain*/
	{
		new->prev = NULL;
		BH.hash[ key ] = new;
	}
	new->next = NULL;

	/*update time based linked list*/
	new->next_t = NULL;

	if( ! BH.start_t )
	{
		new->prev_t = NULL;
		BH.cur_t = BH.start_t = BH.end_t = new;
		pthread_cond_signal( &BH.watch_wait );
	}
	else
	{
		new->prev_t = BH.end_t;
		if( ! BH.cur_t )
		{
			BH.cur_t = new;
			pthread_cond_signal( &BH.watch_wait );
		}
		BH.end_t->next_t = new;
		BH.end_t = new;
	}

	/*bchain linked list*/
	new->bchain_next = NULL;

	return 1;
}

/*
   Mutual exclusion should be in effect before calling this.
 */
static Bchild *bh_entry_locate( const char *name )
{
	int key;
	unsigned int val;
	Bchild *bc;

	val = string_hash( name );
	key = val % BHASH_SIZE;
	bc = BH.hash[ key ];

	while( bc )
	{
		if( val == bc->hash && 
				*name == bc->dname[0] &&
		    		! strcmp( bc->dname, name ) )
			return bc;
		bc = bc->next;
	}
	return NULL;
}


/***********************************************/
/* forked proc helper functions                */
/***********************************************/


static void reset_signals( void )
{
        sigset_t set;
                                                                                
        sigfillset( &set );

        if( sigprocmask( SIG_UNBLOCK, &set, NULL ) )
		msglog( MSG_ERR|LOG_ERRNO, "reset_signals: sigprocmask" );
}

/*
   For reading stdout,stdin from backup for logging purpose.
   This function code taken from autofs package 'as is' unmodified.
 */

static void backup_log( int pipefd, const char *name )
{
	char errbuf[ MSG_BUFSZ ], *p, *sp;
	int errp = 0, errn;

	do {
		while( ( errn = read( pipefd, errbuf + errp,
				MSG_BUFSZ - errp -2 ) ) == -1
			      && errno == EINTR );
      
		if( errn == -1 )
		{
			msglog( MSG_ERR|LOG_ERRNO, "backup_log: read pipe" );
			return;
		}

		if( ! errn ) break;

		errp += errn;
		sp = errbuf;

		while( errp && ( p = memchr( sp, '\n', errp ) ) )
		{
			*p++ = '\0';
			/* Don't output empty lines */
			if( sp[ 0 ] )
				msglog( MSG_INFO, "%s>> %s", name, sp );
			errp -= ( p - sp );
			sp = p;
		}

		if( errp && sp != errbuf )
			memmove( errbuf, sp, errp );

		if( errp >= MSG_BUFSZ -2 )
		{
		/* Line too long, split */
			errbuf[errp] = '\0';
			msglog( MSG_INFO, "%s>> %s", name, errbuf );
			errp = 0;
		}

	} while( errn > 0 );
	
	close( pipefd );

	if( errp > 0 )
	{
		/* End of file without \n */
		errbuf[ errp ] = '\0';
		msglog( MSG_INFO, "%s>> %s", name, errbuf );
	}
}

/* to be called only from forked child process*/
static void backup_exec( Bchild *bc, int lpipe )
{
	char **argv;

	reset_signals();
	setpriority( PRIO_PROCESS, 0, BH.pri );

	dup2( lpipe, STDOUT_FILENO );
	dup2( lpipe, STDERR_FILENO );
	close( lpipe );

	if( ! backarg_get_argv( bc->dname, bc->dpath, &argv ) )
	{
		fprintf( stderr, "backup_exec: " \
			"could not make argument vector" );
		_exit( EXIT_FAILURE );
	}

	if( execv( argv[0], argv ) )
	{
		fprintf( stderr, "backup_exec: execv: %s",
				strerror( errno ) );
		_exit( EXIT_FAILURE );
	}
}



/***********************************************/
/*for monitoring backup child process          */
/***********************************************/



static int backup_wait(pid_t pid, const char *dname, int block)
{
	int status;
	int options = block ? 0 : WNOHANG;
	int wrv;

again:
	while( ( wrv = waitpid( pid, &status, options ) ) == -1
			&& errno == EINTR );

	if( wrv == -1 )
		msglog( MSG_ERR|LOG_ERRNO, "backup_wait: waitpid" );

 	else if( ! wrv ) return 0;

	else if( wrv > 0 )
	{
		if( WIFEXITED( status ) )
			msglog( MSG_INFO, "backup process for %s " \
					"exited with %d",
					dname, WEXITSTATUS( status ) );
		else if( WIFSIGNALED( status ) )
			msglog( MSG_INFO, "backup process for %s " \
					"caught signal %d",
					dname, WTERMSIG( status ) );
		else
		{
			/*should not get here*/
			msglog( MSG_NOTICE, "backup process for %s stopped",
					dname );
			goto again;
		}
	}
	return 1;
}

/* helper function for backup_monitor*/
static void backup_monitor_cleanup( Bchild *bc )
{
    	pid_t pid;

	pthread_detach( pthread_self() );
	pthread_mutex_lock( &BH.lock );

	/*If this entry still in bchain, we wait.*/
	if( bc->in_bchain )
		while( ( errno = pthread_cond_wait( &( BH.bchain_wait ),
						&BH.lock ) ) == EINTR );

	BH.count--;

	/* cleanup method might be waiting. signal it*/
	if( BH.stop && BH.count == 0 )
		pthread_cond_signal( &BH.stop_cond );

	/*bh_remove is waiting for this entry? Leave cleanup to it.*/
	if( bc->waiting )
	{
		bc->back_running = 0;
		pthread_mutex_unlock( &BH.lock );
		pthread_cond_broadcast( &( bc->wait_cond ) );
		return;
	}

	/*no one waiting for us. we have to free it ourself*/
	bh_entry_release( bc );
	pthread_mutex_unlock( &BH.lock );

	/*Entry removed from hash and no one can see this entry anymore.
	  No probelem of race conditions anymore.*/

	/*make sure backup is not running still somehow.*/
	pid = bc->pid;
	bc->pid = -1;
	if( pid > 1 )
		kill( - pid, SIGKILL );
	backup_wait( pid, bc->dname, 1 );
	bh_entry_free( bc );
	return;
}

/*
  Thread to monitor child backup process -- logging and for cleanup.
  Assumes node on which it is operating on, will not be freed by others.
 */
static void *backup_monitor( void *b )
{
	Bchild *bc = (Bchild *) b;

	backup_log( bc->pipe, bc->dname );
	backup_monitor_cleanup( bc );

	pthread_exit(NULL);
}



/***********************************************/
/* backup starting mechanism		       */
/***********************************************/



/* PERFORMANCE IMPROVEMENTS
	This is major performance improvement done with bchain.
	Previsousely backup programs forked while in main mutual exclusion.
	Now in mutual exculsion state, only integer and pointer manipulations
	are done. Forking is taken outside with additional benifit of
	starting as many backup programs as we wish at a time in between waitings.
*/

static void bchain_process( void )
{
	int pipefd[ 2 ];
	void *tmp;
	Bchild *bc;
	pid_t f;

	for( bc = BH.bchain ; bc ; bc = bc->bchain_next )
	{
		if( bc->waiting || BH.stop )
		{
			bc->back_running = 0;
			continue;
		}

		if( pipe( pipefd ) )
		{
			msglog( MSG_NOTICE|LOG_ERRNO, "could not start " \
					"backup for %s: pipe",
					bc->dname );
			continue;
		}

		if( fcntl( pipefd[ 0 ], F_SETFD, FD_CLOEXEC ) == -1 )
			msglog( MSG_ERR|LOG_ERRNO, "bchain_process: " \
						"fcntl FD_CLOEXEC" );

		msglog( MSG_INFO, "starting backup for %s", bc->dname );
		f = fork();
		if( f < 0 )
		{
			msglog( MSG_NOTICE|LOG_ERRNO, "could not start " \
					"backup for %s: fork",
					bc->dname );
			close( pipefd[ 1 ] );
			close( pipefd[ 0 ] );
			continue;
		}
		else if( f == 0 ) 
		{
			if( setpgrp() == -1 )
				fprintf( stderr, "setpgrp: %s",
						strerror( errno ) );
			close( pipefd[ 0 ] );
			backup_exec( bc, pipefd[ 1 ] ); 
		}
		else
		{
			if( setpgid( f, f ) == -1 && errno != EACCES )
				msglog( MSG_ERR|LOG_ERRNO,
					"bchain_process: setpgid" );

			bc->pid = f;
			bc->pipe = pipefd[ 0 ];
			tmp = (void *) bc;
			close( pipefd[ 1 ] );

			/*set this before thread starts*/
			bc->back_running = 1;

			if( ! thread_new( backup_monitor, tmp, NULL ) )
			{
				bc->back_running = 0;
				close( pipefd[ 0 ] );
				bc->pid = -1;
				kill( - f, SIGKILL );
				backup_wait( f, bc->dname, 1 );

				msglog( MSG_NOTICE,
					"could not start backup for %s",
					bc->dname );
				continue;
			}
		}
	}
}

static void bchain_end( void )
{
	int count = 0;
	pid_t pid;
	Bchild *bc;
	Bchild *failed = NULL, *tmp;

	pthread_mutex_lock( &BH.lock );

	bc = BH.bchain;
	while( bc )
	{
		bc->in_bchain = 0;

		if( bc->back_running )
			count++;
		else /*separate failed ones*/
		{
			if( ! bc->waiting )
			{
				bh_entry_release( bc );
				tmp = failed;
				failed = bc;
				bc = bc->bchain_next;
				failed->bchain_next = tmp;
				continue;
			}
		}
		bc = bc->bchain_next;
	}

	BH.count += count;
	pthread_cond_broadcast( &BH.bchain_wait );
	pthread_mutex_unlock( &BH.lock );

	while( failed )
	{
		tmp = failed;
		failed = failed->bchain_next;

		/*make sure it is not running still somehow.*/
		pid = tmp->pid;
		tmp->pid = -1;
		if( pid > 1 )
		{
			kill( - pid, SIGKILL );
			backup_wait( pid, tmp->dname, 1 );
		}
		bh_entry_free( tmp );
	}
}

/*min must time in seconds to wait
  before forking new backup process.
  to avoid rapid firing of exec.
 */
#define NEW_PROC_WAIT		1

static void pthread_mutex_unlock_vp( void *p )
{
	pthread_mutex_unlock( ( pthread_mutex_t * ) p );
}

/*monitor time based linked list
  and start backup when wait time expired
 */

/* helper functions*/
static void bh_watch_get_lock( void )
{
	pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, NULL );
	pthread_mutex_lock( &BH.lock );

	/*should not happen*/
	while( BH.cur_t && BH.cur_t->pid > -1 )
		BH.cur_t = BH.cur_t->next_t;

	while( ! BH.cur_t ) /*nothing to workon*/
	{
		pthread_cleanup_push( pthread_mutex_unlock_vp,
				(void *) &BH.lock );
		pthread_cond_wait( &BH.watch_wait, &BH.lock );
		pthread_cleanup_pop( 0 );
	}
	
	if( BH.stop )
	{
		pthread_mutex_unlock( &BH.lock );
		pthread_exit( NULL );
	}

}

static void bh_watch_unlock( void )
{
	pthread_mutex_unlock( &BH.lock );
	pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, NULL );
}

/*IMPORTANT: forking is taken out of critical section
to bchains to speed up things*/
static void *bh_watch( void *x )
{
	int dift;
	int i;
	time_t cte;
	Bchild **bchain;

	while( 1 )
	{
		bchain = &BH.bchain;
		*bchain = NULL;

		/* get locks and set something to workon*/
		bh_watch_get_lock();

		cte = time_mono();

		/* still got to wait for starting backup?*/
		if( ( dift = BH.wait - ( cte - BH.cur_t->estamp ) ) > 0 )
		{
			bh_watch_unlock();
			sleep( dift );
			continue;
		}

		/*who is ready for backup? make a list first while mutex locked*/
		for( i = 0 ; i < BACK_START_MAX
				&& ( ! BH.wait || dift <= 0 )
				&& BH.cur_t
				&& BH.count + i <= BH.maxproc; i++ )
		{
			*bchain = BH.cur_t;
			BH.cur_t->in_bchain = 1;
			bchain = &( BH.cur_t->bchain_next );
			*bchain = NULL;
			if( ( BH.cur_t = BH.cur_t->next_t ) )
				dift = BH.wait - ( cte - BH.cur_t->estamp );
		}
		bh_watch_unlock();

		/*someone in bchain list?*/
		if( BH.bchain )
		{
			pthread_mutex_lock( &BH.bchain_lock );
			bchain_process();
			bchain_end();
			pthread_mutex_unlock( &BH.bchain_lock );
		}

		/*take some rest*/
		sleep( NEW_PROC_WAIT );
	}
}


/***********************************************/
/*main interface for requesting backup services*/
/***********************************************/

/*These function pointer are assigned to back_add and back_remove
  if backup feature requested.
*/

static void bh_add( const char *name, const char *path )
{
	int r;
	Bchild *bc;

	if( BH.stop )
		return;

	if( ! ( bc = bh_entry_allocate() ) )
	{
		msglog( MSG_ALERT, "bh_add: could not get free entry" );
		return;
	}

	/* initialize entry data here*/
	string_n_copy( bc->dname, name, sizeof(bc->dname) );
	string_n_copy( bc->dpath, path, sizeof(bc->dpath) );

	bc->pid = -1;
	bc->pipe = -1;
	bc->waiting = 0;
        thread_cond_init(&bc->wait_cond);
	bc->in_bchain = 0;
	bc->back_running = 0;
	bc->estamp = time_mono();

	/*add to the list and update links while mutex locked*/
	pthread_mutex_lock( &BH.lock );
	r = ( BH.stop || bh_entry_locate( name) ) ? 0 : bh_entry_add( bc );
	pthread_mutex_unlock( &BH.lock );

	if( ! r ) bh_entry_free( bc );
}

/*main interface for removing entries from backup hash*/
/*force remove if backup already running.*/
static int bh_remove( const char *name, int force )
{
	int first;
	Bchild *bc;
  	pid_t pid;

	if( BH.stop )
		return 0;

	if( BH.nokill )
		return 1;

	pthread_mutex_lock( &BH.lock );

	if( BH.stop ) /*check again*/
	{
		pthread_mutex_unlock( &BH.lock );
	       	return 0; 
	}

	/* no entry exists. return*/
	if( ! ( bc = bh_entry_locate( name ) ) )
	{
		pthread_mutex_unlock( &BH.lock );
		return 1;
	}

	/*we tell others not to free this entry*/
	first = ( ++( bc->waiting ) == 1 );

	/* entry in bchain list? then wait*/
	if( bc->in_bchain )
		while( ( errno = pthread_cond_wait( &( BH.bchain_wait ),
						&BH.lock ) ) == EINTR );

	if( BH.wait4b && ! force && bc->back_running )
		while( ( errno = pthread_cond_wait( &( bc->wait_cond ),
						&BH.lock ) ) == EINTR );

	pid = bc->pid;
	bc->pid = -1;

	if( first )
		bh_entry_release( bc );
	else
		bc = NULL;

	pthread_mutex_unlock( &BH.lock );

	/* backup process running.
	   Terminate it first then free the entry */
	if( pid > 1 )
	{

		/*send soft signal first and wait for some time*/
		kill( - pid, SIGTERM );

		/*backup ignoring our signal? kill it forcefully*/
		if( ! backup_wait( pid, name, 0 ) )
		{
			sleep( 1 );
			kill( - pid, SIGKILL );
			backup_wait( pid, name, 1);
		}
	}

	if( bc )
		bh_entry_free( bc );

	return 1;
}


/***********************************************/
/*Initialization and clean up		       */
/***********************************************/


/* To terminate all back procs.  */
static void bh_send_signal_all( int sig )
{
	Bchild *bc;

	bc = BH.start_t;
	while( bc )
	{
		if( bc->pid > 1 )
			kill( - bc->pid, sig );
		bc = bc->next_t;
	}
}

/* No thread should be waiting on mutex
  when this cleanup functions is called
*/

static void bh_hash_free( void )
{
	int i;
	Bchild *bc, *tmp;

	for( i = 0 ; i < BHASH_SIZE ; i++ )
	{
		bc = BH.hash[ i ];
		while( bc )
		{
			tmp = bc;
			bc = bc->next;
			free( tmp );
		}
	}
	free( BH.hash );
}

void backup_stop_set(void)
{
    BH.stop = 1;
}

/*before autodir exit*/
void backup_stop( void )
{
	struct timespec timeout;

	if( ! BH.backup ) return;

	/* make sure no backup is started from bchain */
	pthread_mutex_lock( &BH.bchain_lock );
	pthread_mutex_lock( &BH.lock );

	if( BH.count )
	{
		msglog( MSG_INFO, "sending SIGTERM for all backup procs" );
		bh_send_signal_all( SIGTERM );
                thread_cond_timespec( &timeout, 2 );
		while( ( errno = pthread_cond_timedwait( &BH.stop_cond,
						&BH.lock, &timeout ) ) == EINTR );
	}

	if( BH.count )
	{
		msglog( MSG_INFO, "sending SIGKILL for all backup procs" );
		bh_send_signal_all( SIGKILL );
		pthread_cond_wait( &BH.stop_cond, &BH.lock );
	}

	pthread_mutex_unlock( &BH.lock );
	pthread_cancel( BH.watch );
	pthread_join( BH.watch, NULL );
	pthread_mutex_unlock( &BH.bchain_lock );
}

/*before autodir exit*/
static void back_clean( void )
{
	bh_hash_free();

	pthread_cond_destroy( &BH.watch_wait );
	pthread_cond_destroy( &BH.stop_cond );
	pthread_cond_destroy( &BH.bchain_wait );
	pthread_mutex_destroy( &BH.lock );
	pthread_mutex_destroy( &BH.bchain_lock );
}

/* startup initialization*/
void back_init(void)
{
	if( ! BH.backup )
	{
		back_add = back_add_null;
		back_remove = back_remove_null;
		return;
	}

	BH.hash = (Bchild **) calloc( BHASH_SIZE, sizeof(Bchild *) );
	if( ! BH.hash )
		msglog( MSG_FATAL, "bh_init: could not allocate memory" );

	BH.count = 0;
	BH.stop = 0;
	BH.start_t = NULL;
	BH.cur_t = NULL;
	BH.end_t = NULL;
	BH.bchain = NULL;
        thread_mutex_init(&BH.lock);
        thread_mutex_init(&BH.bchain_lock);
        thread_cond_init(&BH.watch_wait);
        thread_cond_init(&BH.stop_cond);
        thread_cond_init(&BH.bchain_wait);

        thread_mutex_init(&bcache.lock);
	bcache.list = NULL;
	bcache.count = 0;

	back_add = bh_add;
	back_remove = bh_remove;

	backarg_init( BH.backup );

	if( ! thread_new_joinable( bh_watch, NULL, &BH.watch ) )
	{
		free( BH.hash );
		msglog( MSG_FATAL, "back_init: could not start new thread" );
	}
	
	if( atexit( back_clean ) )
	{
		pthread_cancel( BH.watch );
		free( BH.hash );
		msglog( MSG_FATAL, "back_init: could not register cleanup method" );
	}
}

/**********command line option handling funtions***************/

void back_option_backup( char ch, char *arg, int valid )
{
	/*argument not given. so make these ineffective*/
	if( ! valid )
	{
		BH.backup = NULL;
		return;
	}
	/*first argument must be absolute path to executable*/
	if( arg[ 0 ] != '/' )
		msglog( MSG_FATAL, "absolute path expected for option -%c",ch);

	BH.backup = arg;
}

void back_option_wait( char ch, char *arg, int valid )
{
	int wt;

	if( ! valid ) BH.wait = DFLT_BACK_WAIT;

	else if( ! string_to_number( arg, &wt ) )
		msglog( MSG_FATAL, "invalid argument for -%c", ch );

	else if( wt > BACKUP_WAIT_MAX )
		msglog( MSG_FATAL, "argument exceeding upper limit for -%c", ch );

	else BH.wait = (time_t) wt;
}

void back_option_wait4b( char ch, char *arg, int valid )
{
	BH.wait4b = valid ? 1 : 0;
}

void back_option_nokill( char ch, char *arg, int valid )
{
	BH.nokill = valid ? 1 : 0;
}

void back_option_max_proc( char ch, char *arg, int valid )
{
	if( ! valid ) BH.maxproc = DFLT_MAX_BPROC;

	else if( ! string_to_number( arg, &BH.maxproc ) ||
		       BH.maxproc == 0 )
		msglog( MSG_FATAL, "invalid argument for -%c", ch );
}

void back_option_bpri( char ch, char *arg, int valid )
{
	int pri;

	if( ! valid ) BH.pri = DFLT_BPROC_PRI;

	else if( ! string_to_number( arg, &pri ) )
		msglog( MSG_FATAL, "invalid argument for -%c", ch );

	else if( pri < PRIORITY_MAX || pri > PRIORITY_MIN )
		msglog( MSG_FATAL, "invalid priority %d for -%c", pri, ch );

	else BH.pri = pri - 21; /*change to -20 to +20 range*/
}

/**********end of command line option handling funtions***************/
