/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_network_broker.c,v 1.60.2.11 2004/11/24 00:00:50 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_network_broker.h"

#include "monitor.h"
#include "worker.h"

#include "marshal.h"

#include <ghttp.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>

#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include <zlib.h>


#define DISABLE_PERSISTENT_CONNECTION		0
#define ENABLE_GZIPPED_DIFFERENCIAL_READ	0


#define CACHE_COMPARE_SIZE	128	/* Ĥѹǽˤ٤ */
#define HEADER_BUFFER_LENGTH	256
#define LOG_MESSAGE_SIZE	4096

#define POLLING_INTERVAL_MILLIS	500
#define POLLING_TIMEOUT_COUNT	60


static void ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass);
static void ochusha_network_broker_init(OchushaNetworkBroker *broker);
static void ochusha_network_broker_finalize(GObject *object);


GType
ochusha_network_broker_get_type(void)
{
  static GType broker_type = 0;

  if (broker_type == 0)
    {
      static const GTypeInfo broker_info =
	{
	  sizeof(OchushaNetworkBrokerClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)ochusha_network_broker_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(OchushaNetworkBroker),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_network_broker_init,
	};

      broker_type = g_type_register_static(G_TYPE_OBJECT,
					   "OchushaNetworkBroker",
					   &broker_info, 0);
    }

  return broker_type;
}


enum {
  OUTPUT_LOG_SIGNAL,
  LAST_SIGNAL
};



static GObjectClass *parent_class = NULL;
static int broker_signals[LAST_SIGNAL] = { 0 };
static GQuark broker_id;
static GQuark broker_buffer_status_id;
static GQuark broker_job_args_id;
static GQuark filedesc_id;
static GQuark worker_sync_object_id;


static void
ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_network_broker_finalize;

  broker_signals[OUTPUT_LOG_SIGNAL] =
    g_signal_new("output_log",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, output_log),
		 NULL, NULL,
		 libochusha_marshal_VOID__STRING,
		 G_TYPE_NONE, 1,
		 G_TYPE_STRING);

  broker_buffer_status_id
    = g_quark_from_static_string("OchushaNetworkBroker::BufferStatus");
  broker_job_args_id
    = g_quark_from_static_string("OchushaNetworkBroker::JobArgs");
  worker_sync_object_id
    = g_quark_from_static_string("OchushaNetworkBroker::SyncObject");
  broker_id
    = g_quark_from_static_string("OchushaNetworkBroker::Broker");
  filedesc_id
    = g_quark_from_static_string("OchushaNetworkBroker::FileDescriptor");

  klass->output_log = NULL;
}


#if ENABLE_NET_CONTEXT
static void
ochusha_network_broker_main(WorkerThread *employee, gpointer args)
{
  OchushaNetworkBroker *broker = (OchushaNetworkBroker *)args;

  g_main_loop_run(broker->main_loop);
}
#endif


static void
ochusha_network_broker_init(OchushaNetworkBroker *broker)
{
#if ENABLE_NET_CONTEXT
  WorkerJob *job = G_NEW0(WorkerJob, 1);

  broker->context = g_main_context_new();
  broker->main_loop = g_main_loop_new(broker->context, FALSE);

  job->canceled = FALSE;
  job->job = ochusha_network_broker_main;
  job->args = broker;

  commit_job(job);
#endif

  broker->config = NULL;
}


static void
ochusha_network_broker_finalize(GObject *object)
{
  OchushaNetworkBroker *broker = (OchushaNetworkBroker *)object;

  if (broker->main_loop != NULL)
    {
      g_main_loop_quit(broker->main_loop);
      g_main_loop_unref(broker->main_loop);
      broker->main_loop = NULL;
    }

  if (broker->context != NULL)
    {
      g_main_context_unref(broker->context);
      broker->context = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaNetworkBroker *
ochusha_network_broker_new(OchushaConfig *config)
{
  OchushaNetworkBroker *broker
    = OCHUSHA_NETWORK_BROKER(g_object_new(OCHUSHA_TYPE_NETWORK_BROKER, NULL));
  broker->config = config;

  return broker;
}


void
ochusha_network_broker_terminate(OchushaNetworkBroker *broker)
{
  g_return_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker));

  /* event sourceƤinterruptꤹ٤ */
  g_main_loop_quit(broker->main_loop);
}


void
ochusha_network_broker_output_log(OchushaNetworkBroker *broker,
				  const gchar *message)
{
  g_signal_emit(G_OBJECT(broker),
		broker_signals[OUTPUT_LOG_SIGNAL],
		0,
		message);
}


static void
ochusha_network_broker_buffer_status_free(OchushaNetworkBrokerBufferStatus *status)
{
  if (status->last_modified != NULL)
    G_FREE(status->last_modified);
  if (status->date != NULL)
    G_FREE(status->date);
  G_FREE(status);
}


static int
read_cache_to_buffer(const OchushaConfig *config, const char *url,
		     OchushaAsyncBuffer *buffer)
{
  int fd = ochusha_config_cache_open_file(config, url, O_RDONLY);

  if (fd < 0)
    return 0;	/* åʤξ */

  /* å夢ꡣ*/
  ochusha_async_buffer_read_file(buffer, fd);

  return buffer->length;
}


static void
write_buffer_to_cache(OchushaNetworkBroker *broker, const char *url,
		      OchushaAsyncBuffer *buffer)
{
  char *tmp_pos;
  gchar message[LOG_MESSAGE_SIZE];
  int fd = ochusha_config_cache_open_file(broker->config, url,
					  O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      if ((tmp_pos = strstr(url, ".dat.gz")) != NULL && tmp_pos[7] == '\0')
	{
	  /* Ķad-hoc塼fordat.gz */
	  int len = 0;
	  gzFile gzfile = gzdopen(fd, "wb");

	  if (gzfile != NULL)
	    {
	      len = gzwrite(gzfile, (void *)buffer->buffer, buffer->length);
	      gzclose(gzfile);
	    }
	  else
	    close(fd);

	  if (len == 0)
	    {
	      ochusha_config_cache_unlink_file(broker->config, url);
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Couldn't update cache file for %s: %s (%d)\n"),
		       url, strerror(errno), errno);
	      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
	      fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		      url, strerror(errno), errno);
#endif
	    }
	}
      else
	{
	  ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
	  close(fd);
	  if (len != buffer->length)
	    {
	      ochusha_config_cache_unlink_file(broker->config, url);
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Couldn't update cache file for %s: %s (%d)\n"),
		       url, strerror(errno), errno);
	      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
	      fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		      url, strerror(errno), errno);
#endif
	    }
	}
    }
  else
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Couldn't open cache file for %s: %s (%d)\n"),
	       url, strerror(errno), errno);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "Couldn't open cache file for %s: %s (%d)\n",
	      url, strerror(errno), errno);
#endif
    }
}


/*
 * ºݤnetwork˥뵡ǽ
 */


static gboolean
setup_common_request_headers(OchushaNetworkBroker *broker,
			     ghttp_request *request,
			     gboolean posting, gboolean allow_proxy)
{
  gchar message[LOG_MESSAGE_SIZE];
  gboolean result = TRUE;
  const OchushaConfig *config = broker->config;

  if (allow_proxy &&
      (config->enable_proxy
       || (config->enable_proxy_only_for_posting && posting))
      && config->proxy_url != NULL)
    {
      int result = ghttp_set_proxy(request, config->proxy_url);
      if (result == 0)
	{
	  if (config->enable_proxy_auth
	      && config->proxy_user != NULL && config->proxy_password != NULL)
	    {
	      result = ghttp_set_proxy_authinfo(request, config->proxy_user,
						config->proxy_password);
	      if (result != 0)
		{
		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Invalid proxy auth info: user=\"%s\", password=\"%s\"\n"),
			   config->proxy_user, config->proxy_password);
		  ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
		  fprintf(stderr, "Invalid proxy auth info.\n");
#endif
		  result = FALSE;
		}
	    }
	}
      else
	{
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Invalid proxy URL: \"%s\"\n"),
		   config->proxy_url);
	  ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "Invalid proxy URL: %s\n", config->proxy_url);
#endif
	  result = FALSE;
	}
    }

  ghttp_set_header(request, http_hdr_User_Agent, OCHUSHA_USER_AGENT);

  return result;
}


/*
 * OchushaAsyncBufferbroker_job_args_idqdataȤƤäĤ롣
 */
typedef struct _NetworkBrokerJobArgs
{
  int length;
  int chunksize;
  char *url;
  char *cache_url;
  char *if_modified_since;

  ghttp_request *request;
} NetworkBrokerJobArgs;


/*
 * OchushaAsyncBufferworker_sync_object_idqdataȤƤäĤ롣
 */
#define OCHUSHA_TYPE_WORKER_SYNC_OBJECT			(worker_sync_object_get_type())
#define OCHUSHA_WORKER_SYNC_OBJECT(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObject))
#define OCHUSHA_WORKER_SYNC_OBJECT_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST((klass), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObjectClass))
#define OCHUSHA_IS_WORKER_SYNC_OBJECT(obj)		(G_TYPE_CHECK_INSTANCE_TYPE((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT))
#define OCHUSHA_IS_WORKER_SYNC_OBJECT_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), OCHUSHA_TYPE_WORKER_SYNC_OBJECT))
#define OCHUSHA_WORKER_SYNC_OBJECT_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObjectClass))


typedef struct _WorkerSyncObject WorkerSyncObject;
typedef struct _WorkerSyncObjectClass WorkerSyncObjectClass;

struct _WorkerSyncObject
{
  GObject parent_object;

  OchushaNetworkBroker *broker;
  Monitor *monitor;
  guint timeout_id;
  guint poll_id;
  volatile guint timeout_count;
  volatile gboolean read_ok;
  volatile gboolean finished;
  volatile gboolean interrupted;
};


struct _WorkerSyncObjectClass
{
  GObjectClass parent_class;
};


static void worker_sync_object_class_init(WorkerSyncObjectClass *klass);
static void worker_sync_object_init(WorkerSyncObject *sync_object);
static void worker_sync_object_finalize(GObject *object);


static GType
worker_sync_object_get_type(void)
{
  static GType sync_object_type = 0;

  if (sync_object_type == 0)
    {
      static const GTypeInfo sync_object_info =
	{
	  sizeof(WorkerSyncObjectClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)worker_sync_object_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(WorkerSyncObject),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)worker_sync_object_init,
	};

      sync_object_type = g_type_register_static(G_TYPE_OBJECT,
						"WorkerSyncObject",
						&sync_object_info, 0);
    }

  return sync_object_type;
}


GObjectClass *sync_object_parent_class = NULL;


static void
worker_sync_object_class_init(WorkerSyncObjectClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  sync_object_parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = worker_sync_object_finalize;
}


static void
worker_sync_object_init(WorkerSyncObject *sync_object)
{
  sync_object->monitor = ochusha_monitor_new(NULL);
}


static void
worker_sync_object_finalize(GObject *object)
{
  WorkerSyncObject *sync_object = OCHUSHA_WORKER_SYNC_OBJECT(object);

#if 0
  fprintf(stderr, "worker_sync_object_finalize.\n");
#endif

  if (sync_object->monitor != NULL)
    {
      ochusha_monitor_free(sync_object->monitor);
      sync_object->monitor = NULL;
    }

  if (sync_object_parent_class->finalize)
    (*sync_object_parent_class->finalize)(object);
}


static WorkerSyncObject *
worker_sync_object_new(OchushaNetworkBroker *broker)
{
  WorkerSyncObject *sync_object
    = OCHUSHA_WORKER_SYNC_OBJECT(g_object_new(OCHUSHA_TYPE_WORKER_SYNC_OBJECT, NULL));
  sync_object->broker = broker;
  return sync_object;
}


#define LOCK_WORKER_SYNC_OBJECT(sync_object)			\
  do								\
    {								\
      ochusha_monitor_enter(sync_object->monitor);		\
    }								\
  while (0)

#define UNLOCK_WORKER_SYNC_OBJECT(sync_object)			\
  do								\
    {								\
      ochusha_monitor_exit(sync_object->monitor);		\
    }								\
  while (0)


static gboolean
timeout_cb(WorkerSyncObject *sync_object)
{
  gboolean finished;
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: timeout_cb(%p)\n", sync_object);
#endif
  g_return_val_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object), FALSE);

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->timeout_count++;
  finished = sync_object->finished;
  ochusha_monitor_notify(sync_object->monitor);
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: timeout_cb(%p) count=%d\n",
	  sync_object, sync_object->timeout_count);
#endif

  if (!finished)
    return TRUE;

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "finishing: timeout_cb(%p) count=%d\n",
	  sync_object, G_OBJECT(sync_object)->ref_count);
#endif

  OCHU_OBJECT_UNREF(sync_object);

  return FALSE;
}


static gboolean
poll_cb(GIOChannel *channel, GIOCondition condition,
	WorkerSyncObject *sync_object)
{
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: poll_cb(%p)\n", sync_object);
#endif
  g_return_val_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object), FALSE);

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  if (condition & (G_IO_IN | G_IO_PRI))
    sync_object->read_ok = TRUE;

  ochusha_monitor_notify(sync_object->monitor);
  sync_object->poll_id = 0;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: poll_cb(%p): count=%d\n",
	  sync_object, G_OBJECT(sync_object)->ref_count);
#endif
  OCHU_OBJECT_UNREF(sync_object);

  return FALSE;
}


static guint
ochusha_network_broker_io_add_watch_full(OchushaNetworkBroker *broker,
					 GIOChannel *channel,
					 int priority,
					 GIOCondition condition,
					 GIOFunc io_func,
					 gpointer user_data,
					 GDestroyNotify notify)
{
#if ENABLE_NET_CONTEXT
  GSource *source;
  guint id;

  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker), 0);
  g_return_val_if_fail(io_func != NULL, 0);

  source = g_io_create_watch(channel, condition);

  if (priority != G_PRIORITY_DEFAULT)
    g_source_set_priority(source, priority);
  g_source_set_callback(source, (GSourceFunc)io_func, user_data, notify);

  id = g_source_attach(source, broker->context);
  g_source_unref(source);

  return id;
#else
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker), 0);
  g_return_val_if_fail(io_func != NULL, 0);

  return g_io_add_watch_full(channel, priority, condition,
			     io_func, user_data, notify);
#endif
}


static guint
ochusha_network_broker_timeout_add_full(OchushaNetworkBroker *broker,
					int priority,
					guint interval,
					GSourceFunc timeout_func,
					gpointer user_data,
					GDestroyNotify notify)
{
#if ENABLE_NET_CONTEXT
  GSource *source;
  guint id;

  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker), 0);
  g_return_val_if_fail(timeout_func != NULL, 0);

  source = g_timeout_source_new(interval);
  if (priority != G_PRIORITY_DEFAULT)
    g_source_set_priority(source, priority);

  g_source_set_callback(source, timeout_func, user_data, notify);
  id = g_source_attach(source, broker->context);
  g_source_unref(source);

  return id;
#else
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker), 0);
  g_return_val_if_fail(timeout_func != NULL, 0);

  return g_timeout_add_full(priority, interval,
			    timeout_func, user_data, notify);
#endif
}


static void
register_polling_function_for_read(WorkerSyncObject *sync_object, int fd)
{
  if (sync_object->poll_id == 0)
    {
      GIOChannel *channel = g_io_channel_unix_new(fd);
      OCHU_OBJECT_REF(sync_object);
      sync_object->poll_id
	= ochusha_network_broker_io_add_watch_full(
				sync_object->broker, channel,
				G_PRIORITY_DEFAULT,
				G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
				(GIOFunc)poll_cb, sync_object, NULL);
      g_io_channel_unref(channel);
#if DEBUG_NETWORK_MOST
      fprintf(stderr,
	      "leaving: register_polling_function_for_read(%p): count=%d\n",
	      sync_object, G_OBJECT(sync_object)->ref_count);
#endif
    }
  sync_object->read_ok = FALSE;
}


static gboolean
http_read_from_url(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer)
{
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  WorkerSyncObject *sync_object = g_object_get_qdata(G_OBJECT(buffer),
						     worker_sync_object_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  const char *url = args->url;
  const char *if_modified_since = args->if_modified_since;
  const char *error_message = NULL;
  gchar message[LOG_MESSAGE_SIZE];

  int sock_fd = -1;
  int status_code = 0;
  ghttp_request *request = NULL;
  ghttp_status state = ghttp_not_done;
  ghttp_current_status current_status;

  ochusha_async_buffer_reset(buffer);
  ochusha_async_buffer_update_length(buffer, 0);

  request = ghttp_request_new();

  if (args->chunksize > 0)
    ghttp_set_chunksize(request, args->chunksize);

  args->request = request;	/* 塹Τˡġ */

  ghttp_set_uri(request, url);
  ghttp_set_type(request, ghttp_type_get);

  if (g_str_has_suffix(url, ".dat.gz") || !g_str_has_suffix(url, ".gz"))
    ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate, gzip");

#if DISABLE_PERSISTENT_CONNECTION
  ghttp_set_header(request, http_hdr_Connection, "close");
#endif

  if (if_modified_since != NULL
      && ochusha_config_cache_file_exist(broker->config, url))
    {
      ghttp_set_header(request, http_hdr_If_Modified_Since, if_modified_since);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "If-Modified-Since: %s\n", if_modified_since);
#endif
    }
  else
    if_modified_since = NULL;

  if (!setup_common_request_headers(broker, request, FALSE, TRUE))
    {
      ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
			_("Proxy setting may be wrong."));

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      if (args->request != NULL)
	{
	  args->request = NULL;
	  ghttp_request_destroy(request);
	}

      return FALSE;
    }    

  ochusha_async_buffer_emit_access_started(buffer);

  snprintf(message, LOG_MESSAGE_SIZE, _("Starting GET request: %s\n"), url);
  ochusha_network_broker_output_log(broker, message);

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  current_status = ghttp_get_status(request);

  sock_fd = -1;
  while (TRUE)
    {
      int body_len;
      const char *body;
      ghttp_proc prev_proc;
      gboolean read_ok;

      prev_proc = current_status.proc;

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      if (sync_object->interrupted)
	{
	  sync_object->interrupted = FALSE;
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	  read_ok = FALSE;
	}
      else if ((read_ok = sync_object->read_ok)
	       || (state == ghttp_not_done && sock_fd == -1))
	{
	  sync_object->timeout_count = 0;
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	  state = ghttp_process(request);
	  current_status = ghttp_get_status(request);
	  if (state == ghttp_not_done && prev_proc == ghttp_proc_response_hdrs
	      && current_status.proc == ghttp_proc_response)
	    {
	      state = ghttp_process(request);
	      current_status = ghttp_get_status(request);
	    }
	}
      else
	{
	  if (read_ok)
	    sync_object->timeout_count = 0;
	  else if (sync_object->timeout_count > POLLING_TIMEOUT_COUNT)
	    {
	      sync_object->finished = TRUE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	      if (args->request != NULL)
		{
		  args->request = NULL;
		  ghttp_request_destroy(request);
		}

	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Access Timeout: %s\n"), url);
#if 0
	      fprintf(stderr, "Access Timeout(%p): %s\n", sync_object, url);
#endif
	      ochusha_async_buffer_emit_access_failed(buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_TIMEOUT,
			    message);
	      ochusha_network_broker_output_log(broker, message);
	      return FALSE;
	    }
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	}

#if DEBUG_NETWORK_MOST
      if (state == ghttp_error)
	{
	  fprintf(stderr, "\n* ghttp_process() returns ghttp_error\n");
	  fprintf(stderr, "sock_fd = %d\n", ghttp_get_socket(request));
	  fprintf(stderr, "error: \"%s\"\n", ghttp_get_error(request));
	  fprintf(stderr, "current_status.proc = %d\n", current_status.proc);
	  fprintf(stderr, "current_status.bytes_read = %d\n", current_status.bytes_read);
	  fprintf(stderr, "current_status.bytes_total = %d\n", current_status.bytes_total);
	}
      else
	{
	  fprintf(stderr, "state = %d, read_ok = %d, sock_fd = %d\n",
		  state, read_ok, sock_fd);
	  fprintf(stderr, "prev_proc = %d\n", prev_proc);
	  fprintf(stderr, "current_status.proc = %d\n", current_status.proc);
	}
#endif

      if (current_status.proc == ghttp_proc_response
	  || (sock_fd > 0 && current_status.proc == ghttp_proc_none))
	{
	  status_code = ghttp_status_code(request);
	  status->status_code = status_code;

	  if (status_code != 200)
	    {
	      if (status_code == 304)
		{
		  const char *tmp_header
		    = ghttp_get_header(request, http_hdr_Date);
		  if (tmp_header != NULL)
		    status->date = G_STRDUP(tmp_header);
		  else
		    status->date = NULL;

		  LOCK_WORKER_SYNC_OBJECT(sync_object);
		  sync_object->finished = TRUE;
		  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

		  if (args->request != NULL)
		    {
		      args->request = NULL;
		      ghttp_request_destroy(request);
		    }

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Cache is fresh: %s\n"), url);
		  ochusha_network_broker_output_log(broker, message);

		  return FALSE;
		}
	      goto error_exit;
	    }
	}

      if (state == ghttp_not_done)
	{
	  if (read_ok)
	    ghttp_flush_response_buffer(request);
	}
      else if (state == ghttp_error)
	{
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "ghttp_process() returns ghttp_error.\n");
#endif
	  error_message = _("Unknown I/O error.");
	  goto error_exit;
	}

      if (read_ok)
	{
	  body = ghttp_get_body(request);
	  body_len = ghttp_get_body_len(request);
	}
      else
	{
	  body = NULL;
	  body_len = 0;
	}
      
      if (body_len > 0)
	{
	  if (current_status.bytes_total > 0)
	    ochusha_async_buffer_emit_access_progressed(buffer,
				  current_status.bytes_read,
				  current_status.bytes_total);

	  if (!ochusha_async_buffer_append_data(buffer, body, body_len))
	    {
	      if (args->request != NULL)
		{
		  args->request = NULL;
		  ghttp_request_destroy(request);
		}

	      status->state
		= OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Access terminated: %s\n"),
		       url);
	      ochusha_network_broker_output_log(broker, message);

	      LOCK_WORKER_SYNC_OBJECT(sync_object);
	      sync_object->finished = TRUE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	      return FALSE;
	    }
	}
      else
	{
	  if (!ochusha_async_buffer_signal(buffer))
	    {
	      if (args->request != NULL)
		{
		  args->request = NULL;
		  ghttp_request_destroy(request);
		}

	      status->state
		= OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Access terminated: %s\n"),
		       url);
	      ochusha_network_broker_output_log(broker, message);

	      LOCK_WORKER_SYNC_OBJECT(sync_object);
	      sync_object->finished = TRUE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	      return FALSE;
	    }
	}

      if (state == ghttp_done)
	break;

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->read_ok = FALSE;
      if (read_ok)
	sync_object->timeout_count = 0;

      if (current_status.proc > ghttp_proc_request)
	{
	  sock_fd = ghttp_get_socket(request);
	  if (sock_fd > 0)
	    register_polling_function_for_read(sync_object, sock_fd);
	}

      if (!sync_object)
	ochusha_monitor_wait(sync_object->monitor);
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
    }

  if (status_code == 200)
    {
      const char *tmp_header
	= ghttp_get_header(request, http_hdr_Last_Modified);
      if (tmp_header != NULL)
	status->last_modified = G_STRDUP(tmp_header);
      else
	status->last_modified = NULL;
      tmp_header = ghttp_get_header(request, http_hdr_Date);
      if (tmp_header != NULL)
	status->date = G_STRDUP(tmp_header);
      else
	status->date = NULL;

      ochusha_async_buffer_emit_access_finished(buffer);

      if (args->request != NULL)
	{
	  args->request = NULL;
	  ghttp_request_destroy(request);
	}

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access completed: %s\n"),
	       url);
      ochusha_network_broker_output_log(broker, message);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      return TRUE;
    }

 error_exit:
  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->finished = TRUE;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

  if (error_message != NULL)
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): status_code=%d (%s)\n"),
	       url, status_code, error_message);
      ochusha_async_buffer_emit_access_failed(buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    message);

      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "access failed: status_code=%d %s\n",
	      status_code, error_message);
#endif
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }
  else if (status_code != 0)
    {
      snprintf(message, LOG_MESSAGE_SIZE, _("Access failed(%s): %d (%s)\n"),
	       url, status_code, ghttp_reason_phrase(request));
      ochusha_async_buffer_emit_access_failed(buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    message);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "%s\n", message);
#endif
    }
  else
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): unknown reason.\n"),
	       url);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "%s\n", message);
#endif
    }

  if (args->request != NULL)
    {
      args->request = NULL;
      ghttp_request_destroy(request);
    }

  return FALSE;

}


static void
destruct_job_args(NetworkBrokerJobArgs *job_args)
{
  if (job_args == NULL)
    return;

  if (job_args->url != NULL)
    G_FREE(job_args->url);

  if (job_args->cache_url != NULL)
    G_FREE(job_args->cache_url);

  if (job_args->if_modified_since != NULL)
    G_FREE(job_args->if_modified_since);

  if (job_args->request != NULL)
    {
      ghttp_request_destroy(job_args->request);
      job_args->request = NULL;
    }

  G_FREE(job_args);
}


static void
wakeup_now_cb(OchushaAsyncBuffer *buffer, WorkerSyncObject *sync_object)
{
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: wakeup_now_cb\n");
#endif

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->interrupted = TRUE;
  ochusha_monitor_notify(sync_object->monitor);
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: wakeup_now_cb\n");
#endif
}


static void
ochusha_network_broker_worker_sync_object_free(WorkerSyncObject *sync_object)
{
  g_return_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object));

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: sync_object_free(%p): count=%d\n",
	  sync_object, G_OBJECT(sync_object)->ref_count);
#endif

  OCHU_OBJECT_UNREF(sync_object);
}


static OchushaAsyncBuffer *
ochusha_network_broker_employ_worker_thread(OchushaNetworkBroker *broker,
					    OchushaAsyncBuffer *buffer,
					    const char *url,
					    const char *cache_url,
					    const char *if_modified_since,
					    JobFunc *job_function,
					    gboolean modestly, int chunksize)
{
  NetworkBrokerJobArgs *args;
  WorkerJob *job;
  WorkerSyncObject *sync_object = worker_sync_object_new(broker);
  OchushaNetworkBrokerBufferStatus *status
    = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  if (buffer == NULL || !ochusha_async_buffer_reset(buffer))
    buffer = ochusha_async_buffer_new(NULL, 0, NULL);

  g_object_set_qdata(G_OBJECT(buffer), broker_id, broker);

  g_object_set_qdata_full(G_OBJECT(buffer), worker_sync_object_id, sync_object,
		(GDestroyNotify)ochusha_network_broker_worker_sync_object_free);
  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id, status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);

  g_signal_connect_object(G_OBJECT(buffer), "wakeup_now",
			  G_CALLBACK(wakeup_now_cb), sync_object, 0);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "\"%s\" installing: timeout_cb(%p) count=%d\n",
	  url, sync_object, G_OBJECT(sync_object)->ref_count);
#endif

  sync_object->timeout_id
    = ochusha_network_broker_timeout_add_full(sync_object->broker,
					      G_PRIORITY_DEFAULT - 1,
					      POLLING_INTERVAL_MILLIS,
					      (GSourceFunc)timeout_cb,
					      sync_object, NULL);

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_UNKNOWN;

  args = G_NEW0(NetworkBrokerJobArgs, 1);
  args->length = buffer->length;
  buffer->length = 0;
  args->url = G_STRDUP(url);
  args->cache_url = G_STRDUP(cache_url);
  args->chunksize = chunksize;
  if (if_modified_since != NULL
      && (args->length > 0
	  || ochusha_config_cache_file_exist(broker->config, url)))
    args->if_modified_since = G_STRDUP(if_modified_since);

  while (g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id) != NULL)
    {
#ifdef HAVE_SCHED_YIELD
      sched_yield();
#else
# ifdef HAVE_NANOSLEEP
      struct timespec rqtp { 0, 100000000 };
      nanosleep(&rqtp, NULL);
# else
#  ifdef HAVE_USLEEP
      usleep(1000);
#  else
      sleep(1);
#  endif
# endif
#endif
    }

  g_object_set_qdata_full(G_OBJECT(buffer), broker_job_args_id,
			  args, (GDestroyNotify)destruct_job_args);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = job_function;
  job->args = buffer;

  /* åɤΤ˳ */
  OCHU_OBJECT_REF(sync_object);
  OCHU_OBJECT_REF(buffer);

  if (modestly)
    commit_modest_job(job);
  else
    commit_job(job);

  return buffer;
}


static void
try_update_cache(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  NetworkBrokerJobArgs *args;
  OchushaNetworkBroker *broker = g_object_get_qdata(G_OBJECT(buffer),
						    broker_id);
  WorkerSyncObject *sync_object = g_object_get_qdata(G_OBJECT(buffer),
						     worker_sync_object_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  char *url;
  int len;
  ghttp_request *request = NULL;
  int status_code = 0;
  const char *error_message = NULL;
  gchar message[LOG_MESSAGE_SIZE];

  if (!ochusha_async_buffer_active_ref(buffer))
    {
      args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
      if (args != NULL)
	url = args->url;
      else
	url = NULL;

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_emit_access_failed(buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."));

      ochusha_async_buffer_fix(buffer);
      g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
      OCHU_OBJECT_UNREF(buffer);

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access terminated: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      return;
    }

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL)
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

      ochusha_async_buffer_fix(buffer);
      ochusha_async_buffer_active_unref(buffer);
      g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
      OCHU_OBJECT_UNREF(buffer);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      return;
    }

  url = args->url;

  if (args->length == 0)
    len = read_cache_to_buffer(broker->config, args->cache_url, buffer);
  else
    {
      len = args->length;
      if (!ochusha_async_buffer_update_length(buffer, len))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
	  ochusha_async_buffer_fix(buffer);
	  ochusha_async_buffer_active_unref(buffer);
	  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
	  OCHU_OBJECT_UNREF(buffer);

	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access terminated: %s\n"), url);
	  ochusha_network_broker_output_log(broker, message);

	  LOCK_WORKER_SYNC_OBJECT(sync_object);
	  sync_object->finished = TRUE;
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	  return;
	}
    }

  if (len > 0)
    {	/* å夢ξ */
      ghttp_status state = ghttp_not_done;
      int sock_fd = -1;
      char header[HEADER_BUFFER_LENGTH];
      char cache_compare_buffer[CACHE_COMPARE_SIZE];
      int offset = len - CACHE_COMPARE_SIZE;
      const char *tmp_header;
      ghttp_current_status current_status;

      if (offset < 0)
	{
	  /* å微 */
	  goto cache_is_dirty;
	}

      if (snprintf(header, HEADER_BUFFER_LENGTH, "bytes=%d-", offset)
	  >= HEADER_BUFFER_LENGTH)
	goto cache_is_dirty;	/* ľͭʤκƼ */

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
      request = ghttp_request_new();
      if (args->chunksize > 0)
	ghttp_set_chunksize(request, args->chunksize);

      args->request = request;
      if (request == NULL)
	{
	  /* Out of memory? */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  error_message = _("Out of memory.");
	  goto error_exit;
	}

      ghttp_set_uri(request, url);

      ghttp_set_type(request, ghttp_type_get);
      ghttp_set_header(request, http_hdr_Connection, "close");

#if ENABLE_GZIPPED_DIFFERENCIAL_READ
      ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate, gzip");
#else
      ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate");
#endif

      ghttp_set_header(request, http_hdr_Range, header);

      if (args->if_modified_since != NULL)
	{
	  ghttp_set_header(request, http_hdr_If_Modified_Since,
			   args->if_modified_since);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "If-Modified-Since: %s\n", args->if_modified_since);
#endif
	}
      if (!setup_common_request_headers(broker, request, FALSE, TRUE))
	{
	  /* ۤäƥåϤɤΤ */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
			_("Proxy setting may be wrong."));

	  if (args->request != NULL)
	    {
	      args->request = NULL;
	      ghttp_request_destroy(request);
	    }

	  goto finish_try_update_cache;
	}

      ochusha_async_buffer_emit_access_started(buffer);

      snprintf(message, LOG_MESSAGE_SIZE, _("Updating cache file: %s\n"),
	       url);
      ochusha_network_broker_output_log(broker, message);

      ghttp_set_sync(request, ghttp_async);
      ghttp_prepare(request);

      memcpy(cache_compare_buffer, (char *)buffer->buffer + offset,
	     CACHE_COMPARE_SIZE);

      if (!ochusha_async_buffer_update_length(buffer, offset))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  if (args->request != NULL)
	    {
	      args->request = NULL;
	      ghttp_request_destroy(request);
	    }

	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access terminated: %s\n"), url);
	  ochusha_async_buffer_emit_access_failed(buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		message);
	  ochusha_network_broker_output_log(broker, message);

	  goto finish_try_update_cache;
	}

      current_status = ghttp_get_status(request);

      while (TRUE)
	{
	  int body_len;
	  const char *body;
	  gboolean before_cache_check = TRUE;
	  ghttp_proc prev_proc;
	  gboolean read_ok;

	  prev_proc = current_status.proc;

	  LOCK_WORKER_SYNC_OBJECT(sync_object);
	  if (sync_object->interrupted)
	    {
	      sync_object->interrupted = FALSE;
	      sync_object->read_ok = FALSE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	      read_ok = FALSE;
	    }
	  else if ((read_ok = sync_object->read_ok)
		   || (state == ghttp_not_done && sock_fd == -1))
	    {
	      sync_object->timeout_count = 0;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	      state = ghttp_process(request);
	      current_status = ghttp_get_status(request);
	      if (state == ghttp_not_done
		  && prev_proc == ghttp_proc_response_hdrs
		  && current_status.proc == ghttp_proc_response)
		{
		  state = ghttp_process(request);
		  current_status = ghttp_get_status(request);
		}
	    }
	  else
	    {
	      sync_object->read_ok = FALSE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	    }

	  if (status_code == 0
	      && (current_status.proc == ghttp_proc_response
		  || (sock_fd > 0
		      && current_status.proc == ghttp_proc_none)))
	    {
	      /* ˤ1٤ʤϤ */
	      status_code = ghttp_status_code(request);
	      if (status_code == 304)
		{
		  tmp_header = ghttp_get_header(request, http_hdr_Date);
		  if (tmp_header != NULL)
		    status->date = G_STRDUP(tmp_header);
		  else
		    status->date = NULL;

		  ochusha_async_buffer_update_length(buffer, len);

		  status->status_code = status_code;
		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;

		  ochusha_async_buffer_emit_access_finished(buffer);

		  if (args->request != NULL)
		    {
		      args->request = NULL;
		      ghttp_request_destroy(request);
		    }

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Cache is fresh: %s\n"), url);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}
	      if (status_code != 206)
		{
		  goto cache_is_dirty;
		}
	    }

	  if (state == ghttp_not_done)
	    {
	      if (read_ok)
		ghttp_flush_response_buffer(request);
	    }
	  else if (state == ghttp_error)
	    {
	      error_message = _("Unknown I/O error.");
	      goto error_exit;
	    }

	  if (read_ok)
	    {
	      body = ghttp_get_body(request);
	      body_len = ghttp_get_body_len(request);
	    }
	  else
	    {
	      body = NULL;
	      body_len = 0;
	    }

	  if (body_len > 0)
	    {
	      if (current_status.bytes_total > 0)
		ochusha_async_buffer_emit_access_progressed(buffer,
				      current_status.bytes_read,
				      current_status.bytes_total);

	      if (!ochusha_async_buffer_append_data(buffer, body, body_len))
		{
		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		  if (args->request != NULL)
		    {
		      args->request = NULL;
		      ghttp_request_destroy(request);
		    }
		  
		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Access terminated: %s\n"), url);
		  ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
			message);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}

	      /* buffer->length >= lenˤʤä饭å */
	      if (before_cache_check && buffer->length >= len)
		{
		  /*
		   * MEMO: ޤ褿顢consumeråɤȶĴ
		   *       ȻפäƤΤμȤߤconsumer
		   *       åɤ鸫ǽΤ ġĤȡ
		   *       դΤǡα
		   */
		  if (memcmp(cache_compare_buffer,
			     (char *)buffer->buffer + offset,
			     CACHE_COMPARE_SIZE) != 0)
		    {
		      goto cache_is_dirty;	/* åäƤ */
		    }
		  else
		    before_cache_check = FALSE;	/* åϿ */
		}
	    }
	  else
	    {
	      if (!ochusha_async_buffer_signal(buffer))
		{
		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		  if (args->request != NULL)
		    {
		      args->request = NULL;
		      ghttp_request_destroy(request);
		    }

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Access terminated: %s\n"),
			   url);
		  ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
			message);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}
	    }

	  if (state == ghttp_done)
	    break;

	  LOCK_WORKER_SYNC_OBJECT(sync_object);
	  sync_object->read_ok = FALSE;
	  if (read_ok)
	    sync_object->timeout_count = 0;

	  if (current_status.proc > ghttp_proc_request)
	    {
	      sock_fd = ghttp_get_socket(request);
	      if (sock_fd > 0)
		register_polling_function_for_read(sync_object, sock_fd);
	    }

	  if (!sync_object->interrupted)
	    ochusha_monitor_wait(sync_object->monitor);
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	}

      if (buffer->length < len)
	goto cache_is_dirty;	/* ͽǡʤäΤ
				 * åäƤ롣
				 */
      tmp_header = ghttp_get_header(request, http_hdr_Last_Modified);
      if (tmp_header != NULL)
	status->last_modified = G_STRDUP(tmp_header);
      else
	status->last_modified = NULL;
      tmp_header = ghttp_get_header(request, http_hdr_Date);
      if (tmp_header != NULL)
	status->date = G_STRDUP(tmp_header);
      else
	status->date = NULL;

      status->status_code = status_code;
      ochusha_async_buffer_emit_access_finished(buffer);

      if (args->request != NULL)
	{
	  args->request = NULL;
	  ghttp_request_destroy(request);
	}
      write_buffer_to_cache(broker, args->cache_url, buffer);

      snprintf(message, LOG_MESSAGE_SIZE, _("Cache updated: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      goto finish_try_update_cache;

    cache_is_dirty:
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY;
      if (request != NULL)
	{
	  args->request = NULL;
	  ghttp_request_destroy(request);
	}

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Cache is dirty: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      if (!ochusha_async_buffer_update_length(buffer, 0))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access terminated: %s\n"), url);
	  ochusha_async_buffer_emit_access_failed(buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		message);
	  ochusha_network_broker_output_log(broker, message);

	  goto finish_try_update_cache;
	}
    }
  else
    {
      /* åʤξ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
    }

  /* å夬ʤääƤ */
  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS)
	status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;

      write_buffer_to_cache(broker, args->cache_url, buffer);
    }
  ochusha_async_buffer_fix(buffer);
  ochusha_async_buffer_active_unref(buffer);
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  OCHU_OBJECT_UNREF(buffer);
  return;

 error_exit:
  if (error_message != NULL)
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): %s\n"), url, error_message);
      ochusha_async_buffer_emit_access_failed(buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    message);
      ochusha_network_broker_output_log(broker, message);
    }
  else
    {
      status_code = ghttp_status_code(request);
      if (status_code != 0)
	{
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access failed(%s): %d (%s)\n"),
		   url, status_code, ghttp_reason_phrase(request));
	  ochusha_async_buffer_emit_access_failed(buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				message);
	  ochusha_network_broker_output_log(broker, message);
	}
      else
	{
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access failed(%s): unknown reason.\n"),
		   url);
	  ochusha_async_buffer_emit_access_failed(buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				message);
	  ochusha_network_broker_output_log(broker, message);
	}
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }

  if (args->request != NULL)
    {
      args->request = NULL;
      ghttp_request_destroy(request);
    }

 finish_try_update_cache:
  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->finished = TRUE;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

  ochusha_async_buffer_fix(buffer);
  ochusha_async_buffer_active_unref(buffer);
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  OCHU_OBJECT_UNREF(buffer);
}


static void
refresh_cache_after_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  NetworkBrokerJobArgs *args;
  OchushaNetworkBroker *broker = g_object_get_qdata(G_OBJECT(buffer),
						    broker_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  gchar message[LOG_MESSAGE_SIZE];

#if DEBUG_THREAD_MOST
  fprintf(stderr, "refresh_cache_after_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_emit_access_failed(buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."));
      ochusha_async_buffer_fix(buffer);
      goto finish_refresh_cache_after_read;
    }

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL)
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_emit_access_failed(buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."));
      ochusha_async_buffer_fix(buffer);
      ochusha_async_buffer_active_unref(buffer);
      goto finish_refresh_cache_after_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
      /* ХåեƤ򥭥å˽񤭹ࡣ*/
      write_buffer_to_cache(broker, args->cache_url, buffer);
    }
  else
    {
      /* ̿Ԥå夬(if_modified_sinceNULL)ʾ */
      /* ochusha_async_buffer_reset(buffer); */
      ochusha_async_buffer_update_length(buffer, 0);
      if (read_cache_to_buffer(broker->config, args->cache_url, buffer))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
	  if (status->status_code == 304)
	    {	/* å夬ä */
	      if (args->if_modified_since != NULL)
		status->last_modified = G_STRDUP(args->if_modified_since);
	      ochusha_async_buffer_emit_access_finished(buffer);
	    }
	  else
	    {	/* ̿˼ԤΤǻʤåȤ */
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Use cached file(%s): due to networking error.\n"),
		       args->url);
	      ochusha_async_buffer_emit_access_failed(buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				message);
	      ochusha_network_broker_output_log(broker, message);
	    }
	}
      else
	{	/* åɤʤ */
	  if (status->status_code == 304)
	    {
	      /* ٤å夬Ĥʤ */
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Couldn't find cache file: %s\n"),
		       args->url);
	      ochusha_async_buffer_emit_access_failed(buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				message);
	      ochusha_network_broker_output_log(broker, message);
	    }
	  else
	    {
	      if (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED)
		ochusha_async_buffer_emit_access_failed(buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				_("Couldn't read data via the network."));
	      else
		ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
			_("Access terminated."));

	    }
	}
    }

  ochusha_async_buffer_fix(buffer);
  ochusha_async_buffer_active_unref(buffer);

 finish_refresh_cache_after_read:
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  OCHU_OBJECT_UNREF(buffer);
}


static void
force_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  NetworkBrokerJobArgs *args;
  OchushaNetworkBroker *broker = g_object_get_qdata(G_OBJECT(buffer),
						    broker_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

#if DEBUG_THREAD_MOST
  fprintf(stderr, "force_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_fix(buffer);
      goto finish_force_read;
    }

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL)
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_fix(buffer);
      ochusha_async_buffer_active_unref(buffer);
      goto finish_force_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (args->if_modified_since != NULL)
    {
      G_FREE(args->if_modified_since);
      args->if_modified_since = NULL;
    }

  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s succeeded.\n", args->url);
#endif
    }
  else
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s failed.\n", args->url);
#endif
    }

  ochusha_async_buffer_fix(buffer);
  ochusha_async_buffer_active_unref(buffer);

 finish_force_read:
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  OCHU_OBJECT_UNREF(buffer);
}


static void
background_read_cache(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  int fd = (int)(long)g_object_get_qdata(G_OBJECT(buffer), filedesc_id);
  OchushaNetworkBroker *broker = g_object_get_qdata(G_OBJECT(buffer),
						    broker_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

  if (!ochusha_async_buffer_active_ref(buffer))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      ochusha_async_buffer_fix(buffer);
      goto finish_job;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;

  if (ochusha_async_buffer_read_file(buffer, fd))
    {
      ochusha_async_buffer_emit_access_finished(buffer);
    }
  else
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
    }

  ochusha_async_buffer_fix(buffer);
  ochusha_async_buffer_active_unref(buffer);

 finish_job:
  OCHU_OBJECT_UNREF(buffer);
  OCHU_OBJECT_UNREF(broker);
}


/*
 * ochusha_network_broker_read_from_url
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǥå夬¸ߤʤNULL֤
 *
 * modeOCHUSHA_NETWORK_BROKER_CACHE_IGNOREǤ硢ɬͥåȥ
 * ǡɤ߹ߡ̤򥭥å夷ʤ
 *
 * ʳξˤɤ߹߻˲餫ηǥå夬ФȤ
 * ˥ͥåȥɹǤˤϥǡ򥭥å¸롣
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_AS_ISξ硢å夬¸ߤˤϤ
 * Τޤ޻Ȥ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATEξ硢å夵Ƥǡ
 * append onlyǹΤǤȸʤå夵Ƥǡ
 * ǿǤκʬΤߤͥåȥɤ߹褦ĩ魯롣⤷append only
 * ǹǤʤäˤɤ߹ľ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESHξ硢å夬ͥå
 * 鴰˥ǡɤ߹ߡͥåȥɹԲǽǤä
 * ˤΤߡå夵ƤǡȤ
 *
 * ͥåȥ˥硢̿˼ԤǤĹ0ΥХåե
 * ֤Τա
 */
OchushaAsyncBuffer *
ochusha_network_broker_read_from_url(OchushaNetworkBroker *broker,
				     OchushaAsyncBuffer *buffer,
				     const char *url,
				     const char *if_modified_since,
				     OchushaNetworkBrokerCacheMode mode,
				     gboolean modestly, int chunksize)
{
  return ochusha_network_broker_read_from_url_full(broker, buffer, url,
						   NULL, if_modified_since,
						   mode, modestly, chunksize);
}


/*
 * ochusha_network_broker_read_from_url_full
 *
 * NULLcache_urlͿ줿硢cache_url򥭡Ȥƥåθ
 * ¸Ԥcache_urlNULLǤϡɤ߹URL(url)򥭡Ȥ
 * Ѥ롣
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǥå夬¸ߤʤNULL֤
 *
 * modeOCHUSHA_NETWORK_BROKER_CACHE_IGNOREǤ硢ɬͥåȥ
 * ǡɤ߹ߡ̤򥭥å夷ʤ
 *
 * ʳξˤɤ߹߻˲餫ηǥå夬ФȤ
 * ˥ͥåȥɹǤˤϥǡ򥭥å¸롣
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_AS_ISξ硢å夬¸ߤˤϤ
 * Τޤ޻Ȥ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATEξ硢å夵Ƥǡ
 * append onlyǹΤǤȸʤå夵Ƥǡ
 * ǿǤκʬΤߤͥåȥɤ߹褦ĩ魯롣⤷append only
 * ǹǤʤäˤɤ߹ľ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESHξ硢å夬ͥå
 * 鴰˥ǡɤ߹ߡͥåȥɹԲǽǤä
 * ˤΤߡå夵ƤǡȤ
 *
 * ͥåȥ˥硢̿˼ԤǤĹ0ΥХåե
 * ֤Τա
 */
OchushaAsyncBuffer *
ochusha_network_broker_read_from_url_full(OchushaNetworkBroker *broker,
					  OchushaAsyncBuffer *buffer,
					  const char *url,
					  const char *cache_url,
					  const char *if_modified_since,
					  OchushaNetworkBrokerCacheMode mode,
					  gboolean modestly, int chunksize)
{
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && broker->config != NULL && url != NULL, NULL);

  if (cache_url == NULL)
    cache_url = url;
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "ochusha_network_broker_read_from_url: %s\n", url);
#endif

  if (mode == OCHUSHA_NETWORK_BROKER_CACHE_IGNORE && broker->config->offline)
    {
#if 0	/* ƤӽФ¦Թǰ褺ȥ */
      ochusha_async_buffer_emit_access_failed(buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_NOT_CONNECTED,
			_("Network not connected on offline mode."));
#endif
      return buffer;
    }

  if (broker->config->offline
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_AS_IS
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY
      || mode == OCHUSHA_NETWORK_BROKER_IMAGE_CACHE)
    {
      /* å夬ä餽򤽤Τޤ޻Ȥ */
      int fd;
      if (buffer != NULL && buffer->length > 0)
	return buffer;

      fd = ochusha_config_image_cache_open_file(broker->config,
						cache_url, O_RDONLY);
      if (fd < 0)
	{
	  char *cache_name = ochusha_utils_url_encode_string(cache_url);
	  fd = ochusha_config_open_file(broker->config, cache_name,
					"image_cache", O_RDONLY);
	  G_FREE(cache_name);

	  if (fd < 0)
	    fd = ochusha_config_cache_open_file(broker->config,
						cache_url, O_RDONLY);
	}
	
      if (fd >= 0)
	{
	  WorkerJob *job = G_NEW0(WorkerJob, 1);
	  OchushaNetworkBrokerBufferStatus *status
	    = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
	  if (buffer == NULL || !ochusha_async_buffer_reset(buffer))
	    buffer = ochusha_async_buffer_new(NULL, 0, NULL);

	  g_object_set_qdata(G_OBJECT(buffer), filedesc_id,
			     (gpointer)(long)fd);
	  g_object_set_qdata(G_OBJECT(buffer), broker_id, broker);
	  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id,
		status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);

	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_UNKNOWN;

	  job->canceled = FALSE;
	  job->job = (JobFunc *)background_read_cache;
	  job->args = buffer;

	  OCHU_OBJECT_REF(broker);
	  OCHU_OBJECT_REF(buffer);

	  commit_loader_job(job);

	  return buffer;
	}

      if (mode != OCHUSHA_NETWORK_BROKER_CACHE_AS_IS
	  || broker->config->offline)
	return buffer;
    }

  switch (mode)
    {
    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATE:
      return ochusha_network_broker_employ_worker_thread(broker, buffer,
						url, cache_url,
						if_modified_since,
						(JobFunc *)try_update_cache,
						modestly, chunksize);

    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH:
    case OCHUSHA_NETWORK_BROKER_CACHE_AS_IS:
      return ochusha_network_broker_employ_worker_thread(broker, buffer,
					url, cache_url, if_modified_since,
					(JobFunc *)refresh_cache_after_read,
					modestly, chunksize);

    case OCHUSHA_NETWORK_BROKER_CACHE_IGNORE:
      return ochusha_network_broker_employ_worker_thread(broker, buffer,
							 url, cache_url, NULL,
							 (JobFunc *)force_read,
							 modestly, chunksize);

    default:
      break;
    }

  /* ߥͭʤΤabort */
  abort();
}


/*
 * ochusha_network_broker_get_response_header:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤λꤷإåƤ֤бإåʤ
 * HTTP쥹ݥ󥹥إåΤΤ̵ˤNULL֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 */
const char *
ochusha_network_broker_get_response_header(OchushaNetworkBroker *broker,
					   OchushaAsyncBuffer *buffer,
					   const char *header)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), NULL);

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return NULL;

  return ghttp_get_header(args->request, header);
}


/*
 * ochusha_network_broker_get_header_names:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤˴ޤޤƤΥإå̾ȡθĿͿ
 * 줿ݥ󥿤˳Ǽ롣
 * ˤ0֤顼ˤ-1֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 *
 * ƤӽФheadersפˤʤäfreeǤ餦
 */
int
ochusha_network_broker_get_header_names(OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					char ***headers, int *num_headers)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer)
		       && headers != NULL && num_headers != NULL, -1);
  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return -1;

  return ghttp_get_header_names(args->request, headers, num_headers);
}


gboolean
ochusha_network_broker_try_post(OchushaNetworkBroker *broker,
				const char *url, const char *server,
				const char *referer,
				const char *cookie, const char *body,
				OchushaNetworkBrokerPostStatus *status)
{
  ghttp_request *request = ghttp_request_new();
#if DEBUG_NETWORK || DEBUG_POST
  char message[LOG_MESSAGE_SIZE];
#endif

  g_return_val_if_fail(request != NULL, FALSE);

  ghttp_set_uri(request, url);
  ghttp_set_type(request, ghttp_type_post);

  setup_common_request_headers(broker, request, TRUE, TRUE);
  ghttp_set_header(request, http_hdr_Host, server);

  /* ɬפʤΤ */
#if 0
  ghttp_set_header(request, http_hdr_Accept, "text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,video/x-mng,image/png,image/jpeg,image/gif;q=0.2,text/css,*/*;q=0.1");
  ghttp_set_header(request, http_hdr_Accept_Language, "ja,en-us;q=0.66,en;q=0.33");
#endif
  ghttp_set_header(request, http_hdr_Accept_Charset, "Shift_JIS,EUC-JP,utf-8;q=0.66,*;q=0.66");
  ghttp_set_header(request, http_hdr_Content_Type, "application/x-www-form-urlencoded");

  ghttp_set_header(request, http_hdr_Connection, "close");
  ghttp_set_header(request, http_hdr_Referer, referer);

  if (cookie != NULL)
    ghttp_set_header(request, "Cookie", cookie);

  snprintf(message, LOG_MESSAGE_SIZE, _("Posting a response to: %s\n"), url);
  ochusha_network_broker_output_log(broker, message);
  snprintf(message, LOG_MESSAGE_SIZE, _("Message Body: %s\n"), body);
  ochusha_network_broker_output_log(broker, message);

  ghttp_set_body(request, (char *)body, strlen(body));
  ghttp_prepare(request);

  if (ghttp_process(request) == ghttp_error)
    {
#if DEBUG_POST
      ochusha_network_broker_output_log(broker, _("Posting failed: ghttp_process() returns ghttp_error\n"));
#endif
#if DEBUG_POST_MOST
      fprintf(stderr, "ghttp_process() returns ghttp_error\n");
#endif
      ghttp_request_destroy(request);
      if (status != NULL)
	{
	  status->status_code = 0;
	  status->body = NULL;
	  status->set_cookie = NULL;
	}

      return FALSE;
    }

  if (status != NULL)
    {
      const char *set_cookie;
      status->status_code = ghttp_status_code(request);
      status->body = G_STRNDUP(ghttp_get_body(request),
			       ghttp_get_body_len(request));
      set_cookie = ghttp_get_header(request, http_hdr_Set_Cookie);
      if (set_cookie != NULL)
	status->set_cookie = G_STRDUP(set_cookie);
      else
	status->set_cookie = NULL;
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	snprintf(message, LOG_MESSAGE_SIZE, "Status: %s (%d)\n",
		 ghttp_reason_phrase(request), status->status_code);
	ochusha_network_broker_output_log(broker, message);
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }

  ghttp_request_destroy(request);

#if DEBUG_POST
  ochusha_network_broker_output_log(broker, _("Posting done.\n"));
#endif

  return TRUE;
}


gboolean
ochusha_network_broker_try_post_raw(OchushaNetworkBroker *broker,
				    const char *url, const char *server,
				    const char **headers, const char *body,
				    OchushaNetworkBrokerPostStatus *status)
{
  ghttp_request *request = ghttp_request_new();
  int i;
#if DEBUG_NETWORK || DEBUG_POST
  char message[LOG_MESSAGE_SIZE];
#endif

  g_return_val_if_fail(request != NULL, FALSE);

  ghttp_set_uri(request, url);
  ghttp_set_type(request, ghttp_type_post);

  setup_common_request_headers(broker, request, TRUE, FALSE);
  if (headers != NULL)
    {
      for (i = 0; headers[i] != NULL; i += 2)
	ghttp_set_header(request, headers[i], headers[i + 1]);
    }
  ghttp_set_header(request, http_hdr_Host, server);

  /* ɬפʤΤ */
  ghttp_set_header(request, http_hdr_Accept_Charset, "Shift_JIS,EUC-JP,utf-8;q=0.66,*;q=0.66");

  ghttp_set_header(request, http_hdr_Connection, "close");

  snprintf(message, LOG_MESSAGE_SIZE, _("Posting to: %s\n"), url);
  ochusha_network_broker_output_log(broker, message);

  ghttp_set_body(request, (char *)body, strlen(body));
  ghttp_prepare(request);
#if 1
  ghttp_load_verify_locations(request, OCHUSHA_CA_BUNDLE_CRT, NULL);
#else
  ghttp_load_verify_locations(request, NULL, NULL);
#endif

  if (ghttp_process(request) == ghttp_error)
    {
#if DEBUG_POST_MOST
      fprintf(stderr, "ghttp_process() returns ghttp_error\n");
#endif
#if DEBUG_NETWORK || DEBUG_POST
      snprintf(message, LOG_MESSAGE_SIZE, _("Posting failed: %s\n"),
	       ghttp_get_error(request));
      ochusha_network_broker_output_log(broker, message);
#else
      ochusha_network_broker_output_log(broker, _("Posting failed: ghttp_process() returns ghttp_error\n"));
#endif
      ghttp_request_destroy(request);
      if (status != NULL)
	{
	  status->status_code = 0;
	  status->body = NULL;
	  status->set_cookie = NULL;
	}

      return FALSE;
    }

  if (status != NULL)
    {
      const char *set_cookie;
      status->status_code = ghttp_status_code(request);
      status->body = G_STRNDUP(ghttp_get_body(request),
			       ghttp_get_body_len(request));
      set_cookie = ghttp_get_header(request, http_hdr_Set_Cookie);
      if (set_cookie != NULL)
	status->set_cookie = G_STRDUP(set_cookie);
      else
	status->set_cookie = NULL;
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	snprintf(message, LOG_MESSAGE_SIZE, "Status: %s (%d)\n",
		 ghttp_reason_phrase(request), status->status_code);
	ochusha_network_broker_output_log(broker, message);
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }

  ghttp_request_destroy(request);

#if DEBUG_POST
  ochusha_network_broker_output_log(broker, _("Posting done.\n"));
#endif

  return TRUE;
}
