//
// simplePageSize.h
//
// Part of KVIEWSHELL - A framework for multipage text/gfx viewers
//
// (C) 2002-2004 Stefan Kebekus
// Distributed under the GPL

// Add header files alphabetically

#ifndef SIMPLEPAGESIZE_H
#define SIMPLEPAGESIZE_H

#include <math.h>
#include <qobject.h>
#include <qsize.h>

class QString;
class QStringList;


/** This class represents page sizes.

    @short Class for storing/retrieving a page size.
    @author Stefan Kebekus <kebekus@kde.org>
    @version 1.0 0
*/

class simplePageSize
{
 public:
  /** Constructs a simplePageSize, and sets the size to 0x0mm,
      i.e. 'invalid' */
  simplePageSize() { pageWidth = pageHeight = 0.0; };
  
  /** Sets the page width and height in mm. Recall that if width or
      height is less or equal than 1mm, the page size is considered
      'invalid' by the isValid() method.  */
  virtual void setPageSize(double width_in_mm, double height_in_mm) { pageWidth = width_in_mm; pageHeight = height_in_mm; }

  /** Returns the page width in millimeters. */
  double      width_in_mm(void) const { return pageWidth; }

  /** Returns the page height in millimeters. */
  double      height_in_mm(void) const { return pageHeight; }

  /** If returns the ratio width/height, if the paper size is
      valid. Returns 1.0 otherwise. */
  double      aspectRatio(void) const { return isValid() ? (pageWidth/pageHeight) : 1.0; };

  /** Converts the size to a QSize. If isValid() is false, the result
      is undefined. */
  QSize       sizeInPixel(double resolution) const {return QSize( (int)(resolution*pageWidth /(10.0 * 2.54) + 0.5), 
								  (int)(resolution*pageHeight/(10.0 * 2.54) + 0.5)); }

  /** If the pageSize is valid, this method returns the zoom value
      required to scale the page size down to 'height' pixels. If the
      pageSize is invalid, an error message is printed, and an
      undefined value is returned. */
  double      zoomForHeight(Q_UINT32 height) const;
  
  /** If the pageSize is valid, this method returns the zoom value
      required to scale the page size down to 'height' pixels. If the
      pageSize is invalid, an error message is printed, and an
      undefined value is returned. */
  double      zoomForWidth(Q_UINT32 width) const;
  
  /** Returns 'yes' if pageWidth and pageHeight are both larger than
      1mm */
  bool isValid() const { return ( (pageWidth > 1.0) && (pageHeight > 1.0) ); }

  /** Returns true if the dimensions of the two sizes differ by at
      most 2mm, false otherwise */
  bool        isNearlyEqual(const simplePageSize &size) const {return (fabs(pageWidth-size.pageWidth) <= 2.0 && fabs(pageHeight-size.pageHeight) <= 2.0); };

 protected:
  double pageWidth;  // in mm
  double pageHeight; // in mm
};


#endif
