/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: CHXStatisticObserverBridge.mm,v 1.5.6.3 2004/07/09 01:49:35 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "CHXStatisticObserverBridge.h"
#include "IHXClientPlayer.h"
#include "HXClientConstants.h"
#include "HXStringUtils.h"
#include "CHXClientDebug.h"

@implementation HXStatisticObserverBridgeUserInfo

- (id)init
{
	[self doesNotRecognizeSelector:@selector(init)];
	return nil;
}

- (id)initWithObserver:(id)observer userInfo:(UInt32)aUserInfo
{
	CHXASSERT( observer ); // @"initWithObserver requires an observer"
	
	self = [super init];
	if ( self )
	{
		m_Observer = observer; // Don't retain to avoid circular references.
		m_OrigUserInfo = aUserInfo;
	}
	return self;
}

- (void)dealloc
{
	[super dealloc];
}

- (BOOL)isEqual:(id)object
{
	if ( [object isMemberOfClass:[self class]] )
	{
		HXStatisticObserverBridgeUserInfo* compareObject = ( HXStatisticObserverBridgeUserInfo* ) object;
		return ( ( m_Observer == [compareObject observer] ) &&
				 ( m_OrigUserInfo == [compareObject origUserInfo] ) );
	}
	return NO;
}

- (id)observer
{
	return m_Observer;
}

- (UInt32)origUserInfo
{
	return m_OrigUserInfo;
}

@end

static HXStatisticsCallbacks gStatisticsCallbacks =
{
	CHXStatisticObserverBridge::OnAddedStatistic,
	CHXStatisticObserverBridge::OnModifiedStatistic,
	CHXStatisticObserverBridge::OnDeletedStatistic
};

CHXStatisticObserverBridge::CHXStatisticObserverBridge( IHXClientPlayer* pIClientPlayer )
	: m_pIClientPlayer( pIClientPlayer )
	, m_StatisticObservers( NULL )
{
	CHXASSERT( m_pIClientPlayer );
}

CHXStatisticObserverBridge::~CHXStatisticObserverBridge( void )
{
	CHXASSERT( [m_StatisticObservers count] <= 0 );
	[m_StatisticObservers release];
}

bool
CHXStatisticObserverBridge::AddObserver( id observer, NSString* pStatisticKey, UInt32 userInfo )
{
	bool outAddedObserver = false;
	if ( !m_StatisticObservers )
	{
		m_StatisticObservers = [[NSMutableDictionary alloc] init];
	}
	if (  m_StatisticObservers )
	{
		NSMutableArray* statisticKeyObservers = ( NSMutableArray* ) [m_StatisticObservers objectForKey:pStatisticKey];
		if ( !statisticKeyObservers )
		{
			statisticKeyObservers = [[[NSMutableArray alloc] init] autorelease];
			if ( statisticKeyObservers )
			{
				[m_StatisticObservers setObject:statisticKeyObservers forKey:pStatisticKey];
			}
		}
		if (  statisticKeyObservers )
		{
			HXStatisticObserverBridgeUserInfo* bridgeUserInfo = [[[HXStatisticObserverBridgeUserInfo alloc] initWithObserver:observer userInfo:userInfo] autorelease];
			unsigned indexPrevAdded = [statisticKeyObservers indexOfObject:bridgeUserInfo];
			if ( indexPrevAdded == NSNotFound )
			{
				outAddedObserver = m_pIClientPlayer->AddStatisticObserver( [pStatisticKey cString], &gStatisticsCallbacks, bridgeUserInfo );
				if ( outAddedObserver )
				{
					[statisticKeyObservers addObject:bridgeUserInfo];
				}
				else if ( [statisticKeyObservers count] <= 0 )
				{
					[m_StatisticObservers removeObjectForKey:pStatisticKey];
				}
			}
		}
	}
	return outAddedObserver;
}

void
CHXStatisticObserverBridge::RemoveObserver( id observer, NSString* pStatisticKey, UInt32 userInfo )
{
	if ( m_StatisticObservers )
	{
		NSMutableArray* statisticKeyObservers = ( NSMutableArray* ) [m_StatisticObservers objectForKey:pStatisticKey];
		if ( statisticKeyObservers )
		{
			HXStatisticObserverBridgeUserInfo* testBridgeUserInfo = [[[HXStatisticObserverBridgeUserInfo alloc] initWithObserver:observer userInfo:userInfo] autorelease];
			unsigned indexToRemove = [statisticKeyObservers indexOfObject:testBridgeUserInfo];
			if ( indexToRemove != NSNotFound )
			{
				m_pIClientPlayer->RemoveStatisticObserver( [pStatisticKey cString], &gStatisticsCallbacks, [statisticKeyObservers objectAtIndex:indexToRemove] );
				[statisticKeyObservers removeObjectAtIndex:indexToRemove];
				if ( [statisticKeyObservers count] <= 0 )
				{
					[m_StatisticObservers removeObjectForKey:pStatisticKey];
				}
			}
		}
	}
}

CFTypeRef
CHXStatisticObserverBridge::CreateCFTypeRefFromValue( int valueType, const unsigned char* pValue )
{
	CFTypeRef outValue = nil;
	switch ( valueType )
	{
		case kValueType32BitSignedInt:
		{
			SInt32* pSInt32Value = ( SInt32* ) pValue;
			outValue = CFNumberCreate( kCFAllocatorDefault, kCFNumberSInt32Type, pSInt32Value );
		}
		break;
		
		case kValueTypeString:
		{
			outValue = CFStringCreateWithCString( kCFAllocatorDefault, ( const char* ) pValue, GuessEncodingForCString( pValue ) );
		}
		break;
	}
	return outValue;
}

void
CHXStatisticObserverBridge::OnAddedStatistic( const char* pStatisticName, int valueType, const unsigned char* pValue, void* observerInfo )
{
	NSAutoreleasePool * pool = [[NSAutoreleasePool alloc] init]; // the calling thread may not have a pool
	
	HXStatisticObserverBridgeUserInfo* bridgeUserInfo = ( HXStatisticObserverBridgeUserInfo* ) observerInfo;
	id observer = [bridgeUserInfo observer];
	if ( [(id)observer respondsToSelector:@selector( onAddedStatistic:value:userInfo: )] )
	{
		NSString* statisticName = ( pStatisticName && *pStatisticName ) ? [NSString stringWithCString:pStatisticName] : [NSString string];
		UInt32 origUserInfo = [bridgeUserInfo origUserInfo];
		CFTypeRef value = CreateCFTypeRefFromValue( valueType, pValue );
		[(id)observer onAddedStatistic:statisticName value:value userInfo:origUserInfo];
		if ( value ) CFRelease( value );
	}
	
	[pool release];
}

void
CHXStatisticObserverBridge::OnModifiedStatistic( const char* pStatisticName, int valueType, const unsigned char* pValue, void* observerInfo )
{
	NSAutoreleasePool * pool = [[NSAutoreleasePool alloc] init]; // the calling thread may not have a pool
	
	HXStatisticObserverBridgeUserInfo* bridgeUserInfo = ( HXStatisticObserverBridgeUserInfo* ) observerInfo;
	id observer = [bridgeUserInfo observer];
	if ( [(id)observer respondsToSelector:@selector( onModifiedStatistic:value:userInfo: )] )
	{
		NSString* statisticName = ( pStatisticName && *pStatisticName ) ? [NSString stringWithCString:pStatisticName] : [NSString string];
		UInt32 origUserInfo = [bridgeUserInfo origUserInfo];
		CFTypeRef value = CreateCFTypeRefFromValue( valueType, pValue );
		[(id)observer onModifiedStatistic:statisticName value:value userInfo:origUserInfo];
		if ( value ) CFRelease( value );
	}
	
	[pool release];
}

void
CHXStatisticObserverBridge::OnDeletedStatistic( const char* pStatisticName, void* observerInfo )
{
	NSAutoreleasePool * pool = [[NSAutoreleasePool alloc] init]; // the calling thread may not have a pool
	
	HXStatisticObserverBridgeUserInfo* bridgeUserInfo = ( HXStatisticObserverBridgeUserInfo* ) observerInfo;
	id observer = [bridgeUserInfo observer];
	if ( [(id)observer respondsToSelector:@selector( onDeletedStatistic:userInfo: )] )
	{
		NSString* statisticName = ( pStatisticName && *pStatisticName ) ? [NSString stringWithCString:pStatisticName] : [NSString string];
		UInt32 origUserInfo = [bridgeUserInfo origUserInfo];
		[(id)observer onDeletedStatistic:statisticName userInfo:origUserInfo];
	}
	
	[pool release];
}

