/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: hxstatus.cpp,v 1.8.2.5 2004/07/09 01:48:55 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include <gtk/gtk.h>

#include "hxstatus.h"

#include "hxplayer.h"

#include "hxplayer-i18n.h"

enum {
    SET_PLAYER_SIGNAL,
    LAST_SIGNAL
};

enum {
    PROP_0,
    PROP_TITLE_OVERRIDE,
    PROP_AUTHOR_OVERRIDE,
    PROP_COPYRIGHT_OVERRIDE,
};


static guint signals[LAST_SIGNAL] = { 0 };

static void hxstatus_display_class_init     (HXStatusDisplayClass* klass);
static void hxstatus_display_init           (HXStatusDisplay* status,
                                             HXStatusDisplayClass* klass);
static void hxstatus_display_real_set_player(HXStatusDisplay* status,
                                             HXPlayer*        player);
static void hxstatus_display_set_property   (GObject         *object,
                                             guint            prop_id,
                                             const GValue    *value,
                                             GParamSpec      *pspec);
static void hxstatus_display_get_property   (GObject         *object,
                                             guint            prop_id,
                                             GValue          *value,
                                             GParamSpec      *pspec);
static void hxstatus_display_finalize       (GObject*         object);
static void hxstatus_display_unrealize      (GtkWidget*       widget);

static GtkVBoxClass* g_parent_class = NULL;

extern "C"
{
static void hsd_play_buffer_contact(HXStatusDisplay* status);
static void hsd_pause(HXStatusDisplay* status);
static void hsd_stop(HXStatusDisplay* status);
static void hsd_unmap(HXStatusDisplay* status);
static void hsd_map(HXStatusDisplay* status);
static void hsd_destroy(HXStatusDisplay* status, HXPlayer* player);
};


GType
hxstatus_display_get_type (void)
{
    static GType hxstatus_display_type = 0;

    if (!hxstatus_display_type)
    {
	static const GTypeInfo hxstatus_display_info =
	    {
		sizeof (HXStatusDisplayClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) hxstatus_display_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (HXStatusDisplay),
		0,		/* n_preallocs */
		(GInstanceInitFunc) hxstatus_display_init,
		NULL,           /* value_table */
	    };

	hxstatus_display_type = g_type_register_static (GTK_TYPE_VBOX, "HXStatusDisplay",
                                                        &hxstatus_display_info, (GTypeFlags)0);
    }

    return hxstatus_display_type;
}


static void
hxstatus_display_class_init (HXStatusDisplayClass* klass)
{
    GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
    GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);

    g_parent_class = (GtkVBoxClass*)g_type_class_peek_parent (klass);

    klass->set_player = hxstatus_display_real_set_player;

    gobject_class->set_property = hxstatus_display_set_property;
    gobject_class->get_property = hxstatus_display_get_property;
    gobject_class->finalize = hxstatus_display_finalize;
    widget_class->unrealize = hxstatus_display_unrealize;
    
    /* Action signals */    
    signals[SET_PLAYER_SIGNAL] =
        g_signal_new("set_player",
                      G_OBJECT_CLASS_TYPE(gobject_class),
                      (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
                      G_STRUCT_OFFSET(HXStatusDisplayClass, set_player),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__POINTER,
                      G_TYPE_NONE, 1, G_TYPE_POINTER);

    /* Properties */
    g_object_class_install_property (gobject_class,
                                     PROP_TITLE_OVERRIDE,
                                     g_param_spec_string ("title_override",
                                                        _("Title override"),
                                                        _("Overrides title value"),
                                                          NULL,
                                                          (GParamFlags)G_PARAM_READWRITE));
    g_object_class_install_property (gobject_class,
                                     PROP_AUTHOR_OVERRIDE,
                                     g_param_spec_string ("author_override",
                                                        _("Author override"),
                                                        _("Overrides author value"),
                                                          NULL,
                                                          (GParamFlags)G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     PROP_COPYRIGHT_OVERRIDE,
                                     g_param_spec_string ("copyright_override",
                                                        _("Copyright override"),
                                                        _("Overrides copyright value"),
                                                          NULL,
                                                          (GParamFlags)G_PARAM_READWRITE));
}

static void
hxstatus_display_init(HXStatusDisplay* status, HXStatusDisplayClass* /* klass */)
{
    status->player = NULL;

    status->title_override = NULL;
    status->author_override = NULL;
    status->copyright_override = NULL;

    status->timer_id = 0;
    status->timer_is_running = FALSE;

    status->player_status_signal_handlers_array = g_array_new (FALSE, FALSE, sizeof (gint));
    status->player_status_signal_handlers_array_len = 0;

    g_signal_connect_swapped(G_OBJECT(status), "unmap",
                             G_CALLBACK(hsd_unmap),
                             status);
    
    g_signal_connect_swapped(G_OBJECT(status), "map",
                             G_CALLBACK(hsd_map),
                             status);
}

void
hxstatus_display_set_player(HXStatusDisplay* status, HXPlayer* player)
{
    g_signal_emit(G_OBJECT(status), signals[SET_PLAYER_SIGNAL], 0, player);
}

void
hxstatus_display_real_set_player(HXStatusDisplay* status, HXPlayer* player)
{
    guint i;
    guint signal;
    
    static const struct
    {
        const gchar* name;
        GCallback callback;
    } signal_map[] = 
    {
        { "play",       G_CALLBACK(hsd_play_buffer_contact) },
        { "pause",      G_CALLBACK(hsd_pause)               },
        { "stop",       G_CALLBACK(hsd_stop)                },
        { "buffering",  G_CALLBACK(hsd_play_buffer_contact) },
        { "contacting", G_CALLBACK(hsd_play_buffer_contact) },
        { "destroy",    G_CALLBACK(hsd_destroy)             },
    };

    /* Disconnect old player */
    if(status->player)
    {
        gint signal;
        guint i;

        for(i = 0; i < status->player_status_signal_handlers_array_len; i++)
        {
            signal = g_array_index(status->player_status_signal_handlers_array, gint, i);
            g_signal_handler_disconnect(G_OBJECT(status->player), signal);                                        
        }
    }        

    status->player_status_signal_handlers_array_len = 0;

    /* Swap in the new player */
    if(status->player)
    {
       g_object_unref(G_OBJECT(status->player));
    }

    if(player)
    {
       g_object_ref(G_OBJECT(player));
    }
       
    status->player = player;

    /* Connect the new player */
    if(status->player)
    {
        for(i = 0; i < sizeof(signal_map) / sizeof(*signal_map); i++)
        {    
            /* Hook up to the new player */
            signal = g_signal_connect_swapped(G_OBJECT(player),
                                              signal_map[i].name,
                                              signal_map[i].callback,
                                              status);

            g_array_insert_val(status->player_status_signal_handlers_array,
                               status->player_status_signal_handlers_array_len,
                               signal);
            status->player_status_signal_handlers_array_len++;
        }
    }
}

static void
hxstatus_display_unrealize (GtkWidget* widget)
{
    HXStatusDisplay* status = HX_STATUS_DISPLAY(widget);

    if(status->player)
    {
        hxstatus_display_set_player(status, NULL);
    }    

    GTK_WIDGET_CLASS(g_parent_class)->unrealize(widget);
}

static void
hxstatus_display_finalize(GObject* object)
{
    HXStatusDisplay* status = HX_STATUS_DISPLAY(object);
    g_return_if_fail(status != NULL);

    if(status->title_override)
    {
        g_free(status->title_override);
        status->title_override = NULL;
    }

    if(status->author_override)
    {
        g_free(status->author_override);
        status->author_override = NULL;
    }

    if(status->copyright_override)
    {
        g_free(status->copyright_override);
        status->copyright_override = NULL;
    }

    if(status->player_status_signal_handlers_array)
    {
        g_array_free(status->player_status_signal_handlers_array, TRUE);
    }

    status->player_status_signal_handlers_array_len = 0;

    hxstatus_display_stop_timer(status);

    G_OBJECT_CLASS(g_parent_class)->finalize(object);
}

static void
hxstatus_display_get_property(GObject         *object,
                              guint            prop_id,
                              GValue          *value,
                              GParamSpec      *pspec)
{
    HXStatusDisplay* status = HX_STATUS_DISPLAY(object);
    
    switch(prop_id)
    {
        case PROP_TITLE_OVERRIDE:
            g_value_set_string(value, status->title_override);
            break;
            
        case PROP_AUTHOR_OVERRIDE:
            g_value_set_string(value, status->author_override);
            break;
            
        case PROP_COPYRIGHT_OVERRIDE:
            g_value_set_string(value, status->copyright_override);
            break;
            
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
 
    }
}

static void
hxstatus_display_set_property(GObject         *object,
                              guint            prop_id,
                              const GValue    *value,
                              GParamSpec      *pspec)
{
    HXStatusDisplay* status = HX_STATUS_DISPLAY(object);

    switch(prop_id)
    {
        case PROP_TITLE_OVERRIDE:
            if(status->title_override)
            {
                g_free(status->title_override);
            }   
            status->title_override = g_strdup(g_value_get_string(value));
            break;
            
        case PROP_AUTHOR_OVERRIDE:
            if(status->author_override)
            {
                g_free(status->author_override);
            }   
            status->author_override = g_strdup(g_value_get_string(value));
            break;
            
        case PROP_COPYRIGHT_OVERRIDE:
            if(status->copyright_override)
            {
                g_free(status->copyright_override);
            }   
            status->copyright_override = g_strdup(g_value_get_string(value));
            break;
            
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
 
    }    
}

void
hxstatus_display_start_timer(HXStatusDisplay* status)
{
    g_return_if_fail(status->player != NULL);
    g_return_if_fail(status->timer_function != NULL);
    
    HXContentStateType state;
    gboolean play_contact_buffer = FALSE;
    
    state = hx_player_get_content_state(status->player);

    play_contact_buffer = (state == HX_CONTENT_STATE_PLAYING ||
                           state == HX_CONTENT_STATE_CONTACTING ||
                           state == HX_CONTENT_STATE_BUFFERING);
    
    if(!status->timer_is_running &&
       GTK_WIDGET_VISIBLE(GTK_WIDGET(status)) &&
       (play_contact_buffer ||
        !status->timer_disable_while_not_playing)
       )
    {
        g_assert(status->timer_id == 0);
        status->timer_is_running = TRUE;
        status->timer_id =
            gtk_timeout_add(status->timer_interval,
                            status->timer_function,
                            status->timer_data);     
    }
}

void
hxstatus_display_stop_timer(HXStatusDisplay* status)
{
    if(status->timer_is_running)
    {
        status->timer_is_running = FALSE;
        gtk_timeout_remove(status->timer_id);
        status->timer_id = 0;
    }
}


void
hsd_destroy(HXStatusDisplay* status, HXPlayer* player)
{
    hxstatus_display_set_player(status, NULL);
}

static void
hsd_play_buffer_contact(HXStatusDisplay* status)
{
    if(status->timer_function)
    {
        hxstatus_display_start_timer(status);

        if(status->timer_fire_on_state_changes)
        {
            /* Fire the timer once when we initially play */
            (*status->timer_function) (status->timer_data);
        }
    }
}

static void
hsd_pause(HXStatusDisplay* status)
{
    if(status->timer_function)
    {
        hxstatus_display_stop_timer(status);

        if(status->timer_fire_on_state_changes)
        {
            /* Fire the timer once when we initially pause */
            (*status->timer_function) (status->timer_data);
        }
    }
}

static void
hsd_stop(HXStatusDisplay* status)
{
    if(status->timer_function)
    {
        hxstatus_display_stop_timer(status);

        if(status->timer_fire_on_state_changes)
        {
            /* Fire the timer once when we initially stop */
            (*status->timer_function) (status->timer_data);
        }
    }
}

static void
hsd_unmap(HXStatusDisplay* status)
{
    if(status->timer_function)
    {
        /* Stop the update timer */
        hxstatus_display_stop_timer(status);
    }
}

static void
hsd_map(HXStatusDisplay* status)
{
    if(status->timer_function)
    {
        /* Try to start the update timer */
        hxstatus_display_start_timer(status);
    }
}

void
hxstatus_display_set_timer(HXStatusDisplay* status,
                           guint            interval,
                           GtkFunction      function,
                           gpointer         data,
                           gboolean         disable_while_not_playing,
                           gboolean         fire_on_state_changes)
{
    if(status->timer_is_running)
    {
        /* We only support one update timer. Stop the current timer. */
       hxstatus_display_stop_timer(status);
    }

    status->timer_interval = interval;
    status->timer_function = function;
    status->timer_disable_while_not_playing = disable_while_not_playing;
    status->timer_fire_on_state_changes = fire_on_state_changes;
    status->timer_data = data;
    
    hxstatus_display_start_timer(status);
}

void
hxstatus_display_change_timer_interval(HXStatusDisplay* status,
                                       guint            interval)
{    
    status->timer_interval = interval;    
    if(status->timer_is_running)
    {
        hxstatus_display_stop_timer(status);
        hxstatus_display_start_timer(status);
    }
}
