#include <mip6log.h>
#include <mip6.h>
#include <mip6mh.h>

#define CMSGBUF	1024

struct mip6_mh_buf {
	struct mip6_mh_hdr hdr;
	union {
		struct mip6_mh_bu bu;
		struct mip6_mh_ba ba;
		struct mip6_mh_be be;
	} h;
	struct mip6_mh_opt opt[0];
}__attribute__ ((__packed__));

/*
 * returns:
 *   <0 end of option
 *    0 required to ignore option
 *   >0 found correctly formatted option
 */
int mip6mh_opt_next(const struct mip6_mh_hdr *mh,
		    int mh_data_len, unsigned char *mh_data_offset)
{
	struct mip6_mh_opt *opt = (struct mip6_mh_opt *)(mh->data + *mh_data_offset);
	int len;

#if 0
	assert(mh_data_len >= 0);
#endif
	if (*mh_data_offset >= mh_data_len)
		goto end;

	switch (opt->type) {
	case MIP6_OPT_PAD_1:
		len = sizeof(struct mip6_mh_opt_pad_1);
		goto ignore;
	case MIP6_OPT_PAD_N:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		goto ignore;
	case MIP6_OPT_BRA:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		if (opt->len != 2) /* compare length of spec */
			goto ignore;
		if (mh->type != MIP6_MH_BA)
			goto ignore;
		break;
	case MIP6_OPT_ALT_COA:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		if (opt->len != 16) /* compare length of spec */
			goto ignore;
		if (mh->type != MIP6_MH_BU)
			goto ignore;
		break;
	case MIP6_OPT_NONCE_INDEX:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		if (opt->len != 4) /* compare length of spec */
			goto ignore;
		if (mh->type != MIP6_MH_BU)
			goto ignore;
		break;
	case MIP6_OPT_BINDING_AUTH:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		if (mh->type != MIP6_MH_BU &&
		    mh->type != MIP6_MH_BA)
			goto ignore;
		break;
	default:
		len = sizeof(struct mip6_mh_opt) + ((opt->len + 1) << 3);
		goto ignore;
	}

	if (*mh_data_offset + len > mh_data_len)
		goto ignore;

	*mh_data_offset += len;
	return 1;

 ignore:
	if (*mh_data_offset + len > mh_data_len) {
		*mh_data_offset = mh_data_len;
	} else
		*mh_data_offset += len;
	return 0;
 end:
	return -1;
}

/* get Mobility Option
 * returns:
 *   0 is found, otherwise is not found.
 * If not-found, *mh_opt is set NULL.
 * If packet format is not follow spec, returns as not-found.
 */
int mip6mh_opt_get(const struct mip6_mh_hdr *mh, int len,
		   unsigned char mh_opt_type, struct mip6_mh_opt **mh_opt)
{
	unsigned char mh_data_offset = 0;
	int hdrlen;
	struct mip6_mh_opt **auth_optp;
	struct mip6_mh_opt *match_opt = NULL;
	struct mip6_mh_opt *last_opt = NULL;
	struct mip6_mh_opt *opt = NULL;

	hdrlen = sizeof(struct mip6_mh_hdr);

	switch (mh->type) {
	case MIP6_MH_BRR:
		hdrlen += sizeof(struct mip6_mh_brr);
		break;
	case MIP6_MH_HOTI:
	case MIP6_MH_COTI:
		hdrlen += sizeof(struct mip6_mh_test_init);
		break;
	case MIP6_MH_HOT:
	case MIP6_MH_COT:
		hdrlen += sizeof(struct mip6_mh_test);
		break;
	case MIP6_MH_BU:
		hdrlen += sizeof(struct mip6_mh_bu);
		break;
	case MIP6_MH_BA:
		hdrlen += sizeof(struct mip6_mh_ba);
		break;
	case MIP6_MH_BE:
		hdrlen += sizeof(struct mip6_mh_be);
		break;
	default:
		goto not_found;
	}

	/* observe binding auth data or not */
	switch (mh_opt_type) {
	case MIP6_OPT_NONCE_INDEX:
	case MIP6_OPT_BINDING_AUTH:
		break;
	default:
		auth_optp = NULL;
		break;
	}

	while (1) {
		int ret;

		ret = mip6mh_opt_next(mh, len - hdrlen, &mh_data_offset);
		if (ret == 0)
			continue;
		else if (ret < 0) {
			last_opt = opt;
			break;
		}

		opt = (struct mip6_mh_opt *)(mh->data + mh_data_offset);

		/* remember the first binding auth data. */
		if (auth_optp) {
			if (opt->type == MIP6_OPT_BINDING_AUTH) {
				if (!*auth_optp)
					*auth_optp = opt;
			}
		}

		/* XXX: if option type is duplicated, 
		 * it is selected the first one.
		 * this is not in spec...
		 */
		if (opt->type == mh_opt_type) {
			if (!match_opt)
				match_opt = opt;
		}
	}

	if (!match_opt)
		goto not_found;

	/* Binding Authorization Data option must be the last option,
	 * Nonce Indices option is valid only when present together
	 * with a Binding Authorization Data option
	 */
	if (auth_optp)
		if (last_opt != *auth_optp)
			goto not_found;

	*mh_opt = match_opt;
	return 0;

 not_found:
	*mh_opt = NULL;
	return -ENOENT;
}


static int mip6mh_output(int sock, void *buf, int len,
			 struct in6_addr *daddr, struct in6_addr *saddr, int ifindex)
{
	struct sockaddr_in6 daddr_sa;
	struct sockaddr_in6 saddr_sa;
	struct msghdr msghdr;
	struct cmsghdr *cmsghdr;
	struct iovec iov;
	unsigned char cmsgbuf[CMSGBUF];
	struct in6_pktinfo *pktinfo;
	

	int cksum_offset;
	int err_code;
	int ret;

	/* Mobility Header checksum */
	/* XXX: Fix me! */
	cksum_offset = 4;
	ret = setsockopt(sock, IPPROTO_IPV6, IPV6_CHECKSUM, &cksum_offset, sizeof(cksum_offset));
	if (ret != 0) {
		__perror("setsockopt");
		err_code = errno;
		goto fin;
	}

	/* Send */
	memset(&daddr_sa, 0, sizeof(daddr_sa));
	daddr_sa.sin6_family = PF_INET6;
	memcpy(&daddr_sa.sin6_addr, daddr, sizeof(daddr_sa.sin6_addr));

#if 0
	memset(&saddr_sa, 0, sizeof(struct sockaddr_in6));
        saddr_sa.sin6_family = AF_INET6;
	if (saddr)
		memcpy(&saddr_sa.sin6_addr, saddr, sizeof(saddr_sa.sin6_addr));

	/* Bind(re-bind) */
        ret = bind(sock, (struct sockaddr *)&saddr_sa, sizeof(struct sockaddr_in6));
	if (ret) {
		__perror("bind");
		err_code = errno;
		goto fin;
	}

	ret = sendto(sock, buf, len, 0, (struct sockaddr *)&daddr_sa, sizeof(daddr_sa));

#else
	/* Specify the source address */
	memset(&msghdr, 0, sizeof(msghdr));
	msghdr.msg_name = &daddr_sa;
	msghdr.msg_namelen = sizeof(daddr_sa);
	msghdr.msg_iov = &iov;
	msghdr.msg_iovlen = 1;
	cmsghdr = (struct cmsghdr *)cmsgbuf;
	msghdr.msg_control = cmsghdr;

	iov.iov_base = (void *)buf;
	iov.iov_len = len;

	memset(cmsgbuf, 0, CMSGBUF);
	cmsghdr->cmsg_len = CMSG_LEN(sizeof(struct in6_pktinfo));
	cmsghdr->cmsg_level = IPPROTO_IPV6;
	cmsghdr->cmsg_type = IPV6_PKTINFO;
	msghdr.msg_controllen = cmsghdr->cmsg_len;

	pktinfo = (struct in6_pktinfo *)CMSG_DATA(cmsghdr);
	if (saddr) {
		char abuf[100];
		printf("source addr = %s, index = %d\n", inet_ntop(PF_INET6, saddr, abuf, 100), ifindex);
		printf("dst addr = %s\n", inet_ntop(PF_INET6, daddr, abuf, 100));
		memcpy(&pktinfo->ipi6_addr, saddr, sizeof(struct in6_addr));
		pktinfo->ipi6_ifindex = ifindex;
	}

	ret = sendmsg(sock, &msghdr, 0);
#endif

	if (ret < 0) {
		__perror("sendto");
		err_code = errno;
		goto fin;
	}
	__eprintf("lengh (sent/buf) = %d/%d\n", ret, len);

	err_code = 0;
 fin:
	return err_code;
}

int mip6mh_ba_output(int sock,
		     struct in6_addr *daddr, struct in6_addr *saddr,
		     int ifindex, unsigned char status, unsigned char flags,
		     unsigned short seq, unsigned short lifetime,
		     void *baopt, int baopt_len)
{
	struct mip6_mh_buf mhbuf; /* buffer is here */
	struct mip6_mh_hdr *mh = &mhbuf.hdr;
	struct mip6_mh_ba *ba = &mhbuf.h.ba;
	int len;

	__dprintf("sending ba code=%d\n", status);

	mh->nexthdr = IPPROTO_NONE;
	mh->type = MIP6_MH_BA;

	memset(ba, 0, sizeof(*ba));
	ba->status = status;
  	ba->flags = flags;
	ba->seq = htons(seq);
	ba->lifetime = lifetime;
	if (baopt)
		memcpy(&ba->opt, baopt, baopt_len);

	len = sizeof(*mh) + sizeof(*ba) + baopt_len;

	mh->hdrlen = len;
	mh->hdrlen = (mh->hdrlen >> 3) - 1;

	return mip6mh_output(sock, mh, len, daddr, saddr, ifindex);
}

int mip6mh_be_output(int sock, struct in6_addr *daddr,
		     unsigned char status, struct in6_addr *be_hoa)
{
	struct mip6_mh_buf mhbuf; /* buffer is here */
	struct mip6_mh_hdr *mh = &mhbuf.hdr;
	struct mip6_mh_be *be = &mhbuf.h.be;
	int len;

	mh->nexthdr = IPPROTO_NONE;
	mh->type = MIP6_MH_BE;

	memset(be, 0, sizeof(*be));
	be->status = status;
	if (be_hoa)
		memcpy(&be->hoa, be_hoa, sizeof(be->hoa));

	len = sizeof(*mh) + sizeof(*be);

	mh->hdrlen = len;
	mh->hdrlen = (mh->hdrlen >> 3) - 1;

	return mip6mh_output(sock, mh, len, daddr, NULL, 0);
}
