/*
 * $Id: ttf2fft.cxx,v 1.1 2002/02/15 23:44:28 skavish Exp $
 *
 * ==========================================================================
 *
 * The JGenerator Software License, Version 1.0
 *
 * Copyright (c) 2000 Dmitry Skavish (skavish@usa.net). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *    "This product includes software developed by Dmitry Skavish
 *     (skavish@usa.net, http://www.flashgap.com/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The name "The JGenerator" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact skavish@usa.net.
 *
 * 5. Products derived from this software may not be called "The JGenerator"
 *    nor may "The JGenerator" appear in their names without prior written
 *    permission of Dmitry Skavish.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL DMITRY SKAVISH OR THE OTHER
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_CACHE_H

#include FT_CACHE_IMAGE_H
#include FT_CACHE_SMALL_BITMAPS_H
#include FT_CACHE_CHARMAP_H
#include <freetype/ftoutln.h>
#include <freetype/ftbbox.h>
#include <freetype/internal/tttypes.h>
#include <freetype/ttnameid.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "swflib.hxx"
#include "vector.hxx"

/* platform and encoding tables */
typedef struct Encoding_ {
    int   id;
    char* name;
} Encoding;

typedef struct Platform_ {
    int        id;
    char*      name;
    Encoding*  encodings;
} Platform;

Encoding AppleUnicodeEncodings[] = {
    { TT_APPLE_ID_DEFAULT,     "Unicode 1.0" },
    { TT_APPLE_ID_UNICODE_1_1, "Unicode 1.1" },
    { TT_APPLE_ID_ISO_10646,   "ISO 10646" },
    { TT_APPLE_ID_UNICODE_2_0, "Unicode 2.0" },
    { -1, NULL }
};

Encoding MacintoshEncodings[] = {
    { TT_MAC_ID_ROMAN                , "Roman" },
    { TT_MAC_ID_JAPANESE             , "Japanese" },
    { TT_MAC_ID_TRADITIONAL_CHINESE  , "Chinese (Traditional)" },
    { TT_MAC_ID_KOREAN               , "Korean" },
    { TT_MAC_ID_ARABIC               , "Arabic" },
    { TT_MAC_ID_HEBREW               , "Hebrew" },
    { TT_MAC_ID_GREEK                , "Greek" },
    { TT_MAC_ID_RUSSIAN              , "Russian" },
    { TT_MAC_ID_RSYMBOL              , "RSymbol" },
    { TT_MAC_ID_DEVANAGARI           , "Devanagari" },
    { TT_MAC_ID_GURMUKHI             , "Gurmukhi" },
    { TT_MAC_ID_GUJARATI             , "Gujarati" },
    { TT_MAC_ID_ORIYA                , "Oriya" },
    { TT_MAC_ID_BENGALI              , "Bengali" },
    { TT_MAC_ID_TAMIL                , "Tamil" },
    { TT_MAC_ID_TELUGU               , "Telugu" },
    { TT_MAC_ID_KANNADA              , "Kannada" },
    { TT_MAC_ID_MALAYALAM            , "Malayalam" },
    { TT_MAC_ID_SINHALESE            , "Sinhalese" },
    { TT_MAC_ID_BURMESE              , "Burmese" },
    { TT_MAC_ID_KHMER                , "Khmer" },
    { TT_MAC_ID_THAI                 , "Thai" },
    { TT_MAC_ID_LAOTIAN              , "Laotian" },
    { TT_MAC_ID_GEORGIAN             , "Georgian" },
    { TT_MAC_ID_ARMENIAN             , "Armenian" },
    { TT_MAC_ID_MALDIVIAN            , "Maldivian" },
    { TT_MAC_ID_SIMPLIFIED_CHINESE   , "Chinese (Simplified)" },
    { TT_MAC_ID_TIBETAN              , "Tibetan" },
    { TT_MAC_ID_MONGOLIAN            , "Mongolian" },
    { TT_MAC_ID_GEEZ                 , "Geez" },
    { TT_MAC_ID_SLAVIC               , "Slavic" },
    { TT_MAC_ID_VIETNAMESE           , "Vietnamese" },
    { TT_MAC_ID_SINDHI               , "Sindhi" },
    { TT_MAC_ID_UNINTERP             , "Uninterp" },
    { -1, NULL }
};

Encoding ISOEncodings[] = {
    { TT_ISO_ID_7BIT_ASCII,     "ASCII (7bit)" },
    { TT_ISO_ID_10646,          "ISO 10646" },
    { TT_ISO_ID_8859_1,         "ISO 8859-1" },
    { -1, NULL }
};

Encoding MicrosoftEncodings[] = {
    { TT_MS_ID_SYMBOL_CS,       "Symbol" },
    { TT_MS_ID_UNICODE_CS,      "Unicode" },
    { TT_MS_ID_SJIS,            "ShiftJIS" },
    { TT_MS_ID_GB2312,          "PRC (GB2312)" },
    { TT_MS_ID_BIG_5,           "Big5" },
    { TT_MS_ID_WANSUNG,         "Wansung" },
    { TT_MS_ID_JOHAB,           "Johab" },
    { TT_MS_ID_UCS_4,           "UCS-4" },
    { -1, NULL }
};

Platform platforms[] = {
    { TT_PLATFORM_APPLE_UNICODE, "Apple Unicode", AppleUnicodeEncodings },
    { TT_PLATFORM_MACINTOSH,     "Macintosh",     MacintoshEncodings },
    { TT_PLATFORM_ISO,           "ISO",           ISOEncodings },
    { TT_PLATFORM_MICROSOFT,     "Microsoft",     MicrosoftEncodings },
//    { TT_PLATFORM_CUSTOM,        "Custom",        CustomEncodings },
    { -1, NULL, NULL }
};

/*************************************************************************/
/*************************************************************************/
/*****                                                               *****/
/*****               FREETYPE-SPECIFIC DEFINITIONS                   *****/
/*****                                                               *****/
/*************************************************************************/
/*************************************************************************/

FT_Error    error;
FT_Library  library;       /* the FreeType library            */
FT_Face     face;          /* the font face                   */
FT_Encoding encoding = ft_encoding_none;

int         face_num = 0;
int         charmap_num = 0;
char        fft_file[1024+5];
char*       font_file = NULL;
char        filename[1024 + 5];

static const char*  file_suffixes[] = { ".ttf", ".ttc", ".pfa", ".pfb", 0 };

/**
 * Returns base name of specified pathname
 *
 * @param name   specified path
 * @return base name of specified path
 */
char* ft_basename ( const char*  name )
{
    const char*  base;
    const char*  current;
    char         c;

    base    = name;
    current = name;

    c = *current;

    while ( c ) {
#ifndef macintosh
        if ( c == '/' || c == '\\' )
#else
        if ( c == ':' )
#endif
            base = current + 1;

        current++;
        c = *current;
    }

    return(char*)base;
}

void error_msg( const char* fmt, ... )
{
    va_list  ap;
    va_start( ap, fmt );
    vfprintf( stderr, fmt, ap );
    va_end( ap );

    exit( 1 );
}

void fterror_msg( const char* fmt, ... )
{
    va_list  ap;
    va_start( ap, fmt );
    vfprintf( stderr, fmt, ap );
    va_end( ap );

    fprintf( stderr, "\nerror = 0x%04x\n", error );
    exit( 1 );
}

void warning_msg( const char* fmt, ... )
{
    va_list  ap;
    va_start( ap, fmt );
    vfprintf( stderr, fmt, ap );
    va_end( ap );
}

/**
 * Makes encoding from its name
 *
 * @param s      encoding name
 * @return encoding in its binary format
 */
static unsigned int make_encoding( char  *s )
{
    int            i;
    unsigned int  l = 0;

    for ( i = 0; i < 4; i++ ) {
        if ( !s[i] )
            break;
        l <<= 8;
        l  += (unsigned int)s[i];
    }

    return l;
}

/**
 * Loads font file and return number of faces or -1
 *
 * @param filepath font file path
 * @return number of faces or -1
 */
static int loadFontFile() {

    int len = strlen( font_file );
    if ( len > 1024 )
        len = 1024;

    strncpy( filename, font_file, len );
    filename[len] = 0;

    font_file = filename;

    error = FT_New_Face( library, filename, 0, &face );

    if( error ) {
        /* could not open the file directly; we will now try various */
        /* suffixes like `.ttf' or `.pfb'                            */

#ifndef macintosh

        const char** suffix = file_suffixes;
        int suffix_len = 0;
        int i = len - 1;

        while ( i > 0 && filename[i] != '\\' && filename[i] != '/' ) {
            if ( filename[i] == '.' ) {
                suffix_len = i;
                break;
            }
            i--;
        }
        if ( suffix_len == 0 ) {
            for ( suffix = file_suffixes; suffix[0]; suffix++ ) {
                /* try with current suffix */
                strcpy( filename + len, suffix[0] );

                error = FT_New_Face( library, filename, 0, &face );
                if ( !error )
                    goto Success;
            }
        }

#endif /* !macintosh */

        error_msg("Could not open font file %s\n", font_file);
    }

    Success:

    int num_faces = face->num_faces;

    return num_faces;
}

/**
 * Prints charmap info
 *
 * @param i charmap index in the current face
 */
static void printCMapInfo( int i )
{
    FT_CharMap cmap = face->charmaps[i];
    int platform_id = cmap->platform_id;
    int encoding_id = cmap->encoding_id;

    char * platform = NULL;
    char * encoding = NULL;
    for( Platform* plat = platforms; plat->id >= 0; plat++ ) {
        if( plat->id == platform_id ) {
            platform = plat->name;
            for( Encoding* enc = plat->encodings; enc->id >= 0; enc++ ) {
                if( enc->id == encoding_id ) {
                    encoding = enc->name;
                    break;
                }
            }
            break;
        }
    }

    if( platform == NULL ) {
        static char s[20];
        sprintf(s, "%d", platform_id);
        platform = s;
    }

    if( encoding == NULL ) {
        unsigned long enc = encoding_id;
        static char s[5];
        s[0] = (char) (enc>>24);
        s[1] = (char) (enc>>16);
        s[2] = (char) (enc>>8);
        s[3] = (char) (enc);
        s[4] = (char) 0;
        encoding = s;
    }

    printf("   %d: platform: %s, encoding: %s\n", i, platform, encoding);
}

/**
 * Prints face info
 *
 * @param face   face
 */
static void printFaceInfo( FT_Face face )
{
    printf( "   family: %s\n", face->family_name );
    printf( "   style:  %s\n", face->style_name );

    /* CharMaps */
    printf( "   charmaps:\n" );

    for( int i=0; i<face->num_charmaps; i++ ) {
        printCMapInfo(i);
    }
}

/**
 * Prints font info
 *
 * @param font_file name of font file to print
 */
static void printFontInfo()
{
    int num_faces = loadFontFile();
    if( num_faces < 0 ) return;

    printf( "There %s %d %s in this file.\n",
            num_faces == 1 ? (char *)"is" : (char *)"are",
            num_faces,
            num_faces == 1 ? (char *)"face" : (char *)"faces" );

    for( int i = 0; i < num_faces; i++ ) {
        if( i > 0 ) {
            error = FT_New_Face( library, font_file, i, &face );
            if ( error )
              fterror_msg("Could not open face %d.\n", i);
        }

        printf( "\n----- Face number: %d -----\n\n", i );
        printFaceInfo( face );
        printf( "\n" );
        FT_Done_Face( face );
    }
}

/**
 * Loads font file, face and charmap
 *
 * @param filepath font file name
 * @return 0 - ok, -1 - error
 */
static int install_font_file()
{
    int num_faces = loadFontFile();
    if( num_faces < 0 ) return -1;
    if( face_num >= num_faces ) {
        error_msg("Invalid face number %d.\n", face_num);
        return -1;
    }

    if( face_num != 0 ) {
        FT_Done_Face( face );
        error = FT_New_Face( library, font_file, face_num, &face );
        if( error )
            fterror_msg("Could not open face %d.\n", face_num);
    }

    if( charmap_num >= face->num_charmaps ) charmap_num = 0;
    error = FT_Set_Charmap(face, face->charmaps[charmap_num]);
    if ( error ) {
        FT_Done_Face( face );
        fterror_msg("Could not set charmap %d.\n", charmap_num);
    }

    return 0;
}

static void init_freetype( void )
{
    error = FT_Init_FreeType( &library );
    if( error )
        fterror_msg("Could not initialize FreeType.\n");
}

static void done_freetype( void )
{
    FT_Done_FreeType( library );
}

static void usage( char* execname ) {
    printf("\nTrueType to FFT (Macromedia Flash Font format) font converter. Version 1.0\n\n");
    printf("Copyright (c) 2002 by Dmitry Skavish. All right reserved.\n\n");
    printf("Usage:\n\n   %s {[-f <face number>] [-e <charmap number>] [-o <outputfile.fft>] |\n"\
                     "              [-l]} <fontfilename>\n", execname);
    printf("\nwhere:\n\n");
    printf("  -f  - specifies face number\n");
    printf("  -e  - specifies charmap number\n");
    printf("  -o  - specifies output file name\n");
    printf("  -l  - prints font info\n\n");
    exit(0);
}

#define HAS_LAYOUT    0x0080
#define SHIFT_JIS     0x0040
#define UNICODE       0x0020
#define ANSI          0x0010
#define WIDE_OFFSETS  0x0008
#define WIDE_CODES    0x0004
#define ITALIC        0x0002
#define BOLD          0x0001

static double ratio_EM;

int Outline_MoveTo_Func( FT_Vector * to, void * user ) {
    Shape* shape = (Shape*) user;
    //printf("  moveTo(%d, %d)\n", to->x, to->y);
    shape->movePenTo((int)(to->x*ratio_EM), -(int)(to->y*ratio_EM));
    return 0;
}

static int Outline_LineTo_Func( FT_Vector * to, void * user ) {
    Shape* shape = (Shape*) user;
    //printf("  lineTo(%d, %d)\n", to->x, to->y);
    shape->drawLineTo((int)(to->x*ratio_EM), -(int)(to->y*ratio_EM));
    return 0;
}

static int Outline_ConicTo_Func( FT_Vector* ctrl, FT_Vector * to, void * user ) {
    Shape* shape = (Shape*) user;
    //printf("  conicTo(%d, %d, %d, %d)\n",ctrl->x, ctrl->y, to->x, to->y);
    shape->drawCurveTo((int)(ctrl->x*ratio_EM), -(int)(ctrl->y*ratio_EM),
                       (int)(to->x*ratio_EM), -(int)(to->y*ratio_EM));
    return 0;
}

static int Outline_CubicTo_Func( FT_Vector* ctrl1, FT_Vector* ctrl2, FT_Vector * to, void * user ) {
    Shape* shape = (Shape*) user;
    //printf("  CUBICTo(%d, %d, %d, %d)\n",ctrl1->x, ctrl1->y, to->x, to->y);
    return 1;
}

static FT_Outline_Funcs outline_funcs = {
    &Outline_MoveTo_Func,
    &Outline_LineTo_Func,
    &Outline_ConicTo_Func,
    &Outline_CubicTo_Func,
    0,
    0
};


/**
 * Creates fft font and writes it to flashbuffer
 *
 * @return new flash buffer
 */
FlashBuffer* createFFTFont() {
    FlashBuffer* fob = new FlashBuffer(100000);

    fob->skip(6);

    fob->writeWord(1);  // font id

    //char* style_name = face->style_name;

    int flags = HAS_LAYOUT;
    if( face->style_flags&FT_STYLE_FLAG_BOLD ) flags |= BOLD;
    if( face->style_flags&FT_STYLE_FLAG_ITALIC ) flags |= ITALIC;

    int p_id = face->charmaps[charmap_num]->platform_id;
    if( p_id == TT_PLATFORM_APPLE_UNICODE ) {
        flags |= UNICODE;
    } else if( p_id == TT_PLATFORM_MICROSOFT ) {
        int e_id = face->charmaps[charmap_num]->encoding_id;
        switch( e_id ) {
            case TT_MS_ID_UNICODE_CS:
                flags |= UNICODE;
                break;
            case TT_MS_ID_SJIS:
                flags |= SHIFT_JIS;
                break;
            case TT_MS_ID_GB2312:
            case TT_MS_ID_BIG_5:
            case TT_MS_ID_WANSUNG:      // may be we need to put all of them under ShiftJIS ????
            case TT_MS_ID_JOHAB:
            case TT_MS_ID_UCS_4:
            case TT_MS_ID_SYMBOL_CS:
            default:
                flags |= ANSI;
                break;
        }
    } else {
        flags |= ANSI;
    }

    fob->writeWord( flags );

    char font_name[200];
    if( face->family_name == NULL ) {
        printf("Enter name of the font: ");
        scanf("%s", &font_name);
    } else {
        strcpy(font_name, face->family_name);
    }
    fob->writeStringL(font_name);

    // create output file name if it's not specified
    if( strlen(fft_file) == 0 ) {
        char * s = fft_file;
        if( flags&BOLD ) *s++ = 'B';
        if( flags&ITALIC ) *s++ = 'I';
        strcpy(s, font_name);
        while( *s ) {
            if( *s == ' ' ) *s = 'G';
            s++;
        }
        strcpy(s, ".fft");
    }

    ratio_EM = 1024.0/face->units_per_EM;

    printf("retrieving chars from current charmap ... ");
    // indexes
    Vector<int> codetable(1000);
    Vector<int> indexes(1000);
    for( int ch=0; ch<0xffff; ch++ ) {
        int gidx = FT_Get_Char_Index(face, ch);
        if( gidx == 0 ) continue;
        indexes.addElement(gidx);
        codetable.addElement(ch);
    }
    int nGlyph = codetable.size();
    printf("found %d chars\n", nGlyph);

    int lastcode = indexes.elementAt(nGlyph-1);
    if( lastcode>255 ) flags |= WIDE_CODES;
    printf("using %s-byte char codes\n", lastcode>255?"2":"1");

    fob->writeWord(nGlyph);

    printf("retrieving glyphs outlines ... ");
    // create vector glyphs
    Shape** glyphs = new Shape*[nGlyph];
    Rectangle2D* bboxes = new Rectangle2D[nGlyph];
    int* advancetable = new int[nGlyph];
    for( int i=0; i<nGlyph; i++ ) {
        Shape * shape = new Shape();
        int idx = indexes.elementAt(i);
        glyphs[i] = shape;

        // FT_LOAD_LINEAR_DESIGN
        error = FT_Load_Glyph(face, idx, FT_LOAD_NO_SCALE|FT_LOAD_IGNORE_TRANSFORM);
        if( error ) {
            warning_msg("\n   error 0x%04x loading glyph at index %d\n", error, idx);
            continue;
        }
        FT_GlyphSlot slot = face->glyph;

        advancetable[i] = slot->advance.x;
        /*int adv = slot->metrics.width;
        if( adv != advancetable[i] ) {
            //printf( "%d <-> %d\n", advancetable[i], adv );
        }
        advancetable[i] = adv;*/

        //printf("decomposing shape outline:\n" );
        error = FT_Outline_Decompose(&slot->outline, &outline_funcs, shape);
        if( error ) {
            warning_msg("\n   error 0x%04x decomposing outline at index %d\n", error, idx);
            continue;
        }

        // get bounding box
        //FT_BBox bbox;
        //error = FT_Outline_Get_BBox(&slot->outline, &bbox);
        //bboxes[i].setRect(bbox.xMin*ratio_EM, bbox.yMin*ratio_EM,
                          //(bbox.xMax-bbox.xMin)*ratio_EM, (bbox.yMax-bbox.yMin)*ratio_EM);
        bboxes[i].setRect(-1024,-1024,2048,2048);
        if( error ) {
            warning_msg("\n   error 0x%04x calculating bounding box at index %d\n", error, idx);
            continue;
        }

    }
    printf("done.\n");

    printf("generating glyph outlines ... ");
    // dumping all glyphs to special flashbuffer to determine offset size
    FlashBuffer* sh_fob = new FlashBuffer(100000);
    int* offsets = new int[nGlyph];
    for( int i=0; i<nGlyph; i++ ) {
        Shape* shape = glyphs[i];
        offsets[i] = sh_fob->getPos();
        shape->write(*sh_fob);
    }

    bool wide_offsets = sh_fob->getSize() > 0xffff;
    if( wide_offsets ) flags |= WIDE_OFFSETS;

    int inc = wide_offsets? 4: 2;
    int off = nGlyph*inc+inc;
    // write offset table
    for( int i=0; i<nGlyph; i++ ) {
        if( wide_offsets ) {
            fob->writeDWord(offsets[i]+off);
        } else {
            fob->writeWord(offsets[i]+off);
        }
    }

    // write offset to codetable
    if( wide_offsets ) {
        fob->writeDWord(sh_fob->getSize()+off);
    } else {
        fob->writeWord(sh_fob->getSize()+off);
    }

    // write glyph shapes
    fob->writeFOB(sh_fob);
    delete sh_fob;

    // write codetable
    for( int i=0; i<nGlyph; i++ ) {
        if( (flags&WIDE_CODES) != 0 ) {
            fob->writeWord( codetable.elementAt(i) );
        } else {
            fob->writeByte( codetable.elementAt(i) );
        }
    }
    printf("done.\n");
    printf("using %s glyph offsets\n", wide_offsets?"wide":"short");

    printf("generating layout\n");
    // write ascender, descender and leading
    int ascender  = (int) (face->ascender*ratio_EM);
    int descender = (int) -(face->descender*ratio_EM);
    int leading   = (int) ((face->height-face->ascender+face->descender)*ratio_EM);
    fob->writeWord( ascender );
    fob->writeWord( descender );
    fob->writeWord( leading );

    printf("   writing advance table\n");
    // write advance table
    for( int i=0; i<nGlyph; i++ ) fob->writeWord( (int) (advancetable[i]*ratio_EM) );

    printf("   writing bbox table\n");
    // write bounding boxes
    for( int i=0; i<nGlyph; i++ ) {
        fob->write( bboxes[i] );
    }

    // write kerning tables
    // probably we need to restrict writing only to codes from codetable

    if( FT_HAS_KERNING(face) && FT_IS_SFNT(face) ) {
        printf("   writing kerning table\n");
        TT_Face ttface = (TT_Face) face;
        int nKern = ttface->num_kern_pairs;

        int* kern_table = new int[nKern*3];

        int sz = 0;
        for( int i=0; i<nKern; i++ ) {
            TT_Kern_0_Pair* pair = ttface->kern_pairs + i;
            int left_idx = pair->left;
            int right_idx = pair->right;
            int value = (int) (pair->value*ratio_EM);
            if( value == 0 ) continue;
            //printf( "left: %d, right: %d, value: %d\n", left_idx, right_idx, value);

            int left_code = -1;
            for( int k=0; k<indexes.size(); k++ ) {
                if( indexes.elementAt(k) == left_idx ) {
                    left_code = codetable.elementAt(k);
                    break;
                }
            }
            if( left_code < 0 ) continue;

            int right_code = -1;
            for( int k=0; k<indexes.size(); k++ ) {
                if( indexes.elementAt(k) == right_idx ) {
                    right_code = codetable.elementAt(k);
                    break;
                }
            }
            if( right_code < 0 ) continue;

            kern_table[sz++] = left_code;
            kern_table[sz++] = right_code;
            kern_table[sz++] = value;
        }

        nKern = sz/3;
        fob->writeWord( nKern );

        for( int j=0; j<nKern; j++ ) {
            if( (flags&WIDE_CODES) != 0 ) {
                fob->writeWord( kern_table[j*3] );
                fob->writeWord( kern_table[j*3+1] );
                fob->writeWord( kern_table[j*3+2] );
            } else {
                fob->writeByte( kern_table[j*3] );
                fob->writeByte( kern_table[j*3+1] );
                fob->writeWord( kern_table[j*3+2] );
            }
        }
    } else {
        printf("   no kerning table found!\n");
        fob->writeWord( 0 );
    }

    fob->writeWordAt( flags, 8 );
    fob->writeLongTagAt(48, fob->getPos()-6, 0);

    return fob;
}

int main( int argc, char*  argv[] ) {
    bool    print_info = false;

    char*   execname = ft_basename( argv[0] );

    if( argc <= 1 ) {
        usage(execname);
    }

    fft_file[0] = 0;
    for( int i=1; i<argc; i++ ) {
        char * s = argv[i];
        if( strcmp(s,"-e" )==0 && i!=argc-1 ) {
            charmap_num = atoi(argv[++i]);
            //encoding = (FT_Encoding) make_tag(argv[++i]);
        } else if( strcmp(s,"-o" )==0 && i!=argc-1 ) {
            strncpy(fft_file, argv[++i], 1024);
            fft_file[1024] = 0;
        } else if( strcmp(s,"-h" )==0 ) {
            usage(execname);
        } else if( strcmp(s,"-l" )==0 ) {
            print_info = true;
        } else if( strcmp(s,"-f" )==0 && i!=argc-1 ) {
            face_num = atoi(argv[++i]);
        } else {
            if( font_file != NULL ) {
                usage(execname);
            }
            font_file = s;
        }
    }

    if( (print_info && (charmap_num != 0 || fft_file[0] != 0)) || font_file == NULL ) {
        //printf( "enc=%d, fft_file=%s, font_file=%s, list=%d\n", encoding, fft_file, font_file, print_info);
        usage(execname);
    }

    /* Initialize engine */
    init_freetype();

    if( print_info ) {
        printFontInfo();
        exit(0);
    }

    install_font_file();

    printf("Converting font %s\n", font_file);
    printf("    family: %s\n", face->family_name);
    printf("    style:  %s\n", face->style_name);
    printf("using charmap:");
    printCMapInfo(charmap_num);
    printf("\n");

    FlashBuffer* font = createFFTFont();

    writeFlashFile(fft_file, font);

    delete font;

    done_freetype();
    exit( 0 );      /* for safety reasons */
    return 0;       /* never reached */
}

