/*
 * $Id: geom.hxx,v 1.1 2002/02/15 23:44:28 skavish Exp $
 *
 * ==========================================================================
 *
 * The JGenerator Software License, Version 1.0
 *
 * Copyright (c) 2000 Dmitry Skavish (skavish@usa.net). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *    "This product includes software developed by Dmitry Skavish
 *     (skavish@usa.net, http://www.flashgap.com/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The name "The JGenerator" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact skavish@usa.net.
 *
 * 5. Products derived from this software may not be called "The JGenerator"
 *    nor may "The JGenerator" appear in their names without prior written
 *    permission of Dmitry Skavish.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL DMITRY SKAVISH OR THE OTHER
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef AFFINETRANSFORM_H
#define AFFINETRANSFORM_H

#include <string.h>
#include <math.h>

#define TYPE_IDENTITY 0
#define TYPE_TRANSLATION 1
#define TYPE_UNIFORM_SCALE 2
#define TYPE_GENERAL_SCALE 4
#define TYPE_MASK_SCALE (TYPE_UNIFORM_SCALE | TYPE_GENERAL_SCALE)
#define TYPE_FLIP 64
#define TYPE_QUADRANT_ROTATION 8
#define TYPE_GENERAL_ROTATION 16
#define TYPE_MASK_ROTATION (TYPE_QUADRANT_ROTATION | TYPE_GENERAL_ROTATION)
#define TYPE_GENERAL_TRANSFORM 32

#define APPLY_IDENTITY 0
#define APPLY_TRANSLATE 1
#define APPLY_SCALE 2
#define APPLY_SHEAR 4
#define HI_SHIFT 3
#define HI_IDENTITY (APPLY_IDENTITY << HI_SHIFT)
#define HI_TRANSLATE (APPLY_TRANSLATE << HI_SHIFT)
#define HI_SCALE (APPLY_SCALE << HI_SHIFT)
#define HI_SHEAR (APPLY_SHEAR << HI_SHIFT)

inline double min( double x, double y ) {
    if( x <= y ) return x;
    return y;
}

inline double max( double x, double y ) {
    if( y <= x ) return x;
    return y;
}

/**
 * The <code>AffineTransform</code> class represents a 2D affine transform
 * that performs a linear mapping from 2D coordinates to other 2D
 * coordinates that preserves the "straightness" and
 * "parallelness" of lines.  Affine transformations can be constructed
 * using sequences of translations, scales, flips, rotations, and shears.
 * <p>
 * Such a coordinate transformation can be represented by a 3 row by
 * 3 column matrix with an implied last row of [ 0 0 1 ].  This matrix
 * transforms source coordinates <code>(x,&nbsp;y)</code> into
 * destination coordinates <code>(x',&nbsp;y')</code> by considering
 * them to be a column vector and multiplying the coordinate vector
 * by the matrix according to the following process:
 * <pre>
 *  [ x']   [  m00  m01  m02  ] [ x ]   [ m00x + m01y + m02 ]
 *  [ y'] = [  m10  m11  m12  ] [ y ] = [ m10x + m11y + m12 ]
 *  [ 1 ]   [   0    0    1   ] [ 1 ]   [         1         ]
 * </pre>
 *
 * @version 1.62, 02/02/00
 * @author Jim Graham
 */
class AffineTransform {

private:

    /**
     * The X coordinate scaling element of the 3x3
     * affine transformation matrix.
     *
     * @serial
     */
    double m00;

    /**
     * The Y coordinate shearing element of the 3x3
     * affine transformation matrix.
     *
     * @serial
     */
    double m10;

    /**
     * The X coordinate shearing element of the 3x3
     * affine transformation matrix.
     *
     * @serial
     */
    double m01;

    /**
     * The Y coordinate scaling element of the 3x3
     * affine transformation matrix.
     *
     * @serial
     */
    double m11;

    /**
     * The X coordinate of the translation element of the
     * 3x3 affine transformation matrix.
     *
     * @serial
     */
    double m02;

    /**
     * The Y coordinate of the translation element of the
     * 3x3 affine transformation matrix.
     *
     * @serial
     */
    double m12;

    /**
     * This field keeps track of which components of the matrix need to
     * be applied when performing a transformation.
     * @see #APPLY_IDENTITY
     * @see #APPLY_TRANSLATE
     * @see #APPLY_SCALE
     * @see #APPLY_SHEAR
     */
    int state;

    AffineTransform(double m00, double m10,
                    double m01, double m11,
                    double m02, double m12,
                    int state) {
        this->m00 = m00;
        this->m10 = m10;
        this->m01 = m01;
        this->m11 = m11;
        this->m02 = m02;
        this->m12 = m12;
        this->state = state;
    }

public:
    /**
     * Constructs a new <code>AffineTransform</code> representing the
     * Identity transformation.
     */
    AffineTransform() {
        m00 = m11 = 1.0;
        m01 = m10 = m02 = m12 = 0.0;     /* Not needed. */
        state = APPLY_IDENTITY;      /* Not needed. */
    }

    /**
     * Constructs a new <code>AffineTransform</code> that is a copy of
     * the specified <code>AffineTransform</code> object.
     * @param Tx the <code>AffineTransform</code> object to copy
     */
    AffineTransform(AffineTransform& Tx) {
        this->m00 = Tx.m00;
        this->m10 = Tx.m10;
        this->m01 = Tx.m01;
        this->m11 = Tx.m11;
        this->m02 = Tx.m02;
        this->m12 = Tx.m12;
        this->state = Tx.state;
    }

    /**
     * Constructs a new <code>AffineTransform</code> from 6 double
     * precision values representing the 6 specifiable entries of the 3x3
     * transformation matrix.
     * @param m00,&nbsp;m01,&nbsp;m02,&nbsp;m10,&nbsp;m11,&nbsp;m12 the
     * 6 floating point values that compose the 3x3 transformation matrix
     */
    AffineTransform(double m00, double m10,
                    double m01, double m11,
                    double m02, double m12) {
        this->m00 = m00;
        this->m10 = m10;
        this->m01 = m01;
        this->m11 = m11;
        this->m02 = m02;
        this->m12 = m12;
        updateState();
    }

    /**
     * Constructs a new <code>AffineTransform</code> from an array of
     * double precision values representing either the 4 non-translation
     * entries or the 6 specifiable entries of the 3x3 transformation
     * matrix. The values are retrieved from the array as
     * {&nbsp;m00&nbsp;m10&nbsp;m01&nbsp;m11&nbsp;[m02&nbsp;m12]}.
     * @param flatmatrix the double array containing the values to be set
     * in the new <code>AffineTransform</code> object. The length of the
     * array is assumed to be at least 4. If the length of the array is
     * less than 6, only the first 4 values are taken. If the length of
     * the array is greater than 6, the first 6 values are taken.
     */
    AffineTransform(double flatmatrix[], int size) {
        m00 = flatmatrix[0];
        m10 = flatmatrix[1];
        m01 = flatmatrix[2];
        m11 = flatmatrix[3];
        if (size > 5) {
            m02 = flatmatrix[4];
            m12 = flatmatrix[5];
        }
        updateState();
    }

    /**
     * Returns a transform representing a translation transformation.
     * The matrix representing the returned transform is:
     * <pre>
     *      [   1    0    tx  ]
     *      [   0    1    ty  ]
     *      [   0    0    1   ]
     * </pre>
     * @param tx the distance by which coordinates are translated in the
     * X axis direction
     * @param ty the distance by which coordinates are translated in the
     * Y axis direction
     * @return an <code>AffineTransform</code> object that represents a
     *  translation transformation, created with the specified vector.
     */
    static AffineTransform* getTranslateInstance(double tx, double ty) {
        AffineTransform* Tx = new AffineTransform();
        Tx->setToTranslation(tx, ty);
        return Tx;
    }

    /**
     * Returns a transform representing a rotation transformation.
     * The matrix representing the returned transform is:
     * <pre>
     *      [   cos(theta)    -sin(theta)    0   ]
     *      [   sin(theta)     cos(theta)    0   ]
     *      [       0              0         1   ]
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     * @return an <code>AffineTransform</code> object that is a rotation
     *  transformation, created with the specified angle of rotation.
     */
    static AffineTransform* getRotateInstance(double theta) {
        AffineTransform* Tx = new AffineTransform();
        Tx->setToRotation(theta);
        return Tx;
    }

    /**
     * Returns a transform that rotates coordinates around an anchor point.
     * This operation is equivalent to translating the coordinates so
     * that the anchor point is at the origin (S1), then rotating them
     * about the new origin (S2), and finally translating so that the
     * intermediate origin is restored to the coordinates of the original
     * anchor point (S3).
     * <p>
     * This operation is equivalent to the following sequence of calls:
     * <pre>
     *      AffineTransform Tx = new AffineTransform();
     *      Tx.setToTranslation(x, y);  // S3: final translation
     *      Tx.rotate(theta);       // S2: rotate around anchor
     *      Tx.translate(-x, -y);   // S1: translate anchor to origin
     * </pre>
     * The matrix representing the returned transform is:
     * <pre>
     *      [   cos(theta)    -sin(theta)    x-x*cos+y*sin  ]
     *      [   sin(theta)     cos(theta)    y-x*sin-y*cos  ]
     *      [       0              0               1        ]
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     * @param x,&nbsp;y the coordinates of the anchor point of the
     * rotation
     * @return an <code>AffineTransform</code> object that rotates
     *  coordinates around the specified point by the specified angle of
     *  rotation.
     */
    static AffineTransform* getRotateInstance(double theta,
                                             double x, double y) {
        AffineTransform* Tx = new AffineTransform();
        Tx->setToRotation(theta, x, y);
        return Tx;
    }

    /**
     * Returns a transform representing a scaling transformation.
     * The matrix representing the returned transform is:
     * <pre>
     *      [   sx   0    0   ]
     *      [   0    sy   0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param sx the factor by which coordinates are scaled along the
     * X axis direction
     * @param sy the factor by which coordinates are scaled along the
     * Y axis direction
     * @return an <code>AffineTransform</code> object that scales
     *  coordinates by the specified factors.
     */
    static AffineTransform* getScaleInstance(double sx, double sy) {
        AffineTransform* Tx = new AffineTransform();
        Tx->setToScale(sx, sy);
        return Tx;
    }

    /**
     * Returns a transform representing a shearing transformation.
     * The matrix representing the returned transform is:
     * <pre>
     *      [   1   shx   0   ]
     *      [  shy   1    0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param shx the multiplier by which coordinates are shifted in the
     * direction of the positive X axis as a factor of their Y coordinate
     * @param shy the multiplier by which coordinates are shifted in the
     * direction of the positive Y axis as a factor of their X coordinate
     * @return an <code>AffineTransform</code> object that shears
     *  coordinates by the specified multipliers.
     */
    static AffineTransform* getShearInstance(double shx, double shy) {
        AffineTransform* Tx = new AffineTransform();
        Tx->setToShear(shx, shy);
        return Tx;
    }

    /**
     * Retrieves the flag bits describing the conversion properties of
     * this transform.
     * The return value is either one of the constants TYPE_IDENTITY
     * or TYPE_GENERAL_TRANSFORM, or a combination of the
     * appriopriate flag bits.
     * A valid combination of flag bits is an exclusive OR operation
     * that can combine
     * the TYPE_TRANSLATION flag bit
     * in addition to either of the
     * TYPE_UNIFORM_SCALE or TYPE_GENERAL_SCALE flag bits
     * as well as either of the
     * TYPE_QUADRANT_ROTATION or TYPE_GENERAL_ROTATION flag bits.
     * @return the OR combination of any of the indicated flags that
     * apply to this transform
     * @see #TYPE_IDENTITY
     * @see #TYPE_TRANSLATION
     * @see #TYPE_UNIFORM_SCALE
     * @see #TYPE_GENERAL_SCALE
     * @see #TYPE_QUADRANT_ROTATION
     * @see #TYPE_GENERAL_ROTATION
     * @see #TYPE_GENERAL_TRANSFORM
     */
    int getType();

    /**
     * Returns the determinant of the matrix representation of the transform.
     * The determinant is useful both to determine if the transform can
     * be inverted and to get a single value representing the
     * combined X and Y scaling of the transform.
     * <p>
     * If the determinant is non-zero, then this transform is
     * invertible and the various methods that depend on the inverse
     * transform do not need to throw a
     * {@link NoninvertibleTransformException}.
     * If the determinant is zero then this transform can not be
     * inverted since the transform maps all input coordinates onto
     * a line or a point.
     * If the determinant is near enough to zero then inverse transform
     * operations might not carry enough precision to produce meaningful
     * results.
     * <p>
     * If this transform represents a uniform scale, as indicated by
     * the <code>getType</code> method then the determinant also
     * represents the square of the uniform scale factor by which all of
     * the points are expanded from or contracted towards the origin.
     * If this transform represents a non-uniform scale or more general
     * transform then the determinant is not likely to represent a
     * value useful for any purpose other than determining if inverse
     * transforms are possible.
     * <p>
     * Mathematically, the determinant is calculated using the formula:
     * <pre>
     *      |  m00  m01  m02  |
     *      |  m10  m11  m12  |  =  m00 * m11 - m01 * m10
     *      |   0    0    1   |
     * </pre>
     *
     * @return the determinant of the matrix used to transform the
     * coordinates.
     * @see #getType
     * @see #createInverse
     * @see #inverseTransform
     * @see #TYPE_UNIFORM_SCALE
     */
    double getDeterminant();

    /**
     * Manually recalculates the state of the transform when the matrix
     * changes too much to predict the effects on the state.
     */
    void updateState();

    /*
     * Convenience method used internally to throw exceptions when
     * a case was forgotten in a switch statement.
     */
    void stateError() {
        //throw new InternalError("missing case in transform state switch");
    }

    /**
     * Retrieves the 6 specifiable values in the 3x3 affine transformation
     * matrix and places them into an array of double precisions values.
     * The values are stored in the array as
     * {&nbsp;m00&nbsp;m10&nbsp;m01&nbsp;m11&nbsp;m02&nbsp;m12&nbsp;}.
     * An array of 4 doubles can also be specified, in which case only the
     * first four elements representing the non-transform
     * parts of the array are retrieved and the values are stored into
     * the array as {&nbsp;m00&nbsp;m10&nbsp;m01&nbsp;m11&nbsp;}
     * @param flatmatrix the double array used to store the returned
     * values.
     * @see #getScaleX
     * @see #getScaleY
     * @see #getShearX
     * @see #getShearY
     * @see #getTranslateX
     * @see #getTranslateY
     */
    void getMatrix(double flatmatrix[], int size) {
        flatmatrix[0] = m00;
        flatmatrix[1] = m10;
        flatmatrix[2] = m01;
        flatmatrix[3] = m11;
        if (size > 5) {
            flatmatrix[4] = m02;
            flatmatrix[5] = m12;
        }
    }

    /**
     * Returns the X coordinate scaling element (m00) of the 3x3
     * affine transformation matrix.
     * @return a double value that is the X coordinate of the scaling
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getScaleX() {
        return m00;
    }

    /**
     * Returns the Y coordinate scaling element (m11) of the 3x3
     * affine transformation matrix.
     * @return a double value that is the Y coordinate of the scaling
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getScaleY() {
        return m11;
    }

    /**
     * Returns the X coordinate shearing element (m01) of the 3x3
     * affine transformation matrix.
     * @return a double value that is the X coordinate of the shearing
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getShearX() {
        return m01;
    }

    /**
     * Returns the Y coordinate shearing element (m10) of the 3x3
     * affine transformation matrix.
     * @return a double value that is the Y coordinate of the shearing
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getShearY() {
        return m10;
    }

    /**
     * Returns the X coordinate of the translation element (m02) of the
     * 3x3 affine transformation matrix.
     * @return a double value that is the X coordinate of the translation
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getTranslateX() {
        return m02;
    }

    /**
     * Returns the Y coordinate of the translation element (m12) of the
     * 3x3 affine transformation matrix.
     * @return a double value that is the Y coordinate of the translation
     *  element of the affine transformation matrix.
     * @see #getMatrix
     */
    double getTranslateY() {
        return m12;
    }

    /**
     * Concatenates this transform with a translation transformation.
     * This is equivalent to calling concatenate(T), where T is an
     * <code>AffineTransform</code> represented by the following matrix:
     * <pre>
     *      [   1    0    tx  ]
     *      [   0    1    ty  ]
     *      [   0    0    1   ]
     * </pre>
     * @param tx the distance by which coordinates are translated in the
     * X axis direction
     * @param ty the distance by which coordinates are translated in the
     * Y axis direction
     */
    void translate(double tx, double ty);

    /**
     * Concatenates this transform with a rotation transformation.
     * This is equivalent to calling concatenate(R), where R is an
     * <code>AffineTransform</code> represented by the following matrix:
     * <pre>
     *      [   cos(theta)    -sin(theta)    0   ]
     *      [   sin(theta)     cos(theta)    0   ]
     *      [       0              0         1   ]
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     */
    void rotate(double theta);

    static int rot90conversion[];

    /**
     * Concatenates this transform with a transform that rotates
     * coordinates around an anchor point.
     * This operation is equivalent to translating the coordinates so
     * that the anchor point is at the origin (S1), then rotating them
     * about the new origin (S2), and finally translating so that the
     * intermediate origin is restored to the coordinates of the original
     * anchor point (S3).
     * <p>
     * This operation is equivalent to the following sequence of calls:
     * <pre>
     *      translate(x, y);    // S3: final translation
     *      rotate(theta);      // S2: rotate around anchor
     *      translate(-x, -y);  // S1: translate anchor to origin
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     * @param x,&nbsp;y the coordinates of the anchor point of the
     * rotation
     */
    void rotate(double theta, double x, double y);

    /**
     * Concatenates this transform with a scaling transformation.
     * This is equivalent to calling concatenate(S), where S is an
     * <code>AffineTransform</code> represented by the following matrix:
     * <pre>
     *      [   sx   0    0   ]
     *      [   0    sy   0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param sx the factor by which coordinates are scaled along the
     * X axis direction
     * @param sy the factor by which coordinates are scaled along the
     * Y axis direction
    */
    void scale(double sx, double sy);

    /**
     * Concatenates this transform with a shearing transformation.
     * This is equivalent to calling concatenate(SH), where SH is an
     * <code>AffineTransform</code> represented by the following matrix:
     * <pre>
     *      [   1   shx   0   ]
     *      [  shy   1    0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param shx the multiplier by which coordinates are shifted in the
     * direction of the positive X axis as a factor of their Y coordinate
     * @param shy the multiplier by which coordinates are shifted in the
     * direction of the positive Y axis as a factor of their X coordinate
     */
    void shear(double shx, double shy);

    /**
     * Resets this transform to the Identity transform.
     */
    void setToIdentity();

    /**
     * Sets this transform to a translation transformation.
     * The matrix representing this transform becomes:
     * <pre>
     *      [   1    0    tx  ]
     *      [   0    1    ty  ]
     *      [   0    0    1   ]
     * </pre>
     * @param tx the distance by which coordinates are translated in the
     * X axis direction
     * @param ty the distance by which coordinates are translated in the
     * Y axis direction
     */
    void setToTranslation(double tx, double ty);

    /**
     * Sets this transform to a rotation transformation.
     * The matrix representing this transform becomes:
     * <pre>
     *      [   cos(theta)    -sin(theta)    0   ]
     *      [   sin(theta)     cos(theta)    0   ]
     *      [       0              0         1   ]
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     */
    void setToRotation(double theta);

    /**
     * Sets this transform to a translated rotation transformation.
     * This operation is equivalent to translating the coordinates so
     * that the anchor point is at the origin (S1), then rotating them
     * about the new origin (S2), and finally translating so that the
     * intermediate origin is restored to the coordinates of the original
     * anchor point (S3).
     * <p>
     * This operation is equivalent to the following sequence of calls:
     * <pre>
     *      setToTranslation(x, y); // S3: final translation
     *      rotate(theta);      // S2: rotate around anchor
     *      translate(-x, -y);      // S1: translate anchor to origin
     * </pre>
     * The matrix representing this transform becomes:
     * <pre>
     *      [   cos(theta)    -sin(theta)    x-x*cos+y*sin  ]
     *      [   sin(theta)     cos(theta)    y-x*sin-y*cos  ]
     *      [       0              0               1        ]
     * </pre>
     * Rotating with a positive angle theta rotates points on the positive
     * x axis toward the positive y axis.
     * @param theta the angle of rotation in radians
     * @param x,&nbsp;y the coordinates of the anchor point of the
     * rotation
     */
    void setToRotation(double theta, double x, double y);

    /**
     * Sets this transform to a scaling transformation.
     * The matrix representing this transform becomes:
     * <pre>
     *      [   sx   0    0   ]
     *      [   0    sy   0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param sx the factor by which coordinates are scaled along the
     * X axis direction
     * @param sy the factor by which coordinates are scaled along the
     * Y axis direction
     */
    void setToScale(double sx, double sy);

    /**
     * Sets this transform to a shearing transformation.
     * The matrix representing this transform becomes:
     * <pre>
     *      [   1   shx   0   ]
     *      [  shy   1    0   ]
     *      [   0    0    1   ]
     * </pre>
     * @param shx the multiplier by which coordinates are shifted in the
     * direction of the positive X axis as a factor of their Y coordinate
     * @param shy the multiplier by which coordinates are shifted in the
     * direction of the positive Y axis as a factor of their X coordinate
     */
    void setToShear(double shx, double shy);

    /**
     * Sets this transform to a copy of the transform in the specified
     * <code>AffineTransform</code> object.
     * @param Tx the <code>AffineTransform</code> object from which to
     * copy the transform
     */
    void setTransform(AffineTransform& Tx);

    /**
     * Sets this transform to the matrix specified by the 6
     * double precision values.
     * @param m00,&nbsp;m01,&nbsp;m02,&nbsp;m10,&nbsp;m11,&nbsp;m12 the
     * 6 floating point values that compose the 3x3 transformation matrix
     */
    void setTransform(double m00, double m10,
                      double m01, double m11,
                      double m02, double m12);

    /**
     * Concatenates an <code>AffineTransform</code> <code>Tx</code> to
     * this <code>AffineTransform</code> Cx in the most commonly useful
     * way to provide a new user space
     * that is mapped to the former user space by <code>Tx</code>.
     * Cx is updated to perform the combined transformation.
     * Transforming a point p by the updated transform Cx' is
     * equivalent to first transforming p by <code>Tx</code> and then
     * transforming the result by the original transform Cx like this:
     * Cx'(p) = Cx(Tx(p))
     * In matrix notation, if this transform Cx is
     * represented by the matrix [this] and <code>Tx</code> is represented
     * by the matrix [Tx] then this method does the following:
     * <pre>
     *      [this] = [this] x [Tx]
     * </pre>
     * @param Tx the <code>AffineTransform</code> object to be
     * concatenated with this <code>AffineTransform</code> object.
     * @see #preConcatenate
     */
    void concatenate(AffineTransform& Tx);

    /**
     * Concatenates an <code>AffineTransform</code> <code>Tx</code> to
     * this <code>AffineTransform</code> Cx
     * in a less commonly used way such that <code>Tx</code> modifies the
     * coordinate transformation relative to the absolute pixel
     * space rather than relative to the existing user space.
     * Cx is updated to perform the combined transformation.
     * Transforming a point p by the updated transform Cx' is
     * equivalent to first transforming p by the original transform
     * Cx and then transforming the result by
     * <code>Tx</code> like this:
     * Cx'(p) = Tx(Cx(p))
     * In matrix notation, if this transform Cx
     * is represented by the matrix [this] and <code>Tx</code> is
     * represented by the matrix [Tx] then this method does the
     * following:
     * <pre>
     *      [this] = [Tx] x [this]
     * </pre>
     * @param Tx the <code>AffineTransform</code> object to be
     * concatenated with this <code>AffineTransform</code> object.
     * @see #concatenate
     */
    void preConcatenate(AffineTransform& Tx);

    /**
     * Transforms an array of double precision coordinates by this transform.
     * The two coordinate array sections can be exactly the same or
     * can be overlapping sections of the same array without affecting the
     * validity of the results.
     * This method ensures that no source coordinates are
     * overwritten by a previous operation before they can be transformed.
     * The coordinates are stored in the arrays starting at the indicated
     * offset in the order <code>[x0, y0, x1, y1, ..., xn, yn]</code>.
     * @param srcPts the array containing the source point coordinates.
     * Each point is stored as a pair of x,&nbsp;y coordinates.
     * @param dstPts the array into which the transformed point
     * coordinates are returned.  Each point is stored as a pair of
     * x,&nbsp;y coordinates.
     * @param srcOff the offset to the first point to be transformed
     * in the source array
     * @param dstOff the offset to the location of the first
     * transformed point that is stored in the destination array
     * @param numPts the number of point objects to be transformed
     */
    void transform(double srcPts[], int srcOff,
                   double dstPts[], int dstOff,
                   int numPts);

    /**
     * Returns <code>true</code> if this <code>AffineTransform</code> is
     * an identity transform.
     * @return <code>true</code> if this <code>AffineTransform</code> is
     * an identity transform; <code>false</code> otherwise.
     */
    bool isIdentity() {
        return(state == APPLY_IDENTITY);
    }

};

/**
 * The <code>Rectangle2D</code> class describes a rectangle
 * defined by a location (x,&nbsp;y) and dimension
 * (w&nbsp;x&nbsp;h).
 * <p>
 * This class is only the abstract superclass for all objects that
 * store a 2D rectangle.
 * The actual storage representation of the coordinates is left to
 * the subclass.
 *
 * @version     1.22, 02/02/00
 * @author  Jim Graham
 */
class Rectangle2D {

public:

    /**
     * The x coordinate of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double x;

    /**
     * The y coordinate of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double y;

    /**
     * The width of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double width;

    /**
     * The height of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double height;

    /**
     * Constructs a new <code>Rectangle2D</code>, initialized to
         * location (0,&nbsp;0) and size (0,&nbsp;0).
     * @since 1.2
     */
    Rectangle2D() {
        x = y = width = height = 0.0;
    }

    /**
     * Constructs and initializes a <code>Rectangle2D</code>
         * from the specified double coordinates.
     * @param x,&nbsp;y the coordinates of the upper left corner
         * of the newly constructed <code>Rectangle2D</code>
     * @param width the width of the
         * newly constructed <code>Rectangle2D</code>
     * @param height the height of the
         * newly constructed <code>Rectangle2D</code>
     * @since 1.2
     */
    Rectangle2D(double x, double y, double w, double h) {
        setRect(x, y, w, h);
    }

    /**
     * Returns the X coordinate of this <code>Rectangle2D</code> in
         * double precision.
         * @return the X coordinate of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double getX() {
        return x;
    }

    /**
     * Returns the Y coordinate of this <code>Rectangle2D</code> in
         * double precision.
         * @return the Y coordinate of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double getY() {
        return y;
    }

    /**
     * Returns the width of this <code>Rectangle2D</code> in
         * double precision.
         * @return the width of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double getWidth() {
        return width;
    }

    /**
     * Returns the height of this <code>Rectangle2D</code> in
         * double precision.
         * @return the height of this <code>Rectangle2D</code>.
     * @since 1.2
     */
    double getHeight() {
        return height;
    }

    double getMinX() {
        return x;
    }

    double getMinY() {
        return y;
    }

    double getMaxX() {
        return x+width;
    }

    double getMaxY() {
        return y+height;
    }

    /**
     * Determines whether or not this <code>Rectangle2D</code>
         * is empty.
         * @return <code>true</code> if this <code>Rectangle2D</code>
         * is empty; <code>false</code> otherwise.
     * @since 1.2
     */
    bool isEmpty() {
        return (width <= 0.0) || (height <= 0.0);
    }

    /**
     * Sets the location and size of this <code>Rectangle2D</code>
         * to the specified double values.
         * @param x,&nbsp;y the coordinates to which to set the
         * upper left corner of this <code>Rectangle2D</code>
         * @param w the value to use to set the width of this
         * <code>Rectangle2D</code>
         * @param h the value to use to set the height of this
         * <code>Rectangle2D</code>
     * @since 1.2
     */
    void setRect(double x, double y, double w, double h) {
        this->x = x;
        this->y = y;
        this->width = w;
        this->height = h;
    }

    /**
     * Sets this <code>Rectangle2D</code> to be the same as the
         * specified <code>Rectangle2D</code>.
         * @param r the specified <code>Rectangle2D</code>
     * @since 1.2
     */
    void setRect(Rectangle2D& r) {
        this->x = r.getX();
        this->y = r.getY();
        this->width = r.getWidth();
        this->height = r.getHeight();
    }

    /**
     * Tests if a specified coordinate is inside the boundary of this
     * <code>Rectangle2D</code>.
     * @param x,&nbsp;y the coordinates to test
     * @return <code>true</code> if the specified coordinates are
     * inside the boundary of this <code>Rectangle2D</code>;
     * <code>false</code> otherwise.
     * @since 1.2
     */
    bool contains(double x, double y) {
        double x0 = getX();
        double y0 = getY();
        return (x >= x0 &&
            y >= y0 &&
            x < x0 + getWidth() &&
            y < y0 + getHeight());
    }

    /**
     * Tests if the interior of this <code>Rectangle2D</code> entirely
     * contains the specified set of rectangular coordinates.
     * @param x,&nbsp;y the coordinates of the upper left corner
     * of the specified set of rectangular coordinates
     * @param w the width of the specified set of rectangular
     * coordinates
     * @param h the height of the specified set of rectangular
     * coordinates
     * @return <code>true</code> if this <code>Rectangle2D</code>
     * entirely contains specified set of rectangular
     * coordinates; <code>false</code> otherwise.
     * @since 1.2
     */
    bool contains(double x, double y, double w, double h) {
        if (isEmpty() || w <= 0 || h <= 0) {
            return false;
        }
        double x0 = getX();
        double y0 = getY();
        return (x >= x0 &&
            y >= y0 &&
            (x + w) <= x0 + getWidth() &&
            (y + h) <= y0 + getHeight());
    }

    /**
     * Adds a point, specified by the double precision arguments
     * <code>newx</code> and <code>newy</code>, to this
     * <code>Rectangle2D</code>.  The resulting <code>Rectangle2D</code>
     * is the smallest <code>Rectangle2D</code> that
     * contains both the original <code>Rectangle2D</code> and the
     * specified point.
     * <p>
     * After adding a point, a call to <code>contains</code> with the
     * added point as an argument does not necessarily return
     * <code>true</code>. The <code>contains</code> method does not
     * return <code>true</code> for points on the right or bottom
     * edges of a rectangle. Therefore, if the added point falls on
     * the left or bottom edge of the enlarged rectangle,
     * <code>contains</code> returns <code>false</code> for that point.
     * @param newx,&nbsp;newy the coordinates of the new point
     * @since     JDK1.0
     */
    void add(double newx, double newy) {
        double x1 = min(getMinX(), newx);
        double x2 = max(getMaxX(), newx);
        double y1 = min(getMinY(), newy);
        double y2 = max(getMaxY(), newy);
        setRect(x1, y1, x2 - x1, y2 - y1);
    }

    /**
     * Adds a <code>Rectangle2D</code> object to this
     * <code>Rectangle2D</code>.  The resulting <code>Rectangle2D</code>
     * is the union of the two <code>Rectangle2D</code> objects.
     * @param r the <code>Rectangle2D</code> to add to this
     * <code>Rectangle2D</code>.
     * @since     JDK1.0
     */
    void add(Rectangle2D& r) {
        double x1 = min(getMinX(), r.getMinX());
        double x2 = max(getMaxX(), r.getMaxX());
        double y1 = min(getMinY(), r.getMinY());
        double y2 = max(getMaxY(), r.getMaxY());
        setRect(x1, y1, x2 - x1, y2 - y1);
    }

};

class Point2D {
public:

    /**
     * The X coordinate of this <code>Point2D</code>.
     * @since 1.2
     */
    double x;

    /**
     * The Y coordinate of this <code>Point2D</code>.
     * @since 1.2
     */
    double y;

    /**
     * Constructs and initializes a <code>Point2D</code> with
         * coordinates (0,&nbsp;0).
     * @since 1.2
     */
    Point2D() {
        x = y = 0.0;
    }

    /**
     * Constructs and initializes a <code>Point2D</code> with the
         * specified coordinates.
         * @param x,&nbsp;y the coordinates to which to set the newly
         * constructed <code>Point2D</code>
     * @since 1.2
     */
    Point2D(double x, double y) {
        this->x = x;
        this->y = y;
    }

    /**
     * Returns the X coordinate of this <code>Point2D</code>
         * in <code>double</code> precision.
         * @return the X coordinate of this <code>Point2D</code>.
     * @since 1.2
     */
    double getX() {
        return x;
    }

    /**
     * Returns the Y coordinate of this <code>Point2D</code> in
         * <code>double</code> precision.
         * @return the Y coordinate of this <code>Point2D</code>.
     * @since 1.2
     */
    double getY() {
        return y;
    }

    /**
     * Sets the location of this <code>Point2D</code> to the
         * specified <code>double</code> coordinates.
         * @param x,&nbsp;y the coordinates to which to set this
         * <code>Point2D</code>
     * @since 1.2
     */
    void setLocation(double x, double y) {
        this->x = x;
        this->y = y;
    }

    /**
     * Returns the square of the distance between two points.
     * @param X1,&nbsp;Y1 the coordinates of the first point
     * @param X2,&nbsp;Y2 the coordinates of the second point
     * @return the square of the distance between the two
     * sets of specified coordinates.
     */
    static double distanceSq(double X1, double Y1, double X2, double Y2) {
        X1 -= X2;
        Y1 -= Y2;
        return (X1 * X1 + Y1 * Y1);
    }

    /**
     * Returns the distance between two points.
     * @param X1,&nbsp;Y1 the coordinates of the first point
     * @param X2,&nbsp;Y2 the coordinates of the second point
     * @return the distance between the two sets of specified
     * coordinates.
     */
    static double distance(double X1, double Y1, double X2, double Y2) {
        X1 -= X2;
        Y1 -= Y2;
        return sqrt(X1 * X1 + Y1 * Y1);
    }

    /**
     * Returns the square of the distance from this
     * <code>Point2D</code> to a specified point.
     * @param PX,&nbsp;PY the coordinates of the other point
     * @return the square of the distance between this
     * <code>Point2D</code> and the specified point.
     */
    double distanceSq(double PX, double PY) {
        PX -= getX();
        PY -= getY();
        return (PX * PX + PY * PY);
    }

    /**
     * Returns the square of the distance from this
     * <code>Point2D</code> to a specified <code>Point2D</code>.
     * @param pt the specified <code>Point2D</code>
     * @return the square of the distance between this
     * <code>Point2D</code> to a specified <code>Point2D</code>.
     */
    double distanceSq(Point2D& pt) {
        double PX = pt.getX() - this->getX();
        double PY = pt.getY() - this->getY();
        return (PX * PX + PY * PY);
    }

    /**
     * Returns the distance from this <code>Point2D</code> to
     * a specified point.
     * @param PX,&nbsp;PY the coordinates of the specified
     * <code>Point2D</code>
     * @return the distance between this <code>Point2D</code>
     * and a specified point.
     */
    double distance(double PX, double PY) {
        PX -= getX();
        PY -= getY();
        return sqrt(PX * PX + PY * PY);
    }

    /**
     * Returns the distance from this <code>Point2D</code> to a
     * specified <code>Point2D</code>.
     * @param pt the specified <code>Point2D</code>
     * @return the distance between this <code>Point2D</code> and
     * the specified <code>Point2D</code>.
     */
    double distance(Point2D& pt) {
        double PX = pt.getX() - this->getX();
        double PY = pt.getY() - this->getY();
        return sqrt(PX * PX + PY * PY);
    }

};

#endif
