/*
 * $Id: geom.cxx,v 1.1 2002/02/15 23:44:28 skavish Exp $
 *
 * ==========================================================================
 *
 * The JGenerator Software License, Version 1.0
 *
 * Copyright (c) 2000 Dmitry Skavish (skavish@usa.net). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *    "This product includes software developed by Dmitry Skavish
 *     (skavish@usa.net, http://www.flashgap.com/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The name "The JGenerator" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact skavish@usa.net.
 *
 * 5. Products derived from this software may not be called "The JGenerator"
 *    nor may "The JGenerator" appear in their names without prior written
 *    permission of Dmitry Skavish.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL DMITRY SKAVISH OR THE OTHER
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "geom.hxx"

int AffineTransform::getType() {
    int ret = TYPE_IDENTITY;
    bool sgn0, sgn1;
    double M0, M1, M2, M3;
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        ret = TYPE_TRANSLATION;
        /* NOBREAK */
    case (APPLY_SHEAR | APPLY_SCALE):
        if ((M0 = m00) * (M2 = m01) + (M3 = m10) * (M1 = m11) != 0) {
            // Transformed unit vectors are not perpendicular...
            return TYPE_GENERAL_TRANSFORM;
        }
        sgn0 = (M0 >= 0.0);
        sgn1 = (M1 >= 0.0);
        if (sgn0 == sgn1) {
            // sgn(M0) == sgn(M1) therefore sgn(M2) == -sgn(M3)
            // This is the "unflipped" (right-handed) state
            if (M0 != M1 || M2 != -M3) {
                ret |= (TYPE_GENERAL_ROTATION | TYPE_GENERAL_SCALE);
            } else if (M0 * M1 - M2 * M3 != 1.0) {
                ret |= (TYPE_GENERAL_ROTATION | TYPE_UNIFORM_SCALE);
            } else {
                ret |= TYPE_GENERAL_ROTATION;
            }
        } else {
            // sgn(M0) == -sgn(M1) therefore sgn(M2) == sgn(M3)
            // This is the "flipped" (left-handed) state
            if (M0 != -M1 || M2 != M3) {
                ret |= (TYPE_GENERAL_ROTATION |
                        TYPE_FLIP |
                        TYPE_GENERAL_SCALE);
            } else if (M0 * M1 - M2 * M3 != 1.0) {
                ret |= (TYPE_GENERAL_ROTATION |
                        TYPE_FLIP |
                        TYPE_UNIFORM_SCALE);
            } else {
                ret |= (TYPE_GENERAL_ROTATION | TYPE_FLIP);
            }
        }
        break;
    case (APPLY_SHEAR | APPLY_TRANSLATE):
        ret = TYPE_TRANSLATION;
        /* NOBREAK */
    case (APPLY_SHEAR):
        sgn0 = ((M0 = m01) >= 0.0);
        sgn1 = ((M1 = m10) >= 0.0);
        if (sgn0 != sgn1) {
            // Different signs - simple 90 degree rotation
            if (M0 == -M1) {
                ret |= (TYPE_QUADRANT_ROTATION | TYPE_UNIFORM_SCALE);
            } else {
                ret |= (TYPE_QUADRANT_ROTATION | TYPE_GENERAL_SCALE);
            }
        } else {
            // Same signs - 90 degree rotation plus an axis flip too
            if (M0 == M1) {
                ret |= (TYPE_QUADRANT_ROTATION |
                        TYPE_FLIP |
                        TYPE_UNIFORM_SCALE);
            } else {
                ret |= (TYPE_QUADRANT_ROTATION |
                        TYPE_FLIP |
                        TYPE_GENERAL_SCALE);
            }
        }
        break;
    case (APPLY_SCALE | APPLY_TRANSLATE):
        ret = TYPE_TRANSLATION;
        /* NOBREAK */
    case (APPLY_SCALE):
        sgn0 = ((M0 = m00) >= 0.0);
        sgn1 = ((M1 = m11) >= 0.0);
        if (sgn0 == sgn1) {
            if (sgn0) {
                // Both scaling factors non-negative - simple scale
                if (M0 == M1) {
                    ret |= TYPE_UNIFORM_SCALE;
                } else {
                    ret |= TYPE_GENERAL_SCALE;
                }
            } else {
                // Both scaling factors negative - 180 degree rotation
                if (M0 == M1) {
                    ret |= (TYPE_QUADRANT_ROTATION | TYPE_UNIFORM_SCALE);
                } else {
                    ret |= (TYPE_QUADRANT_ROTATION | TYPE_GENERAL_SCALE);
                }
            }
        } else {
            // Scaling factor signs different - flip about some axis
            if (M0 == -M1) {
                if (M0 == 1.0 || M0 == -1.0) {
                    ret |= TYPE_FLIP;
                } else {
                    ret |= (TYPE_FLIP | TYPE_UNIFORM_SCALE);
                }
            } else {
                ret |= (TYPE_FLIP | TYPE_GENERAL_SCALE);
            }
        }
        break;
    case (APPLY_TRANSLATE):
        ret = TYPE_TRANSLATION;
        break;
    case (APPLY_IDENTITY):
        break;
    }
    return ret;
}

double AffineTransform::getDeterminant() {
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SHEAR | APPLY_SCALE):
        return m00 * m11 - m01 * m10;
    case (APPLY_SHEAR | APPLY_TRANSLATE):
    case (APPLY_SHEAR):
        return -(m01 * m10);
    case (APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SCALE):
        return m00 * m11;
    case (APPLY_TRANSLATE):
    case (APPLY_IDENTITY):
        return 1.0;
    }
}

void AffineTransform::updateState() {
    if (m01 == 0.0 && m10 == 0.0) {
        if (m00 == 1.0 && m11 == 1.0) {
            if (m02 == 0.0 && m12 == 0.0) {
                state = APPLY_IDENTITY;
            } else {
                state = APPLY_TRANSLATE;
            }
        } else {
            if (m02 == 0.0 && m12 == 0.0) {
                state = APPLY_SCALE;
            } else {
                state = (APPLY_SCALE | APPLY_TRANSLATE);
            }
        }
    } else {
        if (m00 == 0.0 && m11 == 0.0) {
            if (m02 == 0.0 && m12 == 0.0) {
                state = APPLY_SHEAR;
            } else {
                state = (APPLY_SHEAR | APPLY_TRANSLATE);
            }
        } else {
            if (m02 == 0.0 && m12 == 0.0) {
                state = (APPLY_SHEAR | APPLY_SCALE);
            } else {
                state = (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE);
            }
        }
    }
}

void AffineTransform::translate(double tx, double ty) {
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        m02 = tx * m00 + ty * m01 + m02;
        m12 = tx * m10 + ty * m11 + m12;
        return;
    case (APPLY_SHEAR | APPLY_SCALE):
        m02 = tx * m00 + ty * m01;
        m12 = tx * m10 + ty * m11;
        state = APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE;
        return;
    case (APPLY_SHEAR | APPLY_TRANSLATE):
        m02 = ty * m01 + m02;
        m12 = tx * m10 + m12;
        return;
    case (APPLY_SHEAR):
        m02 = ty * m01;
        m12 = tx * m10;
        state = APPLY_SHEAR | APPLY_TRANSLATE;
        return;
    case (APPLY_SCALE | APPLY_TRANSLATE):
        m02 = tx * m00 + m02;
        m12 = ty * m11 + m12;
        return;
    case (APPLY_SCALE):
        m02 = tx * m00;
        m12 = ty * m11;
        state = APPLY_SCALE | APPLY_TRANSLATE;
        return;
    case (APPLY_TRANSLATE):
        m02 = tx + m02;
        m12 = ty + m12;
        return;
    case (APPLY_IDENTITY):
        m02 = tx;
        m12 = ty;
        state = APPLY_TRANSLATE;
        return;
    }
}

void AffineTransform::rotate(double theta) {
    double si = sin(theta);
    double co = cos(theta);
    if (fabs(si) < 1E-15) {
        if (co < 0.0) {
            m00 = -m00;
            m11 = -m11;
            int state = this->state;
            if ((state & (APPLY_SHEAR)) != 0) {
                // If there was a shear, then this rotation has no
                // effect on the state.
                m01 = -m01;
                m10 = -m10;
            } else {
                // No shear means the SCALE state may toggle when
                // m00 and m11 are negated.
                if (m00 == 1.0 && m11 == 1.0) {
                    this->state = state & ~APPLY_SCALE;
                } else {
                    this->state = state | APPLY_SCALE;
                }
            }
        }
        return;
    }
    if (fabs(co) < 1E-15) {
        if (si < 0.0) {
            double M0 = m00;
            m00 = -m01;
            m01 = M0;
            M0 = m10;
            m10 = -m11;
            m11 = M0;
        } else {
            double M0 = m00;
            m00 = m01;
            m01 = -M0;
            M0 = m10;
            m10 = m11;
            m11 = -M0;
        }
        state = rot90conversion[state];
        return;
    }
    double M0, M1;
    M0 = m00;
    M1 = m01;
    m00 =  co * M0 + si * M1;
    m01 = -si * M0 + co * M1;
    M0 = m10;
    M1 = m11;
    m10 =  co * M0 + si * M1;
    m11 = -si * M0 + co * M1;
    updateState();
}

int AffineTransform::rot90conversion[] = {
    APPLY_SHEAR, APPLY_SHEAR | APPLY_TRANSLATE,
    APPLY_SHEAR, APPLY_SHEAR | APPLY_TRANSLATE,
    APPLY_SCALE, APPLY_SCALE | APPLY_TRANSLATE,
    APPLY_SHEAR | APPLY_SCALE,
    APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE,
};

void AffineTransform::rotate(double theta, double x, double y) {
    // REMIND: Simple for now - optimize later
    translate(x, y);
    rotate(theta);
    translate(-x, -y);
}

void AffineTransform::scale(double sx, double sy) {
    int state = this->state;
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SHEAR | APPLY_SCALE):
        m00 *= sx;
        m11 *= sy;
        /* NOBREAK */
    case (APPLY_SHEAR | APPLY_TRANSLATE):
    case (APPLY_SHEAR):
        m01 *= sy;
        m10 *= sx;
        return;
    case (APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SCALE):
        m00 *= sx;
        m11 *= sy;
        return;
    case (APPLY_TRANSLATE):
    case (APPLY_IDENTITY):
        m00 = sx;
        m11 = sy;
        this->state = state | APPLY_SCALE;
        return;
    }
}

void AffineTransform::shear(double shx, double shy) {
    int state = this->state;
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SHEAR | APPLY_SCALE):
        double M0, M1;
        M0 = m00;
        M1 = m01;
        m00 = M0 + M1 * shy;
        m01 = M0 * shx + M1;

        M0 = m10;
        M1 = m11;
        m10 = M0 + M1 * shy;
        m11 = M0 * shx + M1;
        return;
    case (APPLY_SHEAR | APPLY_TRANSLATE):
    case (APPLY_SHEAR):
        m00 = m01 * shy;
        m11 = m10 * shx;
        this->state = state | APPLY_SCALE;
        return;
    case (APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SCALE):
        m01 = m00 * shx;
        m10 = m11 * shy;
        this->state = state | APPLY_SHEAR;
        return;
    case (APPLY_TRANSLATE):
    case (APPLY_IDENTITY):
        m01 = shx;
        m10 = shy;
        this->state = state | APPLY_SCALE | APPLY_SHEAR;
        return;
    }
}

void AffineTransform::setToIdentity() {
    m00 = m11 = 1.0;
    m10 = m01 = m02 = m12 = 0.0;
    state = APPLY_IDENTITY;
}

void AffineTransform::setToTranslation(double tx, double ty) {
    m00 = 1.0;
    m10 = 0.0;
    m01 = 0.0;
    m11 = 1.0;
    m02 = tx;
    m12 = ty;
    state = APPLY_TRANSLATE;
}

void AffineTransform::setToRotation(double theta) {
    m02 = 0.0;
    m12 = 0.0;
    double si = sin(theta);
    double co = cos(theta);
    if (fabs(si) < 1E-15) {
        m01 = m10 = 0.0;
        if (co < 0) {
            m00 = m11 = -1.0;
            state = APPLY_SCALE;
        } else {
            m00 = m11 = 1.0;
            state = APPLY_IDENTITY;
        }
        return;
    }
    if (fabs(co) < 1E-15) {
        m00 = m11 = 0.0;
        if (si < 0.0) {
            m01 = 1.0;
            m10 = -1.0;
        } else {
            m01 = -1.0;
            m10 = 1.0;
        }
        state = APPLY_SHEAR;
        return;
    }
    m00 = co;
    m01 = -si;
    m10 = si;
    m11 = co;
    state = APPLY_SHEAR | APPLY_SCALE;
    return;
}

void AffineTransform::setToRotation(double theta, double x, double y) {
    setToRotation(theta);
    double si = m10;
    double oneMinusCos = 1.0 - m00;
    m02 = x * oneMinusCos + y * si;
    m12 = y * oneMinusCos - x * si;
    state |= APPLY_TRANSLATE;
    return;
}

void AffineTransform::setToScale(double sx, double sy) {
    m00 = sx;
    m10 = 0.0;
    m01 = 0.0;
    m11 = sy;
    m02 = 0.0;
    m12 = 0.0;
    state = APPLY_SCALE;
}

void AffineTransform::setToShear(double shx, double shy) {
    m00 = 1.0;
    m01 = shx;
    m10 = shy;
    m11 = 1.0;
    m02 = 0.0;
    m12 = 0.0;
    state = APPLY_SHEAR | APPLY_SCALE;
}

void AffineTransform::setTransform(AffineTransform& Tx) {
    this->m00 = Tx.m00;
    this->m10 = Tx.m10;
    this->m01 = Tx.m01;
    this->m11 = Tx.m11;
    this->m02 = Tx.m02;
    this->m12 = Tx.m12;
    this->state = Tx.state;
}

void AffineTransform::setTransform(double m00, double m10,
                  double m01, double m11,
                  double m02, double m12) {
    this->m00 = m00;
    this->m10 = m10;
    this->m01 = m01;
    this->m11 = m11;
    this->m02 = m02;
    this->m12 = m12;
    updateState();
}

void AffineTransform::concatenate(AffineTransform& Tx) {
    double M0, M1;
    double T00, T01, T10, T11;
    double T02, T12;
    int mystate = state;
    int txstate = Tx.state;
    switch ((txstate << HI_SHIFT) | mystate) {

    /* ---------- Tx == IDENTITY cases ---------- */
    case (HI_IDENTITY | APPLY_IDENTITY):
    case (HI_IDENTITY | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SCALE):
    case (HI_IDENTITY | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SHEAR):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_SCALE):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        return;

        /* ---------- this == IDENTITY cases ---------- */
    case (HI_SHEAR | HI_SCALE | HI_TRANSLATE | APPLY_IDENTITY):
        m01 = Tx.m01;
        m10 = Tx.m10;
        /* NOBREAK */
    case (HI_SCALE | HI_TRANSLATE | APPLY_IDENTITY):
        m00 = Tx.m00;
        m11 = Tx.m11;
        /* NOBREAK */
    case (HI_TRANSLATE | APPLY_IDENTITY):
        m02 = Tx.m02;
        m12 = Tx.m12;
        state = txstate;
        return;
    case (HI_SHEAR | HI_SCALE | APPLY_IDENTITY):
        m01 = Tx.m01;
        m10 = Tx.m10;
        /* NOBREAK */
    case (HI_SCALE | APPLY_IDENTITY):
        m00 = Tx.m00;
        m11 = Tx.m11;
        state = txstate;
        return;
    case (HI_SHEAR | HI_TRANSLATE | APPLY_IDENTITY):
        m02 = Tx.m02;
        m12 = Tx.m12;
        /* NOBREAK */
    case (HI_SHEAR | APPLY_IDENTITY):
        m01 = Tx.m01;
        m10 = Tx.m10;
        m00 = m11 = 0.0;
        state = txstate;
        return;

        /* ---------- Tx == TRANSLATE cases ---------- */
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        T02 = Tx.m02;
        T12 = Tx.m12;
        m02 = T02 * m00 + T12 * m01 + m02;
        m12 = T02 * m10 + T12 * m11 + m12;
        return;
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_SCALE):
        T02 = Tx.m02;
        T12 = Tx.m12;
        m02 = T02 * m00 + T12 * m01;
        m12 = T02 * m10 + T12 * m11;
        state = APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE;
        return;
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_TRANSLATE):
        m02 = Tx.m12 * m01 + m02;
        m12 = Tx.m02 * m10 + m12;
        return;
    case (HI_TRANSLATE | APPLY_SHEAR):
        m02 = Tx.m12 * m01;
        m12 = Tx.m02 * m10;
        state = APPLY_SHEAR | APPLY_TRANSLATE;
        return;
    case (HI_TRANSLATE | APPLY_SCALE | APPLY_TRANSLATE):
        m02 = Tx.m02 * m00 + m02;
        m12 = Tx.m12 * m11 + m12;
        return;
    case (HI_TRANSLATE | APPLY_SCALE):
        m02 = Tx.m02 * m00;
        m12 = Tx.m12 * m11;
        state = APPLY_SCALE | APPLY_TRANSLATE;
        return;
    case (HI_TRANSLATE | APPLY_TRANSLATE):
        m02 = Tx.m02 + m02;
        m12 = Tx.m12 + m12;
        return;

        /* ---------- Tx == SCALE cases ---------- */
    case (HI_SCALE | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SHEAR | APPLY_SCALE):
        T00 = Tx.m00; T11 = Tx.m11;
        m00 *= T00;
        m11 *= T11;
        m01 *= T11;
        m10 *= T00;
        return;
    case (HI_SCALE | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SHEAR):
        m01 *= Tx.m11;
        m10 *= Tx.m00;
        return;
    case (HI_SCALE | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SCALE):
        m00 *= Tx.m00;
        m11 *= Tx.m11;
        return;
    case (HI_SCALE | APPLY_TRANSLATE):
        m00 = Tx.m00;
        m11 = Tx.m11;
        state = APPLY_SCALE | APPLY_TRANSLATE;
        return;

        /* ---------- Tx == SHEAR cases ---------- */
    case (HI_SHEAR | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SHEAR | APPLY_SCALE):
        T01 = Tx.m01; T10 = Tx.m10;
        M0 = m00;
        m00 = m01 * T10;
        m01 = M0 * T01;
        M0 = m10;
        m10 = m11 * T10;
        m11 = M0 * T01;
        return;
    case (HI_SHEAR | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SHEAR):
        m00 = m01 * Tx.m10;
        m01 = 0.0;
        m11 = m10 * Tx.m01;
        m10 = 0.0;
        state = mystate ^ (APPLY_SHEAR | APPLY_SCALE);
        return;
    case (HI_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SCALE):
        m01 = m00 * Tx.m01;
        m00 = 0.0;
        m10 = m11 * Tx.m10;
        m11 = 0.0;
        state = mystate ^ (APPLY_SHEAR | APPLY_SCALE);
        return;
    case (HI_SHEAR | APPLY_TRANSLATE):
        m00 = 0.0;
        m01 = Tx.m01;
        m10 = Tx.m10;
        m11 = 0.0;
        state = APPLY_TRANSLATE | APPLY_SHEAR;
        return;
    }
    // If Tx has more than one attribute, it is not worth optimizing
    // all of those cases...
    T00 = Tx.m00; T01 = Tx.m01; T02 = Tx.m02;
    T10 = Tx.m10; T11 = Tx.m11; T12 = Tx.m12;
    switch (mystate) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE):
        state = mystate | txstate;
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        M0 = m00;
        M1 = m01;
        m00  = T00 * M0 + T10 * M1;
        m01  = T01 * M0 + T11 * M1;
        m02 += T02 * M0 + T12 * M1;

        M0 = m10;
        M1 = m11;
        m10  = T00 * M0 + T10 * M1;
        m11  = T01 * M0 + T11 * M1;
        m12 += T02 * M0 + T12 * M1;
        return;

    case (APPLY_SHEAR | APPLY_TRANSLATE):
    case (APPLY_SHEAR):
        M0 = m01;
        m00  = T10 * M0;
        m01  = T11 * M0;
        m02 += T12 * M0;

        M0 = m10;
        m10  = T00 * M0;
        m11  = T01 * M0;
        m12 += T02 * M0;
        break;

    case (APPLY_SCALE | APPLY_TRANSLATE):
    case (APPLY_SCALE):
        M0 = m00;
        m00  = T00 * M0;
        m01  = T01 * M0;
        m02 += T02 * M0;

        M0 = m11;
        m10  = T10 * M0;
        m11  = T11 * M0;
        m12 += T12 * M0;
        break;

    case (APPLY_TRANSLATE):
        m00  = T00;
        m01  = T01;
        m02 += T02;

        m10  = T10;
        m11  = T11;
        m12 += T12;
        state = txstate | APPLY_TRANSLATE;
        return;
    }
    updateState();
}

void AffineTransform::preConcatenate(AffineTransform& Tx) {
    double M0, M1;
    double T00, T01, T10, T11;
    double T02, T12;
    int mystate = state;
    int txstate = Tx.state;
    switch ((txstate << HI_SHIFT) | mystate) {
    case (HI_IDENTITY | APPLY_IDENTITY):
    case (HI_IDENTITY | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SCALE):
    case (HI_IDENTITY | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SHEAR):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_SCALE):
    case (HI_IDENTITY | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        // Tx is IDENTITY...
        return;

    case (HI_TRANSLATE | APPLY_IDENTITY):
    case (HI_TRANSLATE | APPLY_SCALE):
    case (HI_TRANSLATE | APPLY_SHEAR):
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_SCALE):
        // Tx is TRANSLATE, this has no TRANSLATE
        m02 = Tx.m02;
        m12 = Tx.m12;
        state = mystate | APPLY_TRANSLATE;
        return;

    case (HI_TRANSLATE | APPLY_TRANSLATE):
    case (HI_TRANSLATE | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_TRANSLATE | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        // Tx is TRANSLATE, this has one too
        m02 = m02 + Tx.m02;
        m12 = m12 + Tx.m12;
        return;

    case (HI_SCALE | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_IDENTITY):
        // Only these two existing states need a new state
        state = mystate | APPLY_SCALE;
        /* NOBREAK */
    case (HI_SCALE | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SHEAR | APPLY_SCALE):
    case (HI_SCALE | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SHEAR):
    case (HI_SCALE | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SCALE | APPLY_SCALE):
        // Tx is SCALE, this is anything
        T00 = Tx.m00;
        T11 = Tx.m11;
        if ((mystate & APPLY_SHEAR) != 0) {
            m01 = m01 * T00;
            m10 = m10 * T11;
            if ((mystate & APPLY_SCALE) != 0) {
                m00 = m00 * T00;
                m11 = m11 * T11;
            }
        } else {
            m00 = m00 * T00;
            m11 = m11 * T11;
        }
        if ((mystate & APPLY_TRANSLATE) != 0) {
            m02 = m02 * T00;
            m12 = m12 * T11;
        }
        return;
    case (HI_SHEAR | APPLY_SHEAR | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SHEAR):
        mystate = mystate | APPLY_SCALE;
        /* NOBREAK */
    case (HI_SHEAR | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_IDENTITY):
    case (HI_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SCALE):
        state = mystate ^ APPLY_SHEAR;
        /* NOBREAK */
    case (HI_SHEAR | APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
    case (HI_SHEAR | APPLY_SHEAR | APPLY_SCALE):
        // Tx is SHEAR, this is anything
        T01 = Tx.m01;
        T10 = Tx.m10;

        M0 = m00;
        m00 = m10 * T01;
        m10 = M0 * T10;

        M0 = m01;
        m01 = m11 * T01;
        m11 = M0 * T10;

        M0 = m02;
        m02 = m12 * T01;
        m12 = M0 * T10;
        return;
    }
    // If Tx has more than one attribute, it is not worth optimizing
    // all of those cases...
    T00 = Tx.m00; T01 = Tx.m01; T02 = Tx.m02;
    T10 = Tx.m10; T11 = Tx.m11; T12 = Tx.m12;
    switch (mystate) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        M0 = m02;
        M1 = m12;
        T02 += M0 * T00 + M1 * T01;
        T12 += M0 * T10 + M1 * T11;

        /* NOBREAK */
    case (APPLY_SHEAR | APPLY_SCALE):
        m02 = T02;
        m12 = T12;

        M0 = m00;
        M1 = m10;
        m00 = M0 * T00 + M1 * T01;
        m10 = M0 * T10 + M1 * T11;

        M0 = m01;
        M1 = m11;
        m01 = M0 * T00 + M1 * T01;
        m11 = M0 * T10 + M1 * T11;
        break;

    case (APPLY_SHEAR | APPLY_TRANSLATE):
        M0 = m02;
        M1 = m12;
        T02 += M0 * T00 + M1 * T01;
        T12 += M0 * T10 + M1 * T11;

        /* NOBREAK */
    case (APPLY_SHEAR):
        m02 = T02;
        m12 = T12;

        M0 = m10;
        m00 = M0 * T01;
        m10 = M0 * T11;

        M0 = m01;
        m01 = M0 * T00;
        m11 = M0 * T10;
        break;

    case (APPLY_SCALE | APPLY_TRANSLATE):
        M0 = m02;
        M1 = m12;
        T02 += M0 * T00 + M1 * T01;
        T12 += M0 * T10 + M1 * T11;

        /* NOBREAK */
    case (APPLY_SCALE):
        m02 = T02;
        m12 = T12;

        M0 = m00;
        m00 = M0 * T00;
        m10 = M0 * T10;

        M0 = m11;
        m01 = M0 * T01;
        m11 = M0 * T11;
        break;

    case (APPLY_TRANSLATE):
        M0 = m02;
        M1 = m12;
        T02 += M0 * T00 + M1 * T01;
        T12 += M0 * T10 + M1 * T11;

        /* NOBREAK */
    case (APPLY_IDENTITY):
        m02 = T02;
        m12 = T12;

        m00 = T00;
        m10 = T10;

        m01 = T01;
        m11 = T11;

        state = mystate | txstate;
        return;
    }
    updateState();
}

void AffineTransform::transform(double srcPts[], int srcOff,
               double dstPts[], int dstOff,
               int numPts) {
    double M00, M01, M02, M10, M11, M12;    // For caching
    if (dstPts == srcPts &&
        dstOff > srcOff && dstOff < srcOff + numPts * 2) {
        // If the arrays overlap partially with the destination higher
        // than the source and we transform the coordinates normally
        // we would overwrite some of the later source coordinates
        // with results of previous transformations.
        // To get around this we use arraycopy to copy the points
        // to their final destination with correct overwrite
        // handling and then transform them in place in the new
        // safer location.
        memcpy(dstPts+dstOff, srcPts+srcOff, numPts*2);
        // srcPts = dstPts;     // They are known to be equal.
        srcOff = dstOff;
    }
    switch (state) {
    default:
        stateError();
        /* NOTREACHED */
    case (APPLY_SHEAR | APPLY_SCALE | APPLY_TRANSLATE):
        M00 = m00; M01 = m01; M02 = m02;
        M10 = m10; M11 = m11; M12 = m12;
        while (--numPts >= 0) {
            double x = srcPts[srcOff++];
            double y = srcPts[srcOff++];
            dstPts[dstOff++] = M00 * x + M01 * y + M02;
            dstPts[dstOff++] = M10 * x + M11 * y + M12;
        }
        return;
    case (APPLY_SHEAR | APPLY_SCALE):
        M00 = m00; M01 = m01;
        M10 = m10; M11 = m11;
        while (--numPts >= 0) {
            double x = srcPts[srcOff++];
            double y = srcPts[srcOff++];
            dstPts[dstOff++] = M00 * x + M01 * y;
            dstPts[dstOff++] = M10 * x + M11 * y;
        }
        return;
    case (APPLY_SHEAR | APPLY_TRANSLATE):
        M01 = m01; M02 = m02;
        M10 = m10; M12 = m12;
        while (--numPts >= 0) {
            double x = srcPts[srcOff++];
            dstPts[dstOff++] = M01 * srcPts[srcOff++] + M02;
            dstPts[dstOff++] = M10 * x + M12;
        }
        return;
    case (APPLY_SHEAR):
        M01 = m01; M10 = m10;
        while (--numPts >= 0) {
            double x = srcPts[srcOff++];
            dstPts[dstOff++] = M01 * srcPts[srcOff++];
            dstPts[dstOff++] = M10 * x;
        }
        return;
    case (APPLY_SCALE | APPLY_TRANSLATE):
        M00 = m00; M02 = m02;
        M11 = m11; M12 = m12;
        while (--numPts >= 0) {
            dstPts[dstOff++] = M00 * srcPts[srcOff++] + M02;
            dstPts[dstOff++] = M11 * srcPts[srcOff++] + M12;
        }
        return;
    case (APPLY_SCALE):
        M00 = m00; M11 = m11;
        while (--numPts >= 0) {
            dstPts[dstOff++] = M00 * srcPts[srcOff++];
            dstPts[dstOff++] = M11 * srcPts[srcOff++];
        }
        return;
    case (APPLY_TRANSLATE):
        M02 = m02; M12 = m12;
        while (--numPts >= 0) {
            dstPts[dstOff++] = srcPts[srcOff++] + M02;
            dstPts[dstOff++] = srcPts[srcOff++] + M12;
        }
        return;
    case (APPLY_IDENTITY):
        if (srcPts != dstPts || srcOff != dstOff) {
            memcpy(dstPts+dstOff, srcPts+srcOff, numPts * 2);
        }
        return;
    }

    /* NOTREACHED */
}


