/*
 * tmms.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <locale.h>
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <getopt.h>
#include <signal.h>
#include <ncurses.h>
#include <readline/readline.h>
#include "windows.h"
#include "commands.h"
#include "tmms.h"
#include "xmem.h"
#include "printlog.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "loadsave.h"
#include "control.h"
#include "modules.h"
#include "mp3_mod.h"
#include "ogg_mod.h"



#if DEBUG_LEVEL!=0
#define LOGFILE "tmms.log"
FILE        *logfile      = NULL;
static int   lgfile       = FALSE;
#endif

static char *playlistloc  = NULL;
char        *outputdevice = NULL;

static Cmdslst commands[] = {
    {'p',        play_pause},
    {'s',        stop},
    {'f',        fforward},
    {'r',        frewind},
    {'n',        next_file},
    {'b',        prev_file},
    {'m',        repeat_mode},
    {'t',        time_data},
    {'L',        reload_list},
    {'l',        load_fileordir},
    {'S',        save_fileordir},
    {'a',        add_fileorstream},
    {'R',        toggle_pm},
    {'q',        quit},
    //{KEY_RESIZE, redisp},
    /* Keys for playlist */
    {KEY_UP,     previous},
    {KEY_DOWN,   next},
    {KEY_PPAGE,  pgup},
    {KEY_NPAGE,  pgdn},
    {KEY_HOME,   first},
    {KEY_END,    last},
    {KEY_ENTER,  enter},
    {10,         enter},
    {KEY_DC,     remove_entry},
    {' ',        select_entry},
    {'A',        sel_all},
    {'u',        unsel_all},
    {'i',        invert_sel},
    {'M',        move_entry},
    {0,          NULL}
};



void init_curses() {
    unsigned int x, y;

    /* Ncurses initialization */
    use_env(TRUE);
    initscr();
#ifdef COLORIZED
    init_colors();
#endif
    clear();
    keypad(stdscr, TRUE);
    cbreak();
    //nonl();
    noecho();
    curs_set(0);
    refresh();
    getmaxyx(stdscr, y, x);
    if (y < 7) playlist_display = FALSE;
}


int redisp() {
    endwin();
    initscr();
    curs_set(0);
    display_player();
    display_playlist();
    return CONT;
}


int quit() {
    if (get_status() == PLAYING) {
        stop_player();
    }
    return QUIT;
}


int reload_list() {
    playlist_free();
    load_playlist(playlistloc);
    return redisp();
}


void interact() {
    int cont = CONT;

    while (cont == CONT) {
        cont = command_loop(commands);
    }
}


void usage(int help) {
    if (help) {
        printf("Usage : " PROGRAM " [-h] [-v] [-P] [-o device] [-p playlist]\n"
               " -h          : this help\n"
               " -v          : program version\n"
#if DEBUG_LEVEL!=0
               " -d          : debug mode (print to " LOGFILE ")\n"
#endif
               " -P          : don't display playlist\n"
               " -o device   : output device (oss, esd...)\n"
               " -p playlist : load playlist file or directory\n"
               "\n");
    } else {
        printf(PACKAGE " v" VERSION
               " ("OSTYPE "-" MACHTYPE "),  Thomas Nemeth 2004.\n");
    }
}


void parse_argv(int argc, char *argv[]) {
    int   option;

#if DEBUG_LEVEL!=0
    while ((option = getopt(argc, argv, "hvdPo:p:")) != -1) {
#else
    while ((option = getopt(argc, argv, "hvPo:p:")) != -1) {
#endif
        switch (option) {
            case 'h' :
                usage(1);
                exit(0);
                break;
            case 'v' :
                usage(0);
                exit(0);
                break;
#if DEBUG_LEVEL!=0
            case 'd' :
                lgfile = TRUE;
                break;
#endif
            case 'P' :
                playlist_display = FALSE;
                break;
            case 'o' :
                outputdevice = xstrdup(optarg);
                break;
            case 'p' :
                playlistloc = xstrdup(optarg);
                break;
            default :
                usage(1);
                exit(-1);
                break;
        }
    }
}


void sig_hand(int sig) {
    if (sig == SIGWINCH) {
        printlog(1, "New screen size = %dLx%dC\n", LINES, COLS);
        redisp();
    } else {
        printlog(1, "Signal error (%d)\n", sig);
    }
}


void tmms_init() {
    pthread_mutexattr_t mutex_attr;

    /* Mutex initialization */
    pthread_mutexattr_init(&mutex_attr);
    pthread_mutex_init(&mutex_request, &mutex_attr);
    pthread_mutex_init(&mutex_status,  &mutex_attr);
    pthread_mutex_init(&mutex_current, &mutex_attr);
    pthread_mutex_init(&mutex_display, &mutex_attr);

#if DEBUG_LEVEL!=0
    /* Debug logfile */
    if (lgfile) logfile = fopen(LOGFILE, "w");
#endif

    /* Modules initialization */
    mod_list = NULL;
    register_mp3();
    register_ogg();

#ifdef COLORIZED
    uses_color = TRUE;
#else
    uses_color = FALSE;
#endif
    /* activate and save current locales */
    all_locale = setlocale(LC_ALL, "");

    /* where to find locale files */
    /*bindtextdomain(PACKAGE, LOCALEDIR); */
    /* set locale domain */
    /*textdomain(PACKAGE); */

    /* Init libncurses */
    init_curses();

    /* Terminal resizing */
    signal(SIGWINCH, sig_hand);
}


void tmms_deinit() {
#if DEBUG_LEVEL!=0
    if (lgfile && logfile) fclose(logfile);
#endif

    pthread_mutex_destroy(&mutex_request);
    pthread_mutex_destroy(&mutex_status);
    pthread_mutex_destroy(&mutex_current);
    pthread_mutex_destroy(&mutex_display);

    playlist_free();
    unregister_modules();
    clear();
    refresh();
    endwin();
}


int main(int argc, char *argv[]) {
    pthread_t pthread_id;

    parse_argv(argc, argv);
    tmms_init();
    load_playlist(playlistloc);

    if (pthread_create(&pthread_id, NULL, (void*)thread_control, NULL) == -1) {
        fprintf(stderr, "Error creating player control thread.\n");
    } else {
        display_player();
        display_playlist();
        interact();
    }

    tmms_deinit();

    return 0; 
}

