/*
 * control.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <pthread.h>
#include "defines.h"
#include <ncurses.h>
#include "tmms.h"
#include "xmem.h"
#include "commands.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "modules.h"
#include "control.h"
#include "printlog.h"


#define BUFLEN 1024

extern char *outputdevice;

static int    frame, input_end;
static int    minutes, seconds;
static int    rmins, rsecs;
static pid_t  p_pid;
static int    fildes[2];
static Status status, request;


/* User Request */
static int request_change(Status *old) {
    int cont = FALSE;

    pthread_mutex_lock(&mutex_request);
    if (request != *old) cont = TRUE;
    *old = request;
    pthread_mutex_unlock(&mutex_request);
    return cont;
}


void set_command(Status s) {
    pthread_mutex_lock(&mutex_request);
    request = s;
    pthread_mutex_unlock(&mutex_request);
}


static Status wait_request() {
    Status s = NOTHING;
    struct timespec ts = { 0, 1000 };

    set_command(NOTHING);
    while (! request_change(&s)) {
        nanosleep(&ts, NULL); // XXX
    }
    set_command(NOTHING);
    return s;
}


/* Internal status */

Status get_status() {
    Status s;
    pthread_mutex_lock(&mutex_status);
    s = status;
    pthread_mutex_unlock(&mutex_status);
    return s;
}


static void set_status(Status s) {
    pthread_mutex_lock(&mutex_status);
    status = s;
    pthread_mutex_unlock(&mutex_status);
    display_status();
}


/* Real control operations (launch, parse, stop)... */

static void player_cmd() {
    char *cmd = get_command();
    char *opt = get_option();
    char *pos = get_position(frame, seconds);
    char *args[8];
    int   i, arg = 0;

    if ((! cmd) || (! opt)) return;
    /* Arguments */
    args[arg++] = cmd;
    args[arg++] = opt;
    if (outputdevice) {
        args[arg++] = get_device();
        args[arg++] = outputdevice;
    }
    if (pos) {
        args[arg++] = get_posopt();
        args[arg++] = pos;
    }
    args[arg++] = playlist_current();
    for (i = arg ; i < 8 ; i++) args[i] = NULL;
    /* file descriptors */
    close(fildes[0]);
    dup2(fildes[1], STDOUT_FILENO);
    dup2(fildes[1], STDERR_FILENO);
    /* new session */
    setsid();
    /* Env */
    setenv("LC_ALL", "C", 1);
    /* execution */
    execvp(cmd, args);
    close(fildes[1]);
    free(pos);
}


static void launch_player() {
    if (! has_module(playlist_current())) return;
    pipe(fildes);
    p_pid = fork();
    switch (p_pid) {
        case 0:
            player_cmd();
            _exit(0);
        case -1:
            perror("fork");
            quit();
        default:
            close(fildes[1]);
            fildes[1] = -1;
            break;
    }
    input_end = FALSE;
}


static void get_datas() {
    char        input[BUFLEN];
    static char datas[BUFLEN];
    static int  dpos = 0;
    Module      *module = get_module();

    bzero(input, BUFLEN);
    if (read(fildes[0], input, BUFLEN) > 0) {
        if (module) {
            int i;
            for (i = 0 ; input[i] ; i++) {
                if ((input[i]!='\n') && (input[i]!='\r') && (input[i]!=7)) {
                    datas[dpos++] = input[i];
                    if (dpos == BUFLEN) {
                        bzero(datas, BUFLEN);
                        dpos = 0;
                        break;
                    }
                } else {
                    datas[dpos] = 0;
                    printlog(2, "MODULE OUTPUT : [%s]\n", datas);
                    get_module()->parse(datas, &frame,
                                &minutes, &seconds, &rmins, &rsecs);
                    dpos = 0;
                    bzero(datas, BUFLEN);
                }
            }
        }
    } else {
        set_command(STOPPED);
        input_end = TRUE;
        unload_module();
        bzero(datas, BUFLEN);
        dpos = 0;
    }
}


void stop_player() {
    if (p_pid != 0) {
        pid_t t;
        close(fildes[0]);
        fildes[0] = -1;
        if (kill(-p_pid, SIGINT) == -1) {
            struct timespec ts = { 1, 0 };
            nanosleep(&ts, NULL);
            if (kill(-p_pid, SIGINT) == -1)
                fprintf(stderr, "Failed to kill player...\n");
        }
        t = waitpid(p_pid, NULL, WNOHANG);
        if ((t != p_pid) && (t != -1))
            waitpid(p_pid, NULL, 0);
        p_pid = 0;
    }
}


/* Actions */

static int do_stop() {
    int cont = FALSE;

    frame   = 0;
    minutes = 0;
    seconds = 0;
    rmins   = 0;
    rsecs   = 0;
    if (input_end == TRUE) {
        printlog(5, "Input end... ");
        if (mode == REPEAT_ONE) {
            printlog(5, "Repeating that title.\n");
            cont = TRUE;
        } else if (mode == REPEAT_ALL) {
            printlog(5, "Repeating all [selected] titles.\n");
            cont = playlist_next(TRUE);
        } else {
            printlog(5, "No repeat, listening next title if it exists.\n");
            cont = playlist_next(FALSE);
            if (! cont) set_status(STOPPED);
        }
    } else {
        printlog(5, "Stop requested !\n");
        set_status(STOPPED);
    }
    printlog(5, "Continuing : %s\n", cont ? "YES" : "NO");
    display_time(minutes, seconds, rmins, rsecs);
    return cont;
}


static void do_next() {
    frame   = 0;
    minutes = 0;
    seconds = 0;
    rmins   = 0;
    rsecs   = 0;
    playlist_next(TRUE);
    display_list();
    display_song();
    display_time(minutes, seconds, rmins, rsecs);
}


static void do_prev() {
    if ((minutes == 0) && (seconds < 3)) {
        playlist_prev(TRUE);
    }
    frame   = 0;
    minutes = 0;
    seconds = 0;
    rmins   = 0;
    rsecs   = 0;
    display_list();
    display_song();
    display_time(minutes, seconds, rmins, rsecs);
}


static int parse_request(Status s) { /* returns TRUE => relaunch player */
    int cont;

    seconds = 60 * minutes + seconds;
    switch (s) {
        case PAUSED:
            cont = FALSE;
            set_status(PAUSED);
            break;
        case STOPPED:
            cont = do_stop();
            break;
        case FFW:
            frame += 150;
            seconds += 10;
            cont = TRUE;
            break;
        case RWD:
            frame -= 150;
            if (frame < 0) frame = 0;
            seconds -= 10;
            if (seconds < 0) seconds = 0;
            cont = TRUE;
            break;
        case NEXT:
            do_next();
            cont = TRUE;
            break;
        case PREV:
            do_prev();
            cont = TRUE;
            break;
        default: /* ERROR */
            cont = FALSE;
            set_status(STOPPED);
    }
    if (cont) {
        set_status(PLAYING);
    }
    return cont;
}


/* get datas from player, displays them and wait for a user command */

static void control_player() {
    int cont = TRUE;

    while (cont) {
        Status s = NOTHING;

        if (is_playable(playlist_current())) {
            set_command(s);
            set_status(PLAYING);
            display_song();
            display_list();
            launch_player();
            while (! request_change(&s)) {
                get_datas();
                display_time(minutes, seconds, rmins, rsecs);
            }
            stop_player();
        } else {
            s = NEXT;
        }
        cont = parse_request(s);
    }
}


/* Main control */

void thread_control() {
    pthread_t tid = pthread_self();

    pthread_detach(tid);
    status  = STOPPED;
    mode    = REPEAT_NUL;
    set_command(NOTHING);
    while (1) {
        Status s = wait_request();
        switch (s) {
            case PLAYING:
                control_player();
                break;
            case PAUSED:
                set_status(PAUSED);
                break;
            case STOPPED:
                do_stop();
                set_status(STOPPED);
                break;
            case NEXT:
                do_next();
                set_status(STOPPED);
                break;
            case PREV:
                do_prev();
                set_status(STOPPED);
                break;
            default: break;
        }
    }
}

