/*
  For MPEG header information,
  see http://www.dv.co.yu/mpgscript/mpeghdr.htm

  Licence: LGPL(?)
  author: goto <harpy@lily.freemail.ne.jp>
*/

#include <stdio.h>
#include "mpeg.h"

/* 0: free, -1: bad */
int MPEG_bitrate_V1_L1[] =
  {  0,  32,  64,  96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, -1};
int MPEG_bitrate_V1_L2[] =
  {  0,  32,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, -1};
int MPEG_bitrate_V1_L3[] =
  {  0,  32,  40,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, -1};
int MPEG_bitrate_V2_L1[] =
  {  0,  32,  48,  56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256, -1};
int MPEG_bitrate_V2_L2L3[] =
  {  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160, -1};

/* 0: reserved */
int MPEG_samplerate_V1[]  = {44100, 48000, 32000, 0};
int MPEG_samplerate_V2[]  = {22050, 24000, 16000, 0};
int MPEG_samplerate_V25[] = {11025, 12000,  8000, 0};

int MPEG_getSamples(FILE* fp) {
  unsigned int n;
  unsigned int address;
  unsigned int total_samples;
  int good = 1;

  n = 0;
  address = 0;
  total_samples = 0;
  while (1) {
    int bytes[4];
    unsigned int flags;
    unsigned int f_frame_sync, f_version, f_layer, f_protection;
    unsigned int f_bitrate, f_samplerate, f_padding, f_private;
    unsigned int f_channel;
    int bitrate, samplerate, padding, private_bit;
    unsigned int frame_length, num_samples;

    fseek(fp, address, SEEK_SET);

    bytes[0] = fgetc(fp);
    if (bytes[0] == EOF) break;
    bytes[1] = fgetc(fp);
    if (bytes[1] == EOF) break;
    bytes[2] = fgetc(fp);
    if (bytes[2] == EOF) break;
    bytes[3] = fgetc(fp);
    if (bytes[3] == EOF) break;

    flags = (bytes[0] << 24) & 0xff000000;
    flags |= (bytes[1] << 16) & 0x00ff0000;
    flags |= (bytes[2] << 8) & 0x0000ff00;
    flags |= bytes[3];

    f_frame_sync = flags & MPEG_FRAME_SYNC;
    f_version    = flags & MPEG_VERSION;
    f_layer      = flags & MPEG_LAYER;
    f_protection = flags & MPEG_PROTECTION;
    f_bitrate    = flags & MPEG_BITRATE;
    f_samplerate = flags & MPEG_SAMPLERATE;
    f_padding    = flags & MPEG_PADDING;
    f_private    = flags & MPEG_PRIVATE;
    f_channel    = flags & MPEG_CHANNEL;

    if (f_frame_sync != MPEG_FRAME_SYNC) {
      good = 0;
    }

    if (f_version == MPEG_VERSION_1 && f_layer == MPEG_LAYER_I) {
      bitrate = MPEG_bitrate_V1_L1[f_bitrate >> MPEG_BITRATE_SHIFT];
    } else if (f_version == MPEG_VERSION_1 && f_layer == MPEG_LAYER_II) {
      bitrate = MPEG_bitrate_V1_L2[f_bitrate >> MPEG_BITRATE_SHIFT];
    } else if (f_version == MPEG_VERSION_1 && f_layer == MPEG_LAYER_III) {
      bitrate = MPEG_bitrate_V1_L3[f_bitrate >> MPEG_BITRATE_SHIFT];
    } else if ((f_version == MPEG_VERSION_2 || f_version == MPEG_VERSION_25)
	       && f_layer == MPEG_LAYER_I) {
      bitrate = MPEG_bitrate_V2_L1[f_bitrate >> MPEG_BITRATE_SHIFT];
    } else if ((f_version == MPEG_VERSION_2 || f_version == MPEG_VERSION_25)
	       && (f_layer == MPEG_LAYER_II || f_layer == MPEG_LAYER_III)) {
      bitrate = MPEG_bitrate_V2_L2L3[f_bitrate >> MPEG_BITRATE_SHIFT];
    } else {
      bitrate = -1;  /* bad */
    }

    if (f_version == MPEG_VERSION_1) {
      samplerate = MPEG_samplerate_V1[f_samplerate >> MPEG_SAMPLERATE_SHIFT];
    } else if (f_version == MPEG_VERSION_2) {
      samplerate = MPEG_samplerate_V2[f_samplerate >> MPEG_SAMPLERATE_SHIFT];
    } else if (f_version == MPEG_VERSION_25) {
      samplerate = MPEG_samplerate_V25[f_samplerate >> MPEG_SAMPLERATE_SHIFT];
    } else {
      samplerate = -1;  /* bad */
    }

    padding = f_padding >> MPEG_PADDING_SHIFT;
    private_bit = f_private >> MPEG_PRIVATE_SHIFT;

    /* calculate flame length */
    if (bitrate < 1 || samplerate < 1) {
      frame_length = 0; /* cannot determine */
    } else {
      if (f_layer == MPEG_LAYER_I) {
	frame_length = (12 * bitrate*1000 / samplerate + padding) * 4;
      } else if (f_layer == MPEG_LAYER_II || f_layer == MPEG_LAYER_III) {
	frame_length = 144 * bitrate*1000 / samplerate + padding;
      } else {
	frame_length = 0; /* cannot determine */
      }
    }

    if (f_layer == MPEG_LAYER_I) {
      num_samples = 384;
    } else if (f_layer == MPEG_LAYER_II || f_layer == MPEG_LAYER_III) {
      num_samples = 1152;
    } else {
      num_samples = 0; /* undefined */
    }

    total_samples += num_samples;
    if (f_protection == MPEG_PROTECTION_PROTECTED) {
      address += 2;
    }
    address += frame_length;

    if (!good) break;
  }

  if (good) {
    return total_samples;
  } else {
    return -1;
  }
}
