/*
    Ming, an SWF output library
    Copyright (C) 2001  Opaque Industries - http://www.opaque.net/

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <math.h>

#include "font.h"
#include "method.h"
#include "utf8.h"

#define glyphLength(font,glyph) \
  ((font)->glyphOffset[(glyph)+1] - (font)->glyphOffset[(glyph)])

int completeSWFFont(SWFBlock block)
{
  SWFFont font = (SWFFont)block;
  int size, i;
  int charmap_len;

  SWFFont_resolveTextList(font);

  /* 2 bytes id, 2 bytes flags, 1 byte name length, 2 bytes nGlyphs,
     2*(nGlyphs+1) bytes offsets, nGlyphs bytes character map: */
  /* MMM - 2*nGlyphs bytes for 2-byte charmap */
  if (font->flags & SWF_FONT_WIDECODES) {
    charmap_len = 2 * font->nGlyphs;
  } else {
    charmap_len = font->nGlyphs;
  }
  size = 9 + strlen(font->name) + 2*font->nGlyphs + charmap_len;

  /* get length of each glyph from its output buffer */
  for(i=0; i<font->nGlyphs; ++i)
    size += glyphLength(font, font->codeToGlyph[i]);

  if(size > 65500)
  {
    size += 2*(font->nGlyphs+1);
    font->flags |= SWF_FONT_WIDEOFFSETS;
  }

  return size;
}
void writeSWFFontToMethod(SWFBlock block,
			  SWFByteOutputMethod method, void *data)
{
  SWFFont font = (SWFFont)block;
  int offset, i;
  byte *p, *s;

  methodWriteUInt16(CHARACTERID(font), method, data);

  method(font->flags & SWF_FONT_WIDEOFFSETS ? 8 : 0, data); /* main flags */
  method(0, data);                                          /* more flags */
  method(strlen(font->name), data);

  for(p = font->name; *p != '\0'; ++p)
    method(*p, data);

  methodWriteUInt16(font->nGlyphs, method, data);

  offset = (font->nGlyphs+1)*(font->flags & SWF_FONT_WIDEOFFSETS ? 4 : 2);

  /* write offset table for glyphs */
  for(i=0; i<=font->nGlyphs; ++i)
  {
    if(font->flags & SWF_FONT_WIDEOFFSETS)
      methodWriteUInt32(offset, method, data);
    else
      methodWriteUInt16(offset, method, data);

    if(i < font->nGlyphs)
      offset += glyphLength(font, font->codeToGlyph[i]);
  }

  /* write shape records for glyphs */
  for(i=0; i<font->nGlyphs; ++i)
  {
    p = font->glyphOffset[font->codeToGlyph[i]];
    s = font->glyphOffset[font->codeToGlyph[i]+1];

    SWF_assert(p < s);

    while(p < s)
      method(*(p++), data);
  }

  /* write (dummy) character mapping */
  /*
   * this is rubbish - would need another
   * table to reverse mapping
   */
  if (font->flags & SWF_FONT_WIDECODES) {
    for(i=0; i<font->nGlyphs; ++i)
      methodWriteUInt16(i, method, data);
  } else {
    for(i=0; i<font->nGlyphs; ++i)
      method(i, data);
  }
}

void destroySWFFont(SWFBlock block)
{
  SWFFont font = (SWFFont)block;

  if(font->shapes)
    free(font->shapes);
  if(font->bounds)
    free(font->bounds);
  if(font->name)
    free(font->name);
  if(font->kernTable)
    free(font->kernTable);

  free(font);
}
SWFFont newSWFFont()
{
  SWFFont font = calloc(1, SWFFONT_SIZE);

  CHARACTER(font)->number = ++SWF_gNumCharacters;
  BLOCK(font)->type = SWF_DEFINEFONT2;
  BLOCK(font)->writeBlock = writeSWFFontToMethod;
  BLOCK(font)->complete = completeSWFFont;
  BLOCK(font)->dtor = destroySWFFont;

  memset(&(font->glyphToCode), 0xff, sizeof(unsigned short)*NUM_FONT_CODES);

  return font;
}

byte *SWFFont_findCharacterGlyph(SWFFont font, unsigned short c)
{
  return font->glyphOffset[font->codeTable[c]];
}

void SWFFont_addTextToList(SWFFont font, struct _textRecord *text)
{
  SWFTextList textList = calloc(1, TEXTLIST_SIZE);
  textList->next = NULL;
  textList->text = text;

  if(font->currentList != NULL)
    font->currentList->next = textList;
  else
    font->textList = textList;

  font->currentList = textList;
}

void SWFFont_addCharToTable(SWFFont font, unsigned short c)
{
  if(font->glyphToCode[c]==0xffff) /* MMM - assuming one can't actually use all 65535 */
  {
    font->codeToGlyph[font->nGlyphs] = font->codeTable[c];
    font->glyphToCode[c] = font->nGlyphs;

    ++font->nGlyphs;
  }
}

/* XXX - big confusion between terms here.  CodeTable isn't font->codeTable */
void SWFFont_buildCodeTable(SWFFont font, SWFTextRecord text)
{
  SWFTextRecord textRecord;
  const byte *string;

  textRecord = text;
  while(textRecord != NULL)
  {
    string = textRecord->string;

    if(string != NULL)
    {
      unsigned int mbcode;
      int i = 0;
      while ((mbcode = UTF8GetChar(string, &i)) != 0xffff)
	SWFFont_addCharToTable(font, mbcode);
    }

    textRecord = textRecord->next;
  }
}

/* build code table from text in all proceding Text blocks */
void SWFFont_resolveTextList(SWFFont font)
{
  SWFTextList textList, oldList;

  textList = font->textList;
  while(textList != NULL)
  {
    oldList = textList;
    SWFFont_buildCodeTable(font, textList->text);
    textList = textList->next;
    free(oldList);
  }

  font->textList = NULL;
}

/* return length of given string in whatever units these are we're using */
int SWFFont_getScaledStringWidth(SWFFont font, const unsigned char *string)
{
  int i, j, width = 0, l = strlen(string);
  int glyph, glyph2;

  i=0;
  for (;;)
  {
    unsigned short mbcode;
    int i1 = i;
    mbcode = UTF8GetChar(string, &i);
    if (mbcode == 0xffff) break;

    glyph = font->codeTable[mbcode];

    if(font->advances)
      width += font->advances[glyph];

    /* looking in kernTable */
    /* MMM - kern table support is not done, in multibyte code */
    if (i1<l-1 && font->kernTable)
    {
      glyph2 = font->codeTable[string[i1+1]];

      j=font->kernCount;

      while (--j >= 0)
      {
	if (glyph == font->kernTable[j].code1 &&
	    glyph2 == font->kernTable[j].code2)
	{
	  width += font->kernTable[j].adjustment;
	  break;
	}
      }
    }
  }

  return width;
}

/* get some font metrics */
short SWFFont_getScaledAscent(SWFFont font)
{
  return font->ascent;
}
short SWFFont_getScaledDescent(SWFFont font)
{
  return font->descent;
}
short SWFFont_getScaledLeading(SWFFont font)
{
  return font->leading;
}
