/*
 * Misc module
 * This module is for generic functions.
 * Independent from specific data structure.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <glib.h>
#include <gdk/gdk.h>/* for gdk_mbstowcs */
#include "misc.h"


/**
 * check_filetype:
 * Simple and generic file type check.
 * Input:
 * const char *fname; file name.
 * Output:
 * Return value: FileType;
 **/
FileType
check_filetype(const char *fname)
{
	struct stat sb;

	if (fname == NULL || fname[0] == '\0')
		return OTHERFILE;
	if (stat(fname, &sb) == -1) {
		g_warning("%s: stat in check_filetype()", fname);
		return OTHERFILE;
	}
	if (sb.st_mode & S_IFREG)
		return REGULARFILE;
	else if (sb.st_mode & S_IFDIR)
		return DIRECTORY;
	else
		return OTHERFILE;
}


/**
 * get_num_chars:
 * Retrun the number of characters in the buffer.
 * In SBCS(Single Byte Character Set) environment, I can assume
 * the number of characters is the same as one of bytes.
 * Input:
 * const char *src; buffer. Not nul-terminated.
 * int lenb; buffer length in bytes.
 * int f_sbcs; true if SBCS(Single Byte Character Set).
 * Output:
 * Return value; the number of characters.
 **/
int
get_num_chars(const char *src, int lenb, int f_sbcs)
{
	int num_char;
	GdkWChar *pwcs;
	GdkWChar wcs[BUFSIZ];
	char *pmbs;
	char mbs[BUFSIZ];/* buffer to keep nul-terminated string */
	gboolean buf_allocated = FALSE;

	if (f_sbcs)/* I can assume char==byte, but this is a special case! */
		return lenb;
	
	if (lenb < BUFSIZ) {
		pwcs = wcs;
		pmbs = mbs;
	} else {
		pmbs = g_new(char, lenb+1);
		pwcs = g_new(GdkWChar, lenb+1);
		buf_allocated = TRUE;
	}
	memcpy(pmbs, src, lenb);
	pmbs[lenb] = '\0';
	/*XXX:this would be better, but gdk doesn't provide it.
	  num_char = gdk_mbstowcs(NULL, pbuf, 0);*/
	num_char = gdk_mbstowcs(pwcs, pmbs, lenb+1);
	if (num_char == -1)
		num_char = 0;/* this might be safer. still a problem */
	
	if (buf_allocated == TRUE) {
		g_free(pmbs);
		g_free(pwcs);
	}	
	return num_char;
}

/**
 * calc_number_places:
 * Calculate the number of places on base 10.
 * e.g. 999 => 3, 9999 => 4, 99999 => 5 ...
 **/
int
calc_number_places(int n)
{
	int ret  = 0;

	while (n) {
		n /= 10;
		ret++;
	}
	return ret;
}

/**
 * get_file_name:
 * Get file name from file path.
 **/
char*
get_file_name(char *fpath)
{
	char *ptr;

	if (fpath == NULL) {
		return NULL;
	}
	ptr = strrchr(fpath, '/');
	if (ptr) {
		return ptr + 1;
	} else {
		return fpath;
	}
}

/**
 * get_rel_file_name:
 * Get relative file name from file path and directory name length.
 **/
char*
get_rel_file_name(char *fpath, int dirlen)
{
	if (strlen(fpath) >= dirlen)
		return fpath + dirlen;
	else
		return fpath;
}

/**
 * close_allfd_except_std:
 * close all file descriptors except standard fds(stdin, stdout, stderr).
 * Usually, called by a forked child process.
 * Its main purpose is to close fd for the socket connected to X server.
 **/
void
close_allfd_except_std(void)
{
	int fd_max;
	int fd;

	fd_max = sysconf(_SC_OPEN_MAX);
	for (fd = STDERR_FILENO+1; fd < fd_max; fd++) {
		close(fd);
	}
}
