/* GDC -- D front-end for GCC
   Copyright (C) 2004 David Friedman
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "d-gcc-includes.h"

// from DMD
#include "total.h"
#include "init.h"
#include "symbol.h"
#include "d-lang.h"
#include "d-codegen.h"

static IRState * cur_irs = 0;

// from c-semantics
static void
genrtl_do_pushlevel ()
{
    // We do the emit_line_note in the caller
    clear_last_expr ();
}

// see d-convert.cc
tree 
convert (tree type, tree expr)
{
    Type * target_type = cur_irs->getDType(type);
    Type * expr_type = cur_irs->getDType(type);
    if (target_type && expr_type)
	return cur_irs->convertTo(expr, expr_type, target_type);
    else
	return d_convert_basic(type, expr);
}

elem *
CondExp::toElem(IRState * irs)
{
    return build(COND_EXPR, type->toCtype(), irs->convertForCondition( econd ),
	e1->toElem( irs ), e2->toElem( irs ));
}

#if D_GCC_VER < 34
// Copied from GCC 3.4
#define COMPLEX_FLOAT_TYPE_P(TYPE)	\
  (TREE_CODE (TYPE) == COMPLEX_TYPE	\
   && TREE_CODE (TREE_TYPE (TYPE)) == REAL_TYPE)
#endif

static void
signed_compare_check(tree * e1, tree * e2)
{
    tree t1 = TREE_TYPE( *e1 );
    tree t2 = TREE_TYPE( *e2 );
    if (INTEGRAL_TYPE_P( t1 ) &&
	INTEGRAL_TYPE_P( t2 )) {
	int u1 = TREE_UNSIGNED( t1 );
	int u2 = TREE_UNSIGNED( t2 );

	if (u1 ^ u2) {
	    if (gen.warnSignCompare) {
		warning ("unsigned comparison with signed operand");
	    }
	    if (! u1)
		* e1 = convert( d_unsigned_type( t1 ), * e1 );
	    if (! u2)
		* e2 = convert( d_unsigned_type( t2 ), * e2 );
	}
    }
}

static tree
make_bool_binop(TOK op, tree e1, tree e2, IRState * irs)
{
    bool is_compare = false; // %% should this be true for unordered comparisons?
    tree_code out_code;
    
    switch (op) {
    case TOKidentity: // fall through
    case TOKequal:
	is_compare = true;
	out_code = EQ_EXPR; break;
    case TOKnotidentity: // fall through
    case TOKnotequal:
	is_compare = true;
	out_code = NE_EXPR;
	break;
    case TOKandand:
	// %% check if these d_truthvalue_conversion calls are only needed
	// for gcc3.3.  Test case is regexp.d:trymatch:REwordboundary.
	e1 = d_truthvalue_conversion(e1);
	e2 = d_truthvalue_conversion(e2);	
	out_code = TRUTH_ANDIF_EXPR;
	break;
    case TOKoror:
	e1 = d_truthvalue_conversion(e1);
	e2 = d_truthvalue_conversion(e2);	
	out_code = TRUTH_ORIF_EXPR;
	break;
    default:
	/*
	    // ordering for complex isn't defined, all that is guaranteed is the 'unordered part'
	    case TOKule:
	    case TOKul:
	    case TOKuge:
	    case TOKug:
		*/
	if ( COMPLEX_FLOAT_TYPE_P( TREE_TYPE( e1 )) ) {
	    // GCC doesn't handle these.
	    e1 = irs->maybeMakeTemp(e1);
	    e2 = irs->maybeMakeTemp(e2);
	    switch (op) {
	    case TOKleg:
		return irs->boolOp(TRUTH_ANDIF_EXPR,
		    make_bool_binop(TOKleg, irs->realPart(e1), irs->realPart(e2), irs),
		    make_bool_binop(TOKleg, irs->imagPart(e1), irs->imagPart(e2), irs));
	    case TOKunord:
		return irs->boolOp(TRUTH_ORIF_EXPR,
		    make_bool_binop(TOKunord, irs->realPart(e1), irs->realPart(e2), irs),
		    make_bool_binop(TOKunord, irs->imagPart(e1), irs->imagPart(e2), irs));
	    case TOKlg:
		return irs->boolOp(TRUTH_ANDIF_EXPR,
		    make_bool_binop(TOKleg, e1, e2, irs),
		    make_bool_binop(TOKnotequal, e1, e2, irs));
	    case TOKue:
		return irs->boolOp(TRUTH_ORIF_EXPR,
		    make_bool_binop(TOKunord, e1, e2, irs),
		    make_bool_binop(TOKequal, e1, e2, irs));
	    default:
		{
		    // From cmath2.d: if imaginary parts are equal,
		    // result is comparison of real parts; otherwise, result false
		    //
		    // Does D define an ordering for complex numbers?
		    
		    // make a target-independent _cmplxCmp ?
		    tree it, rt;
		    TOK hard, soft;
		    bool unordered = false;
		    switch (op) {
		    case TOKule:
		    case TOKul:
		    case TOKuge:
		    case TOKug:
			unordered = true;
		    default:
			break;
		    }
		    
		    switch (op) {
		    case TOKule:
		    case TOKle:
			hard = TOKlt;
			soft = TOKle;
			break;
		    case TOKul:
		    case TOKlt:
			hard = soft = TOKlt;
			break;			
		    case TOKuge:
		    case TOKge:
			hard = TOKlt;
			soft = TOKle;
			break;
		    case TOKug:
		    case TOKgt:
			hard = soft = TOKgt;
			break;
		    default:
			assert(0);
		    }

		    it = make_bool_binop(hard, irs->imagPart(e2), irs->imagPart(e1), irs);
		    if (! unordered)
			it = irs->boolOp(TRUTH_ANDIF_EXPR,
			    make_bool_binop(TOKleg, irs->realPart(e2), irs->realPart(e1), irs),
			    it);
		    rt = irs->boolOp(TRUTH_ANDIF_EXPR,
			make_bool_binop(TOKequal, irs->imagPart(e2), irs->imagPart(e1), irs),
			make_bool_binop(soft, irs->realPart(e2), irs->realPart(e1), irs));
		    it = irs->boolOp(TRUTH_ANDIF_EXPR, it, rt);
		    if (unordered)
			it = irs->boolOp(TRUTH_ORIF_EXPR,
			    make_bool_binop(TOKunord, e1, e2, irs),
			    it);
		    return it;
		}
	    }
	}
	// else, normal
	
	switch (op) {
	case TOKlt: out_code = LT_EXPR; is_compare = true; break;
	case TOKgt: out_code = GT_EXPR; is_compare = true; break;
	case TOKle: out_code = LE_EXPR; is_compare = true; break;
	case TOKge: out_code = GE_EXPR; is_compare = true; break;
	case TOKunord: out_code = UNORDERED_EXPR; break;
	case TOKlg:
	    {
		e1 = irs->maybeMakeTemp(e1);
		e2 = irs->maybeMakeTemp(e2);
		return irs->boolOp(TRUTH_ORIF_EXPR,
		    build(LT_EXPR, boolean_type_node, e1, e2),
		    build(GT_EXPR, boolean_type_node, e1, e2));
	    }
	    break;
	default:
	    /* GCC 3.4 (others?) chokes on these unless
	       at least one operand is floating point. */
	    if (FLOAT_TYPE_P( TREE_TYPE( e1 )) &&
		FLOAT_TYPE_P( TREE_TYPE( e2 ))) {
		switch (op) {
		case TOKleg: out_code = ORDERED_EXPR; break;
		case TOKule: out_code = UNLE_EXPR; break;
		case TOKul:  out_code = UNLT_EXPR; break;
		case TOKuge: out_code = UNGE_EXPR; break;
		case TOKug:  out_code = UNGT_EXPR; break;
		case TOKue:  out_code = UNEQ_EXPR; break;
		default:
		    abort();
		}
	    } else {
		switch (op) {
		case TOKleg:
		    // %% is this properly optimized away?
		    return irs->voidCompound(irs->compound(e1,e2),
			convert(boolean_type_node, integer_one_node));
		    break;
		case TOKule: out_code = LE_EXPR; break;
		case TOKul:  out_code = LT_EXPR; break;
		case TOKuge: out_code = GE_EXPR; break;
		case TOKug:  out_code = GT_EXPR; break;
		case TOKue:  out_code = EQ_EXPR; break;
		default:
		    abort();
		}
	    }
	}
    }

    if (is_compare)
	signed_compare_check(& e1, & e2);

    return build(out_code, boolean_type_node, // exp->type->toCtype(),
	e1, e2);
}

static tree
make_bool_binop(BinExp * exp, IRState * irs)
{
    tree t = make_bool_binop(exp->op, exp->e1->toElem(irs), exp->e2->toElem(irs), irs);
    return convert(exp->type->toCtype(), t);
}

elem *
IdentityExp::toElem(IRState* irs)
{
    TY ty1 = e1->type->toBasetype()->ty;

    // Assuming types are the same from typeCombine
    //if (ty1 != e2->type->toBasetype()->ty)
    //abort();

    switch (ty1) {
    case Tsarray:
	return build(op == TOKidentity ? EQ_EXPR : NE_EXPR,
	    type->toCtype(),
	    irs->addressOf(e1->toElem(irs)),
	    irs->addressOf(e2->toElem(irs)));
    case Treference:
    case Tclass:
    case Tarray:
	return make_bool_binop(this, irs);
    default:
	// For operand types other than class objects, static or dynamic
	// arrays, identity is defined as being the same as equality

	// Assumes object == object has been changed to function call
	// ... impl is really the same as the special cales
	return make_bool_binop(this, irs);
    }
}

elem *
EqualExp::toElem(IRState* irs)
{
    Type * base_type_1 = e1->type->toBasetype();
    TY base_ty_1 = base_type_1->ty;
    TY base_ty_2 = e2->type->toBasetype()->ty;

    if ( (base_ty_1 == Tsarray || base_ty_1 == Tarray ||
	     base_ty_2 == Tsarray || base_ty_2 == Tarray) ) {

	Type * elem_type = base_type_1->next->toBasetype();
	
	// _adCmp compares each element.  If bitwise comparison is ok,
	// use memcmp.
	
	if (elem_type->isfloating() || elem_type->isClassHandle()) {
	    tree args[3] = {
		irs->rawArray(e1),
		irs->rawArray(e2),
		irs->typeinfoReference(elem_type) };
	    tree result = irs->libCall(LIBCALL_ADEQ, 3, args);
	    if (op == TOKnotequal)
		result = build1(TRUTH_NOT_EXPR, type->toCtype(), result);
	    return result;
	} else if (base_ty_1 == Tsarray && base_ty_2 == Tsarray) {
	    // assuming sizes are equal
	    // shouldn't need to check for Tbit
	    return make_bool_binop(this, irs);
	} else if (elem_type->ty != Tbit) {
	    Type * elem_type = e1->type->toBasetype()->next->toBasetype();
	    tree len_expr[2];
	    tree data_expr[2];

	    for (int i = 0; i < 2; i++) {
		Expression * e = i == 0 ? e1 : e2;
		TY e_base_ty = i == 0 ? base_ty_1 : base_ty_2;

		if ( e_base_ty == Tarray ) {
		    tree array_expr = irs->maybeMakeTemp( e->toElem(irs) );
		
		    data_expr[i] = irs->darrayPtrRef( array_expr );
		    len_expr[i]  = irs->darrayLenRef( array_expr ); // may be used twice -- should be okay
		} else {
		    data_expr[i] = irs->addressOf( e->toElem(irs) );
		    len_expr[i]  = ((TypeSArray *) e->type->toBasetype())->dim->toElem(irs);
		}
	    }

	    tree t_memcmp = built_in_decls[BUILT_IN_MEMCMP];
	    tree result;
	    tree size;

	    size = build(MULT_EXPR, size_type_node,
		convert(size_type_node, len_expr[0]), // should be size_type already, though
		size_int(elem_type->size()));
	    size = fold( size );

	    result = irs->buildCall( TREE_TYPE(TREE_TYPE( t_memcmp )),
		irs->addressOf( t_memcmp ),
		tree_cons( NULL_TREE, data_expr[0],
		    tree_cons( NULL_TREE, data_expr[1],
			tree_cons( NULL_TREE, size, NULL_TREE ))));
	    
	    result = irs->boolOp(op == TOKequal ? TRUTH_ANDIF_EXPR : TRUTH_ORIF_EXPR,
		irs->boolOp(op == TOKequal ? EQ_EXPR : NE_EXPR, len_expr[0], len_expr[1]),
		irs->boolOp(op == TOKequal ? EQ_EXPR : NE_EXPR, result,	integer_zero_node));

	    return convert(type->toCtype(), result);
	} else {
	    Array args;
	    args.push(e1);
	    args.push(e2);
	    tree result = irs->libCall(LIBCALL_ADEQBIT, & args);
	    if (op == TOKnotequal)
		result = build1(TRUTH_NOT_EXPR, type->toCtype(), result);
	    return result;
	}
    } else {
	// Taarray case not defined in spec, probably a library call  %% really should assert again tgus
	return make_bool_binop(this, irs);
    }
}

elem *
InExp::toElem(IRState * irs)
{
    Type * e2_base_type = e2->type->toBasetype();
    assert( e2_base_type->ty == Taarray );
    
    tree args[3];
    Type * key_type = ((TypeAArray *) e2_base_type)->key->toBasetype();
    args[0] = irs->rawArray(e2);
    args[1] = irs->typeinfoReference(key_type);
    args[2] = irs->convertTo( e1, key_type ); // %% vararg issues
    return d_convert_basic(type->toCtype(), irs->libCall(LIBCALL_AAIN, 3, args));
}

elem *
CmpExp::toElem(IRState* irs)
{
    Type * base_type_1 = e1->type->toBasetype();
    TY base_ty_1 = base_type_1->ty;
    TY base_ty_2 = e2->type->toBasetype()->ty;

    if ( (base_ty_1 == Tsarray || base_ty_1 == Tarray ||
	     base_ty_2 == Tsarray || base_ty_2 == Tarray) ) {

	Type * elem_type = base_type_1->next->toBasetype();
	tree args[3];
	unsigned n_args = 2;
	LibCall lib_call;

	args[0] = irs->rawArray(e1);
	args[1] = irs->rawArray(e2);
	
	switch (elem_type->ty) {
	case Tvoid:
	case Tchar:
	case Tuns8:
	    lib_call = LIBCALL_ADCMPCHAR;
	    break;
	case Tbit:
	    lib_call = LIBCALL_ADCMPBIT;
	    break;
	default:
	    {
		args[2] = irs->typeinfoReference(elem_type);
		n_args = 3;
	    }
	    lib_call = LIBCALL_ADCMP;
	    break;
	}

	tree result = irs->libCall(lib_call, n_args, args);
	enum tree_code out_code;

	// %% For float element types, warn that NaN is not taken into account ?

	switch (this->op) {
	case TOKlt: out_code = LT_EXPR; break;
	case TOKgt: out_code = GT_EXPR; break;
	case TOKle: out_code = LE_EXPR; break;
	case TOKge: out_code = GE_EXPR; break;
	    
	case TOKlg: out_code = NE_EXPR; break;
	case TOKunord: 
	case TOKleg:
	    // %% Could do a check for side effects and drop the unused condition
	    return build(COMPOUND_EXPR, boolean_type_node,
		result, 
		d_truthvalue_conversion( this->op == TOKunord ? integer_zero_node : integer_one_node ));
	case TOKule: out_code = LE_EXPR; break;
	case TOKul:  out_code = LT_EXPR; break;
	case TOKuge: out_code = GE_EXPR; break;
	case TOKug:  out_code = GT_EXPR; break;
	case TOKue:  out_code = EQ_EXPR; break;	    
	    break;
	default:
	    abort();
	    return 0;
	}

	result = build(out_code, boolean_type_node, result, integer_zero_node);
	return convert(type->toCtype(), result);
    } else {
	return make_bool_binop(this, irs);
    }
}

tree
make_math_op(TOK op, tree e1, Type * e1_type, tree e2, Type * e2_type, Type * exp_type, IRState * irs)
{
    // Integral promotions have already been done in the front end
    tree_code out_code;

    // %% faster: check if result is complex
    if (( ( e1_type->isreal() && e2_type->isimaginary() ) ||
	  ( e1_type->isimaginary() && e2_type->isreal() )) &&
	(op == TOKadd || op == TOKmin )) {
	// %%TODO: need to check size/modes
	tree e2_adj;
	tree e_real, e_imag;

	if (op == TOKadd) {
	    e2_adj = e2;
	} else {
	    e2_adj = build1(NEGATE_EXPR, TREE_TYPE(e2), e2);
	}
	
	if (e1_type->isreal()) {
	    e_real = e1;
	    e_imag = e2_adj;
	} else {
	    e_real = e2_adj;
	    e_imag = e1;
	}

	return build(COMPLEX_EXPR, exp_type->toCtype(), e_real, e_imag);
	
    } else {
	switch (op) {
	case TOKadd: out_code = PLUS_EXPR; break;
	case TOKmin: out_code = MINUS_EXPR; break;
	case TOKmul: out_code = MULT_EXPR; break;
	case TOKxor: out_code = BIT_XOR_EXPR; break;
	case TOKor:  out_code = BIT_IOR_EXPR; break;
	case TOKand: out_code = BIT_AND_EXPR; break;
	case TOKshl: out_code = LSHIFT_EXPR; break;
	case TOKushr: // drop through
	case TOKshr: out_code = RSHIFT_EXPR; break;
	case TOKmod:
	    if (e1_type->isintegral())
		out_code = TRUNC_MOD_EXPR;
	    else {
		return irs->floatMod(e1, e2, e1_type);
	    }
	    break;
	case TOKdiv:
	    if (e1_type->isintegral())
		out_code = TRUNC_DIV_EXPR;
	    else {
		out_code = RDIV_EXPR;
	    }
	    break;
	default:
	    abort();
	}
    }

    bool is_unsigned = e1_type->isunsigned() || e2_type->isunsigned()
	|| op == TOKushr;
    if (exp_type->isintegral() &&
	( exp_type->isunsigned() != 0 ) != is_unsigned) {
	tree e_new_type_1 = is_unsigned ?
	    d_unsigned_type(exp_type->toCtype()) :
	    d_signed_type(exp_type->toCtype());
	tree t = build(out_code, e_new_type_1, e1, e2);
	return convert(exp_type->toCtype(), t);
    } else {
	return build(out_code, exp_type->toCtype(), e1, e2);
    }
}

tree
make_math_op(BinExp * exp, IRState * irs)
{
    return make_math_op(exp->op,
	exp->e1->toElem(irs), exp->e1->type,
	exp->e2->toElem(irs), exp->e2->type,
	exp->type, irs);
}


elem *
AndAndExp::toElem(IRState * irs) { return make_bool_binop(this, irs); }
elem *
OrOrExp::toElem(IRState * irs) { return make_bool_binop(this, irs); }
elem *
XorExp::toElem(IRState * irs) { return make_math_op(this, irs); }
elem *
OrExp::toElem(IRState * irs) { return make_math_op(this, irs); }
elem *
AndExp::toElem(IRState * irs) { return make_math_op(this, irs); }
elem *
UshrExp::toElem(IRState* irs) { return make_math_op(this, irs); }
elem *
ShrExp::toElem(IRState * irs) { return make_math_op(this, irs); }
elem *
ShlExp::toElem(IRState * irs) { return make_math_op(this, irs); }

elem *
ModExp::toElem(IRState * irs)
{
    return make_math_op(this, irs);
}
elem *
DivExp::toElem(IRState * irs)
{
    return make_math_op(this, irs);
}
elem *
MulExp::toElem(IRState * irs) 
{
    return make_math_op(this, irs);
}

elem *
CatExp::toElem(IRState * irs)
{
    // %% assumes both operands are cast to dynamic arrays
    // %% flatten and use arraycatn?
    /*
    Array args;
    args.push(e1->type->dotExp( (Scope *) 0, e1, Id::dup ));
    args.push(e2->type->dotExp( (Scope *) 0, e2, Id::dup ));
    args.push(new Integer(loc, e1->type->next->size(), Type::tuns32));
    return irs->libCall(LIBCALL_ARRAYCAT, & args);
    */

    // Think this is more effecient.. (especially if the concats are flattened)
    // Issue is "Concatenation  always  creates  a copy of its operands, even if one of
    // the operands is a 0 length array"  which _d_arraycat does not do
    tree args[4];
    args[0] = irs->integerConstant(e1->type->next->size(), Type::tuns32); // new IntegerExp(loc, e1->type->next->size(), Type::tuns32);
    args[1] = irs->integerConstant(2, Type::tuns32); // new IntegerExp(loc, 2, Type::tuns32);
    args[2] = irs->rawArray(e1);
    args[3] = irs->rawArray(e2);
    return irs->libCall(LIBCALL_ARRAYCATN, 4, args, type->toCtype());
}

elem *
MinExp::toElem(IRState* irs)
{
    // The front end has already taken care of pointer-int and pointer-pointer
    return make_math_op(this, irs);
}

elem *
AddExp::toElem(IRState* irs)
{
    if ((e1->type->ty == Tarray || e1->type->ty == Tsarray) &&
	(e2->type->ty == Tarray || e2->type->ty == Tsarray) ) {
	::error("Don't know what to do with (array + array)n");
	return error_mark_node;
    } else
	// The front end has already taken care of (pointer + integer)
	return make_math_op(this, irs);
}

class BitArrayAssign
{
    IRState * irs;
    ArrayScope aryscp;
    
    tree word_ref;    // reference to something of type bitConfig.elemType
    tree select_mask;
    tree left_shift;
    tree lhs;
    tree src_exp;
public:
    BitArrayAssign(IndexExp * exp, IRState * irs) :
	aryscp(irs, exp->lengthVar, exp->loc)
    {
	this->irs = irs;

	word_ref = irs->arrayElemRef(exp, & aryscp,
	    & select_mask, (tree *) 0, & left_shift);
	
    }
    // Return the left-hand side for b[x] ?= v
    // Call only once.
    tree getLhs() {
	word_ref = stabilize_reference(word_ref);
	select_mask = irs->maybeMakeTemp( select_mask );
	lhs = d_truthvalue_conversion( build(BIT_AND_EXPR,
				   TREE_TYPE(word_ref), word_ref, select_mask ));
	return lhs;
    }
    void setSrc(tree src_exp) {
	this->src_exp = src_exp;
    }
    tree finish() {
	tree truth_src = src_exp;
	// %% (need this or crash at varasm.c:3172; not sure why)
	// the test may not be broad enough
	if ( ! INTEGRAL_TYPE_P(TREE_TYPE(truth_src)) )
	    truth_src = convert(integer_type_node, truth_src);
	truth_src = d_truthvalue_conversion(truth_src);
	truth_src = irs->maybeMakeTemp( truth_src );
	
	tree word_src = build(LSHIFT_EXPR, bitConfig.elemType->toCtype(),
	    truth_src, left_shift);

	tree tgt_masked = word_ref;
	tgt_masked = build(BIT_AND_EXPR, TREE_TYPE(tgt_masked),
	    tgt_masked, fold(build1(BIT_NOT_EXPR,TREE_TYPE(select_mask),select_mask)));
	
	tree a = build(MODIFY_EXPR, void_type_node, word_ref,
	    build(BIT_IOR_EXPR, TREE_TYPE(tgt_masked), tgt_masked, word_src));
	//return build(COMPOUND_EXPR, TREE_TYPE(truth_src), a, truth_src);
	return aryscp.finish( build(COMPOUND_EXPR, TREE_TYPE(truth_src), a, truth_src) );
    }
};

tree chain_cvt(tree t, Type * typ, Array & casts, IRState * irs)
{
    for (int i = casts.dim - 1; i >= 0; i--) {
	t = irs->convertTo(t, typ, (Type *) casts.data[i]);
	typ = (Type *) casts.data[i];
    }
    return t;
}

tree make_assign_math_op(BinExp * exp, IRState * irs)
{
    Expression * e1_to_use;
    Type * lhs_type = 0;
    tree result;
    TOK out_code;
    Array lhs_casts; // no more than two casts?

    switch (exp->op) {
    case TOKaddass:  out_code = TOKadd; break;
    case TOKminass:  out_code = TOKmin; break;
    case TOKmulass:  out_code = TOKmul; break;
    case TOKxorass:  out_code = TOKxor; break;
    case TOKorass:   out_code = TOKor; break;
    case TOKandass:  out_code = TOKand; break;
    case TOKshlass:  out_code = TOKshl; break;
    case TOKushrass: out_code = TOKushr; break;
    case TOKshrass:  out_code = TOKshr; break;
    case TOKmodass:  out_code = TOKmod; break;
    case TOKdivass:  out_code = TOKdiv; break;
    default:
	abort();
    }

    e1_to_use = exp->e1;
    lhs_type = e1_to_use->type;
    while (e1_to_use->op == TOKcast) {
	CastExp * cast_exp = (CastExp *) e1_to_use;
	assert(cast_exp->type->equals(cast_exp->to)); // %% check, basetype?
	lhs_casts.push(cast_exp->to);
	e1_to_use = cast_exp->e1;
    }

    if ( irs->isBitArrayAccess( e1_to_use )) {
	BitArrayAssign bas( (IndexExp *) e1_to_use, irs );
	
	tree lhs = chain_cvt(bas.getLhs(), Type::tbit, lhs_casts, irs);

	Type * src_type = lhs_type;
	tree src = make_math_op(out_code, lhs, lhs_type,
	    exp->e2->toElem(irs), exp->e2->type,
	    src_type, irs);

	bas.setSrc( src );

	result = bas.finish();
	
    } else {
	tree tgt = stabilize_reference( e1_to_use->toElem( irs ));
	tree lhs = chain_cvt(tgt, e1_to_use->type, lhs_casts, irs);

	/* %%TODO: is lhs_okay safe to use for the whole expression? */
	Type * src_type	= lhs_type;
	tree src = make_math_op(out_code, lhs, lhs_type,
	    exp->e2->toElem(irs), exp->e2->type,
	    src_type, irs);
	result = build(MODIFY_EXPR, exp->type->toCtype(),
	    tgt, irs->convertForAssignment(src, src_type, e1_to_use->type));
    }
    return result;
}

elem *
XorAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
OrAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
AndAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
UshrAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
ShrAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
ShlAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
ModAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
DivAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
MulAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }

elem *
CatAssignExp::toElem(IRState * irs) {
    Expression * e;
    tree args[3];
    Type * elem_type = e1->type->toBasetype()->next->toBasetype();
    //Expression * size_exp = new IntegerExp(loc, elem_type->size(), Type::tuns32);
    tree size_exp = irs->integerConstant(elem_type->size(), Type::tuns32);
    LibCall lib_call;

    if (elem_type->ty == Tbit) {
	error("sorry; can't append to array of bit");
	return error_mark_node;
    }

    e = new AddrExp(loc, e1);
    e->type = e1->type->pointerTo();
    if (elem_type->equals(e2->type->toBasetype())) {
	// append an element
	// %% change to setlength and inline assign last element to avoid varargs
	args[0] = e->toElem(irs);
	args[1] = size_exp;
	args[2] = e2->toElem(irs);
	lib_call = LIBCALL_ARRAYAPPENDC;
    } else {
	args[0] = e->toElem(irs);
	args[1] = irs->rawArray(e2);
	args[2] = size_exp;
	lib_call = LIBCALL_ARRAYAPPEND;
    }
    return irs->libCall(lib_call, 3, args, type->toCtype());
}

elem *
MinAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }
elem *
AddAssignExp::toElem(IRState * irs) { return make_assign_math_op(this, irs); }

elem *
AssignExp::toElem(IRState* irs) {
    // First, handle special assignment semantics
    if (e1->op == TOKarraylength) {
	// Assignment to an array's length property; resize the array
	
	Expression * array_exp_1 = ((ArrayLengthExp *) e1)->e1;
	Expression * array_exp = new AddrExp(loc, array_exp_1);
	array_exp->type = array_exp_1->type->pointerTo();
	    
	Type * elem_type = array_exp_1->type->toBasetype()->next;
	Array args;
	tree result;

	if (! elem_type->isbit()) {
	    args.setDim(3);
	    args.data[0] = e2;
	    args.data[1] = new IntegerExp(loc, elem_type->size(), Type::tuns32);
	    args.data[2] = array_exp;
	    result = irs->libCall(LIBCALL_ARRAYSETLENGTH, & args,
		array_exp_1->type->toCtype());
	} else {
	    args.setDim(2);
	    args.data[0] = e2;
	    args.data[1] = array_exp;
	    result = irs->libCall(LIBCALL_ARRAYSETLENGTH_B, & args);
	}
	// the libcall actually returns a whole new dynamic array.., but also
	// modifies the array..
	// result = build(MODIFY_EXPR, e1->type->toCtype(), e1, result);
	// the result of the expression, is still the length...
	result = irs->darrayLenRef( result );

	return result;
    } else if (irs->isBitArrayAccess( e1 )) {
	BitArrayAssign bas( (IndexExp *) e1, irs);
	bas.setSrc(e2->toElem( irs ));
	return bas.finish();
    } else if (e1->op == TOKslice) {
	Type * elem_type = e1->type->toBasetype()->next->toBasetype();
	if (elem_type->equals( e2->type->toBasetype() )) {
	    // Set a range to one value
	    
	    // %% This is used for initing on-stack static arrays..
	    // should optimize with memset if possible
	    // %% vararg issues
	    if (elem_type->ty != Tbit) {
		tree dyn_array_exp = irs->maybeMakeTemp( e1->toElem(irs) );
		
		return build(COMPOUND_EXPR, TREE_TYPE(dyn_array_exp),
		    irs->arraySet( irs->darrayPtrRef(dyn_array_exp), e2->toElem(irs),
			irs->darrayLenRef(dyn_array_exp) ),
		    dyn_array_exp);
	    } else {
		tree args[4];
		SliceExp * e1_slice = (SliceExp *) e1;
		ArrayScope aryscp(irs, e1_slice->lengthVar, loc);
		tree ary_t;

		args[1] = e1_slice->lwr ?
		    e1_slice->lwr->toElem(irs) : size_int(0);
		args[2] = e1_slice->upr ?
		    e1_slice->upr->toElem(irs) : NULL_TREE;
		    
		if (e1_slice->e1->type->toBasetype()->ty == Tpointer) {
		    ary_t = irs->darrayVal(Type::tbit->arrayOf()->toCtype(),
			TYPE_MAX_VALUE(Type::tsize_t->toCtype()),
			e1_slice->e1->toElem(irs));
		    assert(args[2]);
		} else {
		    ary_t = irs->rawArray(e1_slice->e1);
		    if ( ! args[2] ) {
			ary_t = irs->maybeMakeTemp(ary_t);
			args[2] = irs->arrayLength(ary_t, e1_slice->e1->type);
		    }
		}
		
		ary_t = aryscp.setArrayExp(ary_t, e1_slice->e1->type);

		args[0] = ary_t;
		args[3] = irs->convertForAssignment(e2, Type::tbit);
		return aryscp.finish( irs->libCall(LIBCALL_ARRAYSETBIT, 4, args) );
	    }
	} else {
	    // copy a range
	    if (elem_type->ty != Tbit) {
		if (global.params.useArrayBounds) {
		    tree args[3] = {
			irs->integerConstant(elem_type->size(), Type::tuns32),
			irs->rawArray(e2),
			irs->rawArray(e1) };
		    return irs->libCall(LIBCALL_ARRAYCOPY, 3, args, type->toCtype());
		} else {
		    tree array[2] = {
			irs->maybeMakeTemp( irs->rawArray(e1) ),
			irs->rawArray(e2) };
		    tree t_memcpy = built_in_decls[BUILT_IN_MEMCPY];
		    tree result;
		    tree size;

		    size = build(MULT_EXPR, size_type_node,
			convert(size_type_node, irs->darrayLenRef(array[0])),
			size_int(elem_type->size()));
		    size = fold( size );

		    result = irs->buildCall( TREE_TYPE(TREE_TYPE( t_memcpy )),
			irs->addressOf( t_memcpy ),
			tree_cons( NULL_TREE, irs->darrayPtrRef(array[0]),
			    tree_cons( NULL_TREE, irs->darrayPtrRef(array[1]),
				tree_cons( NULL_TREE, size, NULL_TREE))));

		    return irs->compound( result, array[0], type->toCtype() );
		    /*
		    result = irs->buildCall( TREE_TYPE(TREE_TYPE( t_memcpy )),
			irs->addressOf( t_memcpy ),
			List( irs->darrayPtrRef(array[0]) ).
			    ( irs->darrayPtrRef(array[1]) ).
			    (  size ).head());
		    */
		}
		
	    } else {
		tree args[2] = {
		    irs->rawArray(e2),
		    irs->rawArray(e1) };
		return irs->libCall(LIBCALL_ARRAYCOPYBIT, 2, args);
	    }
	}
    } else {

	tree lhs = e1->toElem(irs);
	tree result = build(MODIFY_EXPR, type->toCtype(),
	    lhs, irs->convertForAssignment(e2, e1->type));

	return result;
    }
}

elem *
PostDecExp::toElem(IRState* irs)
{
    tree result = build(POSTDECREMENT_EXPR, type->toCtype(),
	e1->toElem(irs), e2->toElem(irs));
    TREE_SIDE_EFFECTS(result) = 1;
    return result;
}

elem *
PostIncExp::toElem(IRState* irs)
{
    tree result = build(POSTINCREMENT_EXPR, type->toCtype(),
	e1->toElem(irs), e2->toElem(irs));
    TREE_SIDE_EFFECTS(result) = 1;
    return result;
}


elem *
IndexExp::toElem(IRState* irs)
{
    Type * array_type = e1->type->toBasetype();
    
    if (array_type->ty != Taarray) {
	ArrayScope aryscp(irs, lengthVar, loc);
	return aryscp.finish( irs->arrayElemRef( this, & aryscp ) );
    } else {
	Type * key_type = ((TypeAArray *) array_type)->key->toBasetype();
	tree args[4];
	args[0] = irs->addressOf( e1->toElem(irs) );
	args[1] = irs->typeinfoReference(key_type);
	args[2] = irs->integerConstant( array_type->next->size(), Type::tuns32 );
	args[3] = irs->convertTo( e2, key_type ); // %% vararg issues
	return build1(INDIRECT_REF, type->toCtype(),
	    irs->libCall(LIBCALL_AAGET, 4, args, type->pointerTo()->toCtype()));
    }
}

elem *
CommaExp::toElem(IRState * irs)
{
    return build(COMPOUND_EXPR, type->toCtype(), // %% check if ok to use the expression type
	e1->toElem( irs ), e2->toElem( irs ));
}

elem *
ArrayLengthExp::toElem(IRState * irs)
{
    if (e1->type->toBasetype()->ty == Tarray) {
	return irs->darrayLenRef(e1->toElem(irs));
    } else {
	// Tsarray case seems to be handled by front-end
	
	error("unexpected type for array length: %s", type->toChars());
	return error_mark_node;
    }
}

elem *
SliceExp::toElem(IRState * irs)
{
    // front end casts the result to a dynamic array; this function assumes this
    assert(type->toBasetype()->ty == Tarray);
    
    Type * orig_array_type = e1->type->toBasetype();

    tree orig_array_expr = NULL;
    tree orig_pointer_expr;
    tree final_len_expr = NULL;
    tree final_ptr_expr = NULL;
    tree array_len_expr = NULL;
    tree lwr_tree = NULL;

    ArrayScope aryscp(irs, lengthVar, loc);

    orig_array_expr = aryscp.setArrayExp( e1->toElem(irs), e1->type );
    orig_array_expr = irs->maybeMakeTemp( orig_array_expr );
    // specs don't say bounds if are checked for error or clipped to current size

    // This line allow non-Tarray types
    orig_pointer_expr = irs->convertTo(orig_array_expr, orig_array_type, 
	orig_array_type->next->pointerTo());
    
    final_ptr_expr = orig_pointer_expr;

    // orig_array_expr is already a save_expr if necessary, so
    // we don't make array_len_expr a save_expr which is, at most,
    // a COMPONENT_REF on top of orig_array_expr.
    if ( orig_array_type->ty == Tarray ) {
	array_len_expr = irs->darrayLenRef( orig_array_expr );
    } else if ( orig_array_type->ty == Tsarray ) {
	array_len_expr  = ((TypeSArray *) orig_array_type)->dim->toElem(irs);
    } else {
	// array_len_expr == NULL indicates no bounds check is possible
    }

    if (lwr) {
	lwr_tree = lwr->toElem(irs);
	if (integer_zerop(lwr_tree))
	    lwr_tree = NULL_TREE;
    }
    if (upr) {
	final_len_expr = upr->toElem(irs);
	if (global.params.useArrayBounds && array_len_expr) {
	    final_len_expr = irs->maybeMakeTemp(final_len_expr);
	    final_len_expr = irs->checkedIndex(loc, final_len_expr, array_len_expr, true);
	}
	if (lwr_tree) {
	    lwr_tree = irs->maybeMakeTemp(lwr_tree);
	    // %% type
	    final_len_expr = build(MINUS_EXPR, TREE_TYPE(final_len_expr), final_len_expr, lwr_tree);
	}
    } else {
	// If this is the case, than there is no lower bound specified and
	// there is no need to subtract.
	switch (orig_array_type->ty) {
	case Tarray:
	    final_len_expr = irs->darrayLenRef(orig_array_expr);
	    break;
	case Tsarray:
	    final_len_expr = ((TypeSArray *) orig_array_type)->dim->toElem(irs);
	    break;
	default:
	    ::error("Attempt to take length of something that was not an array");
	    return error_mark_node;
	}
    }
    if (lwr_tree) {
	if (global.params.useArrayBounds && array_len_expr) { // %% && ! is zero
	    lwr_tree = irs->maybeMakeTemp(lwr_tree);
	    lwr_tree = irs->checkedIndex(loc, lwr_tree, array_len_expr, true); // lower bound can equal length
	}
	
	if (orig_array_type->next->isbit()) {
	    assert( tree_low_cst( TYPE_SIZE_UNIT( TREE_TYPE( TREE_TYPE( final_ptr_expr ))), 1) == 1 );
	    lwr_tree = irs->bitAlignCheck(loc, lwr_tree);
	    // Need to convert to unsigned for an unsigned shift.
	    if ( ! TREE_UNSIGNED( TREE_TYPE( lwr_tree ))) {
		lwr_tree = convert( d_unsigned_type( TREE_TYPE( lwr_tree )), lwr_tree );
	    }
	    // !! note: lwr_tree can't be used as the logical index after this point
	    lwr_tree = build(RSHIFT_EXPR, TREE_TYPE(lwr_tree), lwr_tree,
		irs->integerConstant(3, Type::tuns32));
	}

	final_ptr_expr = irs->pointerIntSum(irs->pvoidOkay(final_ptr_expr), lwr_tree);
	TREE_TYPE( final_ptr_expr ) = TREE_TYPE( orig_pointer_expr );
    }

    return aryscp.finish( irs->darrayVal(type->toCtype(), final_len_expr, final_ptr_expr) );
}

elem *
CastExp::toElem(IRState * irs)
{
    return irs->convertTo(e1, to);
}

elem *
DeleteExp::toElem(IRState* irs)
{
    // Does this look like an associative array delete?
    if (e1->op == TOKindex) {
	Expression * e_array = ((BinExp *) e1)->e1;
	Expression * e_index = ((BinExp *) e1)->e2;
	// Check that the array is actually an associative array
	if (e_array->type->toBasetype()->ty == Taarray) {
	    // %% convert to void or cast void... and elesewhere
	    tree args[3];
	    Type * key_type = ((TypeAArray *) e_array->type->toBasetype())->key->toBasetype();
	    args[0] = e_array->toElem(irs); // %% no rawArray since the type should already be correct
	    args[1] = irs->typeinfoReference(key_type);
	    args[2] = irs->convertTo( e_index, key_type ); // %% vararg issues
	    return irs->libCall(LIBCALL_AADEL, 3, args);
	}
    }

    // Otherwise, this is normal delete	
    tree t = irs->addressOf( e1->toElem(irs) );
    LibCall lib_call;

    switch (e1->type->toBasetype()->ty) {
    case Tclass: lib_call = LIBCALL_DELCLASS; break;
    case Tarray: lib_call = LIBCALL_DELARRAY; break;
    case Tpointer: lib_call = LIBCALL_DELMEMORY; break;
    default:
	error("don't know how to delete %s", e1->toChars());
	return error_mark_node;
    }

    return irs->libCall(lib_call, 1, & t);
}

elem * BoolExp::toElem(IRState *)
{
    return op == TOKtrue ? boolean_true_node : boolean_false_node;
}

elem *
NotExp::toElem(IRState * irs)
{
    // %% doc: need to convert to boolean type or this will fail.
    tree t = build1(TRUTH_NOT_EXPR, boolean_type_node,
	d_truthvalue_conversion(e1->toElem( irs )));
    return convert(type->toCtype(), t);
}

elem *
ComExp::toElem(IRState * irs)
{
    return build1(BIT_NOT_EXPR, type->toCtype(), e1->toElem( irs ));
}

elem *
NegExp::toElem(IRState * irs)
{
    // %% GCC B.E. won't optimize (NEGATE_EXPR (INTEGER_CST ..))..
    // %% is type correct?
    return build1(NEGATE_EXPR, type->toCtype(), e1->toElem(irs));
}

elem *
PtrExp::toElem(IRState * irs)
{
    /* add this from c-typeck.c:
	  TREE_READONLY (ref) = TYPE_READONLY (t);
	  TREE_SIDE_EFFECTS (ref)
	    = TYPE_VOLATILE (t) || TREE_SIDE_EFFECTS (pointer);
	  TREE_THIS_VOLATILE (ref) = TYPE_VOLATILE (t);
    */

    // Produce better code by converting *(#rec + n) to
    // COMPONENT_REFERENCE.
    if (e1->op == TOKadd) {
	BinExp * add_exp = (BinExp *) e1;
	if (add_exp->e1->op == TOKaddress &&
	    add_exp->e2->isConst() && add_exp->e2->type->isintegral()) {
	    Expression * rec_exp = ((AddrExp*) add_exp->e1)->e1;
	    Type * rec_type = rec_exp->type->toBasetype();
	    if (rec_type->ty == Tstruct) {
		unsigned the_offset = add_exp->e2->toInteger();
		StructDeclaration * sd = ((TypeStruct *)rec_type)->sym;
		for (unsigned i = 0; i < sd->fields.dim; i++) {
		    VarDeclaration * field = (VarDeclaration *) sd->fields.data[i];
		    if (field->offset == the_offset &&
			field->type->equals(this->type)) {
			tree rec_tree = rec_exp->toElem(irs);
			if (rec_tree == error_mark_node)
			    return error_mark_node; // backend will ICE otherwise

			return build(COMPONENT_REF, type->toCtype(),
			    rec_tree, field->toSymbol()->Stree);
		    } else if (field->offset > the_offset) {
			break;
		    }
		}
	    }
	}
    }
    
    tree e = build1(INDIRECT_REF, type->toCtype(), e1->toElem(irs));
    if (irs->inVolatile())
	TREE_THIS_VOLATILE( e ) = 1;
    return e;
}

elem *
AddrExp::toElem(IRState * irs)
{
    // %% check for &array[index] -> array + index ?
    
    tree t = irs->addressOf(e1->toElem(irs));    
    TREE_TYPE(t) = type->toCtype(); // %% is this ever needed?
    return t;
}

elem *
CallExp::toElem(IRState* irs)
{
    tree t = irs->call(e1, arguments);
    // Some library calls are defined to return a generic type.
    // this->type is the real type.
    TREE_TYPE(t) = type->toCtype();
    return t;
}

elem *
Expression::toElem(IRState*)
{
    error("abstract Expression::toElem called\n");
    return error_mark_node;
}

elem *
DotTypeExp::toElem(IRState* irs)
{
    // The only case in which this seems to be a valid expression is when
    // it is used to specify a non-virtual call ( SomeClass.func(...) ).
    // This case is handled in IRState::objectInstanceMethod.
    
    error("cannot use \"%s\" as an expression", toChars());

    // Can cause ICEs later; should just exit now.
    return error_mark_node;
}

// The result will probably just be converted to a CONSTRUCTOR for a Tdelegate struct
elem *
DelegateExp::toElem(IRState* irs)
{
    Type * t = e1->type->toBasetype();
    if (t->ty == Tclass || t->ty == Tstruct) {
	// %% Need to see if DotVarExp ever has legitimate
	// <aggregate>.<static method>.  If not, move this test
	// to objectInstanceMethod.
	if (! func->isThis())
	    error("delegates are only for non-static functions");
	return irs->objectInstanceMethod(e1, func, type);
    } else {
	assert(func->isNested());
	return irs->methodCallExpr(irs->functionPointer(func),
	    func->isNested() ? d_null_pointer : e1->toElem(irs), type);
    }
}

elem *
DotVarExp::toElem(IRState * irs)
{
    FuncDeclaration * func_decl;
    VarDeclaration * var_decl;
    Type * obj_basetype = e1->type->toBasetype();
    TY obj_basetype_ty = obj_basetype->ty;
    switch (obj_basetype_ty) {
    case Tpointer:
	if (obj_basetype->next->toBasetype()->ty != Tstruct) {
	    break;
	}
	// drop through
    case Tstruct:
	// drop through
    case Tclass:
	if ( (func_decl = var->isFuncDeclaration()) ) {
	    // if Tstruct, objInstanceMethod will use the address of e1
	    return irs->objectInstanceMethod(e1, func_decl, type);
	} else if ( (var_decl = var->isVarDeclaration()) ) {
	    tree this_tree = e1->toElem(irs);
	    if ( obj_basetype_ty != Tstruct )
		this_tree = build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(this_tree)), this_tree);
	    
	    return build(COMPONENT_REF, type->toCtype(), this_tree, var_decl->toSymbol()->Stree);
	} else {
	    // error below
	}
	break;
    default:
	break;
    }
    ::error("Don't know how to handle %s", toChars());
    return error_mark_node;
}

elem *
AssertExp::toElem(IRState* irs)
{
    // %% todo: Do we call a Tstruct's invariant if
    // e1 is a pointer to the struct?
    if (global.params.useAssert) {
	Type * base_type = e1->type->toBasetype();
	TY ty = base_type->ty;

	if (ty == Tclass) {
	    tree arg = e1->toElem(irs);
	    return irs->libCall(LIBCALL_INVARIANT, 1, & arg);  // this does a null pointer check
	} else {
	    // build: ( (bool) e1  ? (void)0 : _d_assert(...) )
	    //    or: ( e1 != null ? (void)0 : _d_assert(...), e1._invariant() )
	    tree result;
	    tree invc = NULL_TREE;
	    tree e1_t = e1->toElem(irs);

	    if (ty == Tpointer) {
		Type * sub_type = base_type->next->toBasetype();
		if (sub_type->ty == Tstruct) {
		    AggregateDeclaration * agg_decl = ((TypeStruct *) sub_type)->sym;
		    if (agg_decl->inv) {
			Array args;
			e1_t = irs->maybeMakeTemp(e1_t);
			invc = irs->call( irs->functionPointer(agg_decl->inv),
			    e1_t, & args );
		    }
		}
	    }

	    result = build(COND_EXPR, void_type_node,
		irs->convertForCondition( e1_t, e1->type ),
		d_void_zero_node, irs->assertCall(loc));
	    if (invc)
		result = build(COMPOUND_EXPR, void_type_node, result, invc);
	    return result;
	}
    } else
	return d_void_zero_node;
}

elem *
DeclarationExp::toElem(IRState* irs)
{
#ifdef OLD
    VarDeclaration * v = declaration->isVarDeclaration();
    if ( v && ! v->isDataseg() )
	irs->emitLocalVar(v);
    else
	declaration->toObjFile();
#else
    // VarDeclaration::toObjFile was modified to call d_gcc_emit_local_variable
    // if needed.  This assumes irs == cur_irs
    declaration->toObjFile();
#endif
    return d_void_zero_node;
}

void d_gcc_emit_local_variable(VarDeclaration * v)
{
    cur_irs->emitLocalVar(v);
}


// %% check calling this directly?
elem *
FuncExp::toElem(IRState * irs)
{
    fd->toObjFile();

    Type * func_type = type->toBasetype();

    if (func_type->ty == Tpointer)
	func_type = func_type->next;
    
    switch (func_type->toBasetype()->ty) {
    case Tfunction:
	return irs->addressOf(fd);
    case Tdelegate:
	return irs->methodCallExpr(irs->functionPointer(fd), // trampoline
	    convert(ptr_type_node, integer_one_node), type); // fake argument
    default:
	::error("Unexpected FunExp type");
	return error_mark_node;
    }
    
    // If nested, this will be a trampoline...
}

elem *
VarExp::toElem(IRState* irs)
{
    if (var->storage_class & STCfield) {
	error("Need 'this' to access member %s", var->ident->string);
	return error_mark_node;
    }
    
    // For variables that are references (currently only out/inout arguments;
    // objects don't count), evaluating the variable means we want what it refers to.
    tree e = var->toSymbol()->Stree;

    if ( irs->isDeclarationReferenceType(var) ) {
	e = build1(INDIRECT_REF, var->type->toCtype(), e);
	if (irs->inVolatile()) {
	    TREE_THIS_VOLATILE(e) = 1;
	}
    } else {
	if (irs->inVolatile()) {
	    e = irs->addressOf(e);
	    TREE_THIS_VOLATILE(e) = 1;
	    e = build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(e)), e);
	    TREE_THIS_VOLATILE(e) = 1;
	}
    }
    return e;
}

elem *
SymOffExp::toElem(IRState * irs) {
    tree a, b;
    a = irs->addressOf(var);
    b = irs->integerConstant(offset, Type::tsize_t);
    return build(PLUS_EXPR, type->toCtype(), a, b) ;
}

elem *
NewExp::toElem(IRState * irs)
{
    Type * base_type = type->toBasetype();
    Array actual_new_args;
    tree result;

    if (newargs && allocator) {
	actual_new_args.reserve(newargs->dim + 1);
	actual_new_args.push(0); // size filled in below
	actual_new_args.append( newargs );
    } else
	actual_new_args.reserve(3);
    
    switch (base_type->ty) {
    case Tclass:
	{
	    TypeClass * class_type = (TypeClass *) base_type;
	    tree new_call;

	    // Allocation call (custom allocator or _d_newclass)
	    if (allocator) {
		tree rec_type = TREE_TYPE( class_type->toCtype() );
		tree exp_node;
		
		actual_new_args.data[0] = new IntegerExp(loc, class_type->sym->structsize, Type::tuns32);
		new_call = irs->call(allocator, & actual_new_args);
		// %% playing fast and loose with type vs. base_type vs. class_type
		if (1) {
		    // first, save the result of the allocation call
		    new_call = save_expr( new_call );
		    // copy memory...
		    // type->toCtype() is a REFERENCE_TYPE; we want the RECORD_TYPE
		    exp_node = build(MODIFY_EXPR, rec_type, /* %% FIX */
			build1(INDIRECT_REF, rec_type, new_call ),
			irs->aggDeclStaticInit(class_type->sym));
			//class_type->defaultInit()->toElem( irs ));
		    // the constructor needs the reference, tho..
		    // is addr_of bettern than compound (indirect(ptr),ptr) ?
		    new_call = build(COMPOUND_EXPR, TREE_TYPE(new_call), exp_node, new_call);
		    // new_call = irs->addressOf(exp_node);
		} else {
		    exp_node = build(MODIFY_EXPR, rec_type,
			build1(INDIRECT_REF, rec_type, new_call ),
			irs->aggDeclStaticInit(class_type->sym));
		    new_call = build1(ADDR_EXPR, class_type->toCtype(), exp_node);
		}
	    } else {
		tree arg = irs->addressOf( class_type->sym->toSymbol()->Stree );
		new_call = irs->libCall(LIBCALL_NEWCLASS, 1, & arg);
	    }
	    // Constructor call
	    if (member) {
		result = irs->call( irs->functionPointer( member ),
		    new_call, arguments);
		    
	    } else {
		result = new_call;
	    }
	    return result;
	}
    case Tarray:
	{
	    assert( ! allocator );
	    
	    LibCall lib_call;
	    Type * elem_type = type->toBasetype()->next->toBasetype();

	    tree orig_length = ((Expression *) arguments->data[0])->toElem(irs);  // not new_args..
	    tree final_length = orig_length;
	    tree final_size = NULL_TREE;
	    tree final_init = NULL_TREE;
	    unsigned multiple = 1;
	    
	    if (elem_type->ty != Tbit) {
		if (elem_type->isZeroInit()) {
		    lib_call = LIBCALL_NEW;
		} else {
		    
		    while (elem_type->ty == Tsarray) {
			multiple = ((TypeSArray *) elem_type)->dim->toInteger() * multiple;
			elem_type = elem_type->next->toBasetype();
		    }

		    if (multiple != 1) {
			orig_length = irs->maybeMakeTemp(orig_length);
			
			// %%TODO: type conversions...
			final_length = build(MULT_EXPR, size_type_node,
			    orig_length,
			    size_int(multiple));
			final_length = fold( final_length );
		    }
		    
		    final_init = irs->convertForAssignment(elem_type->defaultInit(), elem_type);

		    lib_call = LIBCALL_NEWARRAYI;
		}
		
		final_size = size_int(elem_type->size());
	    } else {
		final_init = irs->integerConstant(0, Type::tbit->toCtype());
		lib_call = LIBCALL_NEWBITARRAY;
	    }

	    // %%TODO: convert to correct argument types... (for 64-bit)
	    
	    tree args[3];
	    tree * p_arg = args;

	    *p_arg++ = final_length;
	    if (final_size)
		*p_arg++ = final_size;
	    if (final_init)
		*p_arg++ = final_init;
	    result = irs->libCall(lib_call, p_arg - args, args, type->toCtype()); // %% forced result type ok?

	    if (multiple != 1) {
		// need to set the length the original length value
		result = irs->darrayVal(base_type->toCtype(), orig_length,
		    irs->darrayPtrRef(result));
	    }

	    return result;
	}
	break;
    case Tpointer:
	{
	    Type * object_type = base_type->next->toBasetype();
	    tree new_call;
	    if (allocator) {
		actual_new_args.data[0] = new IntegerExp(loc, object_type->size(), Type::tuns32);
		new_call = irs->call(allocator, & actual_new_args);
	    } else {
		// there doesn't seem to be any simple allocation call...
		// just need a wrapper around gc.alloc
		actual_new_args.push( new IntegerExp( 1 ));
		actual_new_args.push( new IntegerExp(loc, object_type->size(), Type::tuns32 ));
		new_call = irs->libCall(LIBCALL_NEW, & actual_new_args);
		new_call = irs->darrayPtrRef(new_call);
	    }
	    new_call = build1(NOP_EXPR, base_type->toCtype(), new_call);
	    if ( ! object_type->isZeroInit() ) {
		new_call = build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(new_call)), new_call);
		new_call = build(MODIFY_EXPR, TREE_TYPE(new_call), new_call,
		    irs->convertForAssignment(object_type->defaultInit(), object_type) );
		new_call = build1(ADDR_EXPR, type->toCtype(), new_call);
	    }
	    return new_call;
	}
	break;
    default:
	abort();
    }
}

elem * ScopeExp::toElem(IRState*) {
    ::error("ScopeExp::toElem: don't know what to do (%s)", toChars());
    return error_mark_node;
}

elem * TypeExp::toElem(IRState*) {
    ::error("TypeExp::toElem: don't know what to do (%s)", toChars());
    return error_mark_node;
}

elem * TypeDotIdExp::toElem(IRState*) {
    ::error("TypeDotIdExp::toElem: don't know what to do (%s)", toChars());
    return error_mark_node;
}

elem *
StringExp::toElem(IRState * irs)
{
    Type * base_type = type->toBasetype();
    TY base_ty = type ? base_type->ty : Tvoid;
    tree value;

    switch (base_ty) {
    case Tarray:
    case Tpointer:
	// Assuming this->string is null terminated
	// .. need to terminate with more nulls for wchar and dchar?
	value = build_string(len * sz + 1,
	    gen.hostToTargetString((char *) string, len + 1, sz));
	break;
    case Tsarray:
    case Tvoid:
	value = build_string(len * sz,
	    gen.hostToTargetString((char *) string, len, sz));
	break;
    default:
	error("Invalid type for string constant: %s", type->toChars());
	return error_mark_node;
    }

    // %% endianess of wchar and dchar
    TREE_CONSTANT (value) = 1;
    TREE_READONLY (value) = 1;
    // %% array type doesn't match string length if null term'd...
    TREE_TYPE( value ) = irs->arrayType(base_ty != Tvoid ?
	base_type->next : Type::tchar, len);

    switch (base_ty) {
    case Tarray:
	value = irs->darrayVal(type, len, irs->addressOf( value ));
	break;
    case Tpointer:
	value = irs->addressOf( value );
	break;
    case Tsarray:
	// %% needed?
	TREE_TYPE(value) = type->toCtype();
	break;
    default:
	// nothing
	break;
    }
    return value;
}

elem *
NullExp::toElem(IRState * irs)
{
    TY base_ty = type->toBasetype()->ty; 
    // 0 -> dynamic array.  This is a special case conversion.  
    // Move to convert for convertTo if it shows up elsewhere.
    switch (base_ty) {
    case Tarray:
	return irs->darrayVal( type, 0, NULL );
    case Taarray:
	{
	    tree ctor = make_node(CONSTRUCTOR);
	    tree fa, fb;
	    TREE_TYPE(ctor) = type->toCtype();
	    TREE_READONLY(ctor)=1;
	    fa = TYPE_FIELDS(TREE_TYPE(ctor));
	    fb = TREE_CHAIN(fa);
	    CONSTRUCTOR_ELTS(ctor) = tree_cons(fa, convert(TREE_TYPE(fa), integer_zero_node),
		tree_cons(fb, convert(TREE_TYPE(fb), integer_zero_node), NULL_TREE));
	    return ctor;
	}
	break;
    case Tdelegate:
	// makeDelegateExpression ?
	return irs->delegateVal(convert(ptr_type_node, integer_zero_node),
	    convert(ptr_type_node, integer_zero_node), type);
    default:
	return convert( type->toCtype(), integer_zero_node );
    }
}

elem *
ThisExp::toElem(IRState * irs) {
    if (var)
	return var->toSymbol()->Stree;
    // %% DMD issue -- creates ThisExp without setting var to vthis
    // %%TODO: updated in 0.79-0.81?
    FuncDeclaration * decl = irs->getCurrentFunction();
    assert( decl );
    assert( decl->vthis );
    return decl->vthis->toSymbol()->Stree;
}

elem *
ComplexExp::toElem(IRState * irs)
{
    TypeBasic * compon_type;
    switch (type->toBasetype()->ty) {
    case Tcomplex32: compon_type = (TypeBasic *) Type::tfloat32; break;
    case Tcomplex64: compon_type = (TypeBasic *) Type::tfloat64; break;
    case Tcomplex80: compon_type = (TypeBasic *) Type::tfloat80; break;
    default:
	abort();
    }
    return build_complex(type->toCtype(),
	irs->floatConstant(creall(value), compon_type),
	irs->floatConstant(cimagl(value), compon_type));
}

elem *
ImaginaryExp::toElem(IRState * irs)
{
    return irs->floatConstant(value, type->toBasetype()->isTypeBasic());
}

elem *
RealExp::toElem(IRState * irs)
{
    return irs->floatConstant(value, type->toBasetype()->isTypeBasic());
}

elem *
IntegerExp::toElem(IRState * irs)
{
    return irs->integerConstant(value, type);
}

void
FuncDeclaration::toObjFile()
{
    if (!gen.shouldEmit(this))
	return;

    // Have to watch out for this...
    if (! global.params.useUnitTests &&
	isUnitTestDeclaration()) {
	return;
    }
    
    // doesn't fit with binding level...
    if (cur_irs->shouldDeferFunction(this))
	return;
    // printf("FuncDeclaration::toObjFile %s\n", ident->string);
    tree fn_decl = toSymbol()->Stree;

    if (fbody) {
	tree param_list;
	tree result_decl;
	tree block;
#if D_NO_TRAMPOLINES
	tree static_chain_parm = NULL_TREE;
#endif

	cur_irs = cur_irs->startFunction(this);

	if (isNested())
	    push_function_context();
	
	announce_function( fn_decl );
	current_function_decl = fn_decl;

	TREE_STATIC( fn_decl ) = 1; // set after body is set?
	
	result_decl = build_decl( RESULT_DECL, NULL_TREE, type->next->toCtype() );
	//TREE_USED(result_decl) = 1;
	IRState::setDeclLoc(result_decl, this);
	DECL_RESULT( fn_decl ) = result_decl;
	DECL_CONTEXT( result_decl ) = fn_decl;
	//layout_decl( result_decl, 0 );

	// %%doc This is not a TREE_LIST! It is PARM_DECLs
	// chained together
	param_list = NULL_TREE;

	int n_parameters = parameters ? parameters->dim : 0;

	// Special arguments...
	static const int VTHIS = -2;
	static const int VARGUMENTS = -1;
	
	for (int i = VTHIS; i < (int) n_parameters; i++) {
	    VarDeclaration * param = 0;
	    tree parm_decl = 0;
	    tree parm_type = 0;

	    if (i == VTHIS) {
		if (isThis())
		    param = vthis;
		else if (isNested()) {
		    // See d-decls.cc: FuncDeclaration::toSymbol
		    parm_type = ptr_type_node;
		    parm_decl = build_decl(PARM_DECL, NULL_TREE, parm_type);
		    DECL_ARTIFICIAL( parm_decl ) = 1;
		    DECL_ARG_TYPE (parm_decl) = TREE_TYPE (parm_decl); // %% doc need this arg silently disappears
#if D_NO_TRAMPOLINES
		    static_chain_parm = parm_decl;
#endif
		} else
		    continue;
	    } else if (i == VARGUMENTS) {
		if (v_arguments /*varargs && linkage == LINKd*/)
		    param = v_arguments;
		else
		    continue;
	    } else {
		param = (VarDeclaration *) parameters->data[i];
	    }
	    if (param) {
		parm_decl = param->toSymbol()->Stree;
	    }

	    DECL_CONTEXT (parm_decl) = fn_decl;
	    // param->loc is not set, so use the function's loc
	    // %%doc not setting this crashes debug generating code
	    IRState::setDeclLoc( parm_decl, param ? (Dsymbol*) param : (Dsymbol*) this );

	    expand_decl(parm_decl);

	    // chain them in the correct order
	    param_list = chainon (param_list, parm_decl);
	}

	// param_list is a number of PARM_DECL trees chained together (*not* a TREE_LIST of PARM_DECLs).
	// The leftmost parameter is the first in the chain.  %% varargs?
	DECL_ARGUMENTS( fn_decl ) = param_list; // %% in treelang, useless ? because it just sets them to getdecls() later

	rest_of_decl_compilation(fn_decl, NULL, /*toplevel*/1, /*atend*/0); // http://www.tldp.org/HOWTO/GCC-Frontend-HOWTO-7.html
	// ... has this here, but with more args...
	
	DECL_INITIAL( fn_decl ) = error_mark_node; // Just doing what they tell me to do...
	
	make_decl_rtl (fn_decl, NULL); 
	IRState::prepareSymbolOutput(toSymbol());
	
	IRState::initFunctionStart(fn_decl, loc);
	expand_function_start (fn_decl, 0);

	/* If this function is `main', emit a call to `__main'
	   to run global initializers, etc.  */
	if (DECL_ASSEMBLER_NAME (fn_decl)
	    && MAIN_NAME_P (DECL_ASSEMBLER_NAME (fn_decl)) // other langs use DECL_NAME..
#if D_GCC_VER >= 34
	    && DECL_FILE_SCOPE_P (fn_decl)
#else
	    && ! DECL_CONTEXT (fn_decl)
#endif
	    )
	    expand_main_function ();
	

	//cfun->x_whole_function_mode_p = 1; // %% I gues...
	//cfun->function_frequency = ; // %% it'd be nice to do something with this..
	//need DECL_RESULT ?

	// Start a binding level for the function/arguments
	(*lang_hooks.decls.pushlevel) (0);
	expand_start_bindings (2);
	
	cur_irs->pushDeclContext( fn_decl );

	// Add the argument declarations to the symbol table for the back end
	set_decl_binding_chain( DECL_ARGUMENTS( fn_decl ));

	// %% TREE_ADDRESSABLE and TREE_USED...

	// Start a binding level for the function body
	//(*lang_hooks.decls.pushlevel) (0);
	cur_irs->startScope();
	
	gen.doLineNote(loc);

#if D_NO_TRAMPOLINES
	if (isNested()) {
	    tree sc_expr = make_node(RTL_EXPR);
	    TREE_TYPE(sc_expr) = ptr_type_node;
	    RTL_EXPR_RTL(sc_expr) = current_function_internal_arg_pointer;
	    RTL_EXPR_SEQUENCE(sc_expr) = 0;

	    //tree mod_expr = build(MODIFY_EXPR, void_type_node, sc_expr, static_chain_parm);
	    tree mod_expr = build(MODIFY_EXPR, void_type_node, static_chain_parm, sc_expr);

	    expand_expr_stmt_value(mod_expr, 0, 1);
	}
#endif
	// 
	//if (isMain() /* && also check not nested like the C compiler does */)
	//    expand_main_function ();

	if (vresult)
	    cur_irs->emitLocalVar(vresult);
	fbody->toIR(cur_irs);

	cur_irs->endScope();

#ifdef D_GCC_VER330
	expand_function_end (input_filename, loc.linnum, 0, fn_decl);
#elif D_GCC_VER == 33	
	expand_function_end (input_filename, loc.linnum, 0/*1?*/);
#else
	expand_function_end ();
#endif
	
	block = (*lang_hooks.decls.poplevel) (1, 0, 1);
	cur_irs->popDeclContext();
	DECL_INITIAL (fn_decl) = block; // %% redundant, see poplevel
	BLOCK_SUPERCONTEXT( DECL_INITIAL (fn_decl) ) = fn_decl; // done in C, don't know effect
	
	expand_end_bindings (NULL_TREE, 0, 1);
	

	//expand_stmt (DECL_SAVED_TREE (fn_decl));
	//if (lang_expand_function_end)
	//   (*lang_expand_function_end) ();

	
	rest_of_compilation( fn_decl );
	if (isNested())
	    pop_function_context();
	else
	    current_function_decl = NULL_TREE; // must come before endFunction

	cur_irs = cur_irs->endFunction();
    } else {
	// %% if this is a forward decl? (needed in D?) would this change
	// later if there was a definition?
	DECL_EXTERNAL (fn_decl) = 1;
	// Wouldn't consider this 'public', but if this isn't set
	// wrapup_global_declarations() prints a warning.  Should I
	// be doing something else?
	TREE_PUBLIC( fn_decl ) = 1;

	rest_of_decl_compilation(fn_decl, NULL, 1, 0);
    }
    
    pushdecl(fn_decl); // %%EXPERIMENTAL -- don't pushdecl functions!
}


// %% fix when we know what doFunctionToCall..etc is good for
static Symbol * crackAddr(tree t)
{
    if ( TREE_CODE(t) == ADDR_EXPR )
	t = TREE_OPERAND(t, 0);
    assert( TREE_CODE(t) == VAR_DECL ||
	TREE_CODE(t) == FUNCTION_DECL );
    return symbol_tree(t);
}

void
Module::genobjfile()
{
    IRState * ir_state = new IRState;
    cur_irs = ir_state;

    if (members) {
	for (unsigned i = 0; i < members->dim; i++) {
	    Dsymbol * dsym = (Dsymbol *) members->data[i];

	    dsym->toObjFile();
	}
    }

    /* zomg
    if (Module::moduleinfo)
	Module::moduleinfo->type->toCtype();
    */

    if (needModuleInfo()) {
	{
	    ModuleInfo & mi = gen.getModuleInfo();

	    if (mi.ctors.dim)
		sctor = crackAddr( gen.doFunctionToCallFunctions(this, "_modctor_", & mi.ctors) );
	    if (mi.dtors.dim)
		sdtor = crackAddr( gen.doFunctionToCallFunctions(this, "_moddtor_", & mi.dtors) );
	    if (mi.unitTests.dim)
		stest = crackAddr( gen.doFunctionToCallFunctions(this, "_modtest_", & mi.unitTests) );
	}
    
	genmoduleinfo();

	// maybe put this into obj_moduleinfo
	{
	    /*
	      dt_t * modref_dt;
	      dtword(modref_dt, 0); // this is a pointer
	      dtxoff(modref_dt, toSymbol(), 0);

	      tree t = dt2tree(modref_dt);
	      tree init = dt2tree();
	      Symbol * modref_s = symbol_tree( build_decl(VAR_DECL,
		  get_identifier("__mod_ref"), TREE_TYPE(t) ));
		  // may need smarts for outdata
		  // no Stree = gather Sident, Sdt to form VAR_DECL..., flags too?
	      DECL_INITIAL(
	      TREE_STATIC( 
	    
	     */
	    
	    // all this for:
	    //  void init() {
	    //    our_mod_ref.next = _Dmodule_ref;
	    //    _Dmodule_ref = & our_mod_ref;
	    //  }

	    // struct ModuleReference in moduleinit.d
	    tree mod_ref_type = gen.twoFieldType(Type::tvoid->pointerTo(), gen.getObjectType());
	    tree f0 = TYPE_FIELDS( mod_ref_type );
	    tree f1 = TREE_CHAIN( f0 );

	    tree our_mod_ref = build_decl(VAR_DECL, get_identifier("__mod_ref"), mod_ref_type);
	    dkeep(our_mod_ref);
	
	    tree init = make_node(CONSTRUCTOR);
	    TREE_TYPE( init ) = mod_ref_type;
	    CONSTRUCTOR_ELTS( init ) = tree_cons(f0, d_null_pointer,
		tree_cons(f1, cur_irs->addressOf(this), NULL_TREE));
	    TREE_STATIC( init ) = 1;
	    DECL_ARTIFICIAL( our_mod_ref ) = 1;
	    TREE_PRIVATE( our_mod_ref ) = 1;
	    TREE_STATIC( our_mod_ref ) = 1;
	    DECL_INITIAL( our_mod_ref ) = init;
	    rest_of_decl_compilation(our_mod_ref, NULL, 1, 0);

	    tree the_mod_ref = build_decl(VAR_DECL, get_identifier("_Dmodule_ref"),
		build_pointer_type( mod_ref_type ));
	    dkeep(the_mod_ref);
	    DECL_EXTERNAL( the_mod_ref ) = 1;

	    tree m1 = build(MODIFY_EXPR, void_type_node,
		build(COMPONENT_REF, TREE_TYPE(f0), our_mod_ref, f0),
		the_mod_ref);
	    tree m2 = build(MODIFY_EXPR, void_type_node,
		the_mod_ref, cur_irs->addressOf(our_mod_ref));
	    tree exp = build(COMPOUND_EXPR, void_type_node, m1, m2);

	    gen.doSimpleFunction(this, "___modinit", exp, true);
	}
    }

    // %% This might cause problems if any if are code is called from the gcc wrapup code..
    cur_irs = 0;
}

// This is not used for GCC
unsigned Type::totym() { return 0; }

type *
Type::toCtype() {
    if (! ctype) {
	switch (ty) {
	case Tvoid: return void_type_node;
	case Tint8: return intQI_type_node;
	case Tuns8: return unsigned_intQI_type_node;
	case Tint16: return intHI_type_node;
	case Tuns16: return unsigned_intHI_type_node;
	case Tint32: return intSI_type_node;
	case Tuns32: return unsigned_intSI_type_node;
	case Tint64: return intDI_type_node;
	case Tuns64: return unsigned_intDI_type_node;
	case Tfloat32: return float_type_node;
	case Tfloat64: return double_type_node;
	case Tfloat80: return long_double_type_node;
	case Tcomplex32: return complex_float_type_node;
	case Tcomplex64: return complex_double_type_node;
	case Tcomplex80: return complex_long_double_type_node;
	    /* The following types copy an existing type so that they can have
	       unique names. Also need to change TYPE_MAIN_VARIANT so tree.c
	       functions don't use the original types for arrays, etc. */
	case Tbit:
	    //ctype = copy_node(unsigned_intQI_type_node); // This is what DMD does
	    ctype = build_type_copy( unsigned_intQI_type_node );
	    return ctype;
	case Tchar:
	    ctype = build_type_copy( unsigned_intQI_type_node );
	    return ctype;
	case Twchar:
	    ctype = build_type_copy( unsigned_intHI_type_node );
	    return ctype;
	case Tdchar:
	    ctype = build_type_copy( unsigned_intSI_type_node );
	    return ctype;
	case Timaginary32:
	    ctype = build_type_copy( float_type_node );
	    return ctype;
	case Timaginary64:
	    ctype = build_type_copy( double_type_node );
	    return ctype;
	case Timaginary80:
	    ctype = build_type_copy( long_double_type_node );
	    return ctype;
	    
	case Terror: return error_mark_node;
	default:
	    ::error("unexpected call to Type::toCtype() for %s\n", this->toChars());
	    abort();
	    return NULL_TREE;
	}
    }
    return ctype;
}

// This is not used for GCC
type * Type::toCParamtype() { return 0; }
// This is not used for GCC
Symbol * Type::toSymbol() { return 0; }

type *
TypeTypedef::toCtype()
{
    // %%TODO: create info for debugging
    tree type_node = sym->basetype->toCtype();
    return type_node;
    /*    
    tree type_decl = build_decl(TYPE_DECL, get_identifier( sym->ident->string ),
	type_node);
    DECL_CONTEXT( type_decl ) = cur_irs->getDeclContext();
    rest_of_decl_compilation(type_decl, NULL, cur_irs->isToplevelDeclContext(), 0); //%% flag
    */
}

type *
TypeTypedef::toCParamtype()
{
    return toCtype();
}

void
TypedefDeclaration::toDebug()
{
}


type *
TypeEnum::toCtype()
{
    if (! ctype) {
	tree enum_mem_type_node = sym->memtype->toCtype();
	
	ctype = make_node( ENUMERAL_TYPE );
	// %% c-decl.c: if (flag_short_enums) TYPE_PACKED(enumtype) = 1;
	TYPE_MIN_VALUE( ctype ) = gen.integerConstant(sym->minval, enum_mem_type_node);
	TYPE_MAX_VALUE( ctype ) = gen.integerConstant(sym->maxval, enum_mem_type_node);
	TYPE_PRECISION( ctype ) = size(0) * 8;
	TYPE_SIZE( ctype ) = 0; // as in c-decl.c
	if (sym->ident)
	    TYPE_NAME( ctype ) = get_identifier(sym->ident->string);
	layout_type( ctype );
	TREE_UNSIGNED( ctype ) = isunsigned() != 0; // layout_type can change this

	// Move this to toDebug() ?
	ListMaker enum_values;
	if (sym->members) {
	    for (unsigned i = 0; i < sym->members->dim; i++) {
		EnumMember * member = (EnumMember *) sym->members->data[i];
		char * ident;

		if (sym->ident)
		    ident = concat(sym->ident->string, ".",
			member->ident->string, NULL);
		else
		    ident = (char *) member->ident->string;

		enum_values.cons( get_identifier(ident),
		    gen.integerConstant(member->value->toInteger(), ctype) );

		if (sym->ident)
		    free(ident);
	    }
	}
	TYPE_VALUES( ctype ) = enum_values.head;

	gen.declareType(ctype, sym);
    }
    return ctype;
}

type *
TypeStruct::toCtype()
{
    if (! ctype) {
	// need to set this right away in case of self-references
	ctype = make_node( sym->isUnionDeclaration() ? UNION_TYPE : RECORD_TYPE );
	
	TYPE_LANG_SPECIFIC( ctype ) = build_d_type_lang_specific(this);
	if (sym->ident) {
	    TYPE_NAME( ctype ) = build_decl(TYPE_DECL,
		get_identifier(sym->ident->string), ctype);
	}
	
	AggLayout agg_layout(sym, ctype);
	agg_layout.go();
	agg_layout.finish();

	gen.addAggMethods(ctype, sym);
	gen.declareType(ctype, sym);
    }
    return ctype;
}

void
StructDeclaration::toDebug()
{
}

Symbol * TypeClass::toSymbol() { return sym->toSymbol(); }

unsigned TypeFunction::totym() { return 0; } // Unused

type *
TypeFunction::toCtype() {
    // %%TODO: If x86, and D linkage, use regparm(1)
    
    if (! ctype) {
	ListMaker type_list;
	tree ret_type;

	if (varargs && linkage == LINKd) {
	    // hidden _arguments parameter
	    type_list.cons( Type::typeinfo->type->arrayOf()->toCtype() );
	}
	
	if (arguments) {
	    for (unsigned i = 0; i < arguments->dim; i++) {
		Argument * arg = (Argument *) arguments->data[i];
		type_list.cons( IRState::trueArgumentType(arg) );
	    }
	}
	
	/* Last parm if void indicates fixed length list (as opposed to
	   printf style va_* list). */
	if (! varargs)
	    type_list.cons( void_type_node );
	
	if (next) {
	    ret_type = next->toCtype();
	} else {
	    ret_type = void_type_node;
	}

	ctype = build_function_type(ret_type, type_list.head);
	TYPE_LANG_SPECIFIC( ctype ) = build_d_type_lang_specific(this);

	if (linkage == LINKwindows)
	    ctype = gen.addTypeAttribute(ctype, "stdcall");

#ifdef D_DMD_CALLING_CONVENTIONS
	// W.I.P.
	/* Setting this on all targets.  TARGET_RETURN_IN_MEMORY has precedence
	   over this attribute.  So, only targets on which flag_pcc_struct_return
	   is considered will be affected. */
	if ( (linkage == LINKd && next->size() <= 8) ||
	     (next && next->toBasetype()->ty == Tarray))
	    ctype = gen.addTypeAttribute(ctype, "no_pcc_struct_return"); // EXPER

#ifdef TARGET_386
	if (linkage == LINKd && ! TARGET_64BIT)
	    ctype = gen.addTypeAttribute(ctype, "regparm", integer_one_node);
#endif
#endif
    }
    return ctype;
}

type *
TypeSArray::toCtype()
{
    if (! ctype) {
	if (dim->isConst() && dim->type->isintegral()) {
	    xdmd_integer_t size = dim->toInteger();
	    if (! next->isbit())
		ctype = cur_irs->arrayType(next, size);
	    else
		ctype = cur_irs->arrayType(bitConfig.elemType,
		    bitConfig.bitCountToWords(size));
	} else {
	    ::error("invalid expressions for static array dimension: %s", dim->toChars());
	    abort();
	}
    } 
    return ctype;
}

type *TypeSArray::toCParamtype() { return 0; }

type *
TypeDArray::toCtype()
{
    if (! ctype)
	ctype = gen.twoFieldType(Type::tsize_t, next->pointerTo(), this);
    return ctype;
}

type *
TypeAArray::toCtype()
{
    if (! ctype)
	ctype = gen.twoFieldType(Type::tsize_t, Type::tvoid->pointerTo()->pointerTo(), this);
    return ctype;
}

type *
TypePointer::toCtype()
{
    if (! ctype)
	ctype = build_pointer_type( next->toCtype() );
    return ctype;
}

type *
TypeDelegate::toCtype()
{
    if (! ctype) {
	assert(next->toBasetype()->ty == Tfunction);
	ctype = gen.twoFieldType(Type::tvoid->pointerTo(), next->pointerTo(), this);
    }
    return ctype;
}

#if D_GCC_VER < 34
#define BINFO_ELTS 8
#endif

/* Create debug information for a ClassDeclaration's inheritance tree.
   Interfaces are included because I do not see any benefit. */
static tree
binfo_for(tree tgt_binfo, ClassDeclaration * cls)
{
    tree binfo = make_tree_vec(BINFO_ELTS);
    TREE_TYPE              (binfo) = TREE_TYPE( cls->type->toCtype() ); // RECORD_TYPE, not REFERENCE_TYPE
    BINFO_INHERITANCE_CHAIN(binfo) = tgt_binfo;
    BINFO_OFFSET           (binfo) = gen.integerConstant(0, Type::tuns32); // %% type?

    if (cls->baseClass) {
	tree prot_tree;
	    
	BINFO_BASETYPES(binfo)    = make_tree_vec(1);
	BINFO_BASETYPE(binfo, 0)  = binfo_for(binfo, cls->baseClass);
#ifdef BINFO_BASEACCESSES
	BINFO_BASEACCESSES(binfo) = make_tree_vec(1);
	switch ( ((BaseClass *) cls->baseclasses.data[0])->protection ) {
	case PROTpublic:
	    prot_tree = access_public_node;
	    break;
	case PROTprotected:
	    prot_tree = access_protected_node;
	    break;
	case PROTprivate:
	    prot_tree = access_private_node;
	    break;
	default:
	    prot_tree = access_public_node;
	    break;
	}
	BINFO_BASEACCESS(binfo,0) = prot_tree;
#endif
    }

    return binfo;
}

/* Create debug information for an InterfaceDeclaration's inheritance
   tree.  In order to access all inherited methods in the debugger,
   the entire tree must be described.

   This function makes assumptions about inherface layout. */
static tree
intfc_binfo_for(tree tgt_binfo, ClassDeclaration * iface, unsigned & inout_offset)
{
    tree binfo = make_tree_vec(BINFO_ELTS);
    tree prot_tree;
    TREE_TYPE              (binfo) = TREE_TYPE( iface->type->toCtype() ); // RECORD_TYPE, not REFERENCE_TYPE
    BINFO_INHERITANCE_CHAIN(binfo) = tgt_binfo;
    BINFO_OFFSET           (binfo) = gen.integerConstant(inout_offset *
	int_size_in_bytes(ptr_type_node), Type::tuns32); // %% type?

    if (iface->baseclasses.dim) {
	BINFO_BASETYPES(binfo)    = make_tree_vec(iface->baseclasses.dim);
#ifdef BINFO_BASEACCESSES
	BINFO_BASEACCESSES(binfo) = make_tree_vec(iface->baseclasses.dim);
#endif
    }
    for (unsigned i = 0; i < iface->baseclasses.dim; i++) {
	BaseClass * bc = (BaseClass *) iface->baseclasses.data[i];

	if (i)
	    inout_offset++;

	BINFO_BASETYPE(binfo, i)  = intfc_binfo_for(binfo, bc->base, inout_offset);
#ifdef BINFO_BASEACCESSES
	switch ( bc->protection ) {
	case PROTpublic:
	    prot_tree = access_public_node;
	    break;
	case PROTprotected:
	    prot_tree = access_protected_node;
	    break;
	case PROTprivate:
	    prot_tree = access_private_node;
	    break;
	default:
	    prot_tree = access_public_node;
	    break;
	}
	BINFO_BASEACCESS(binfo, i) = prot_tree;
#endif
    }

    return binfo;
}

type *
TypeClass::toCtype()
{
    if (! ctype) {
	tree rec_type;
	Array base_class_decls;
	bool inherited = sym->baseClass != 0;
	tree obj_rec_type;
	tree vfield;

	/* Need to set ctype right away in case of self-references to
	   the type during this call. */
	rec_type = make_node( RECORD_TYPE );
	ctype = build_reference_type( rec_type );

	obj_rec_type = TREE_TYPE( gen.getObjectType()->toCtype() );
	
	// Note that this is set on the reference type, not the record type.
	TYPE_LANG_SPECIFIC( ctype ) = build_d_type_lang_specific( this );

	AggLayout agg_layout(sym, rec_type);
	
	// Most of this silly code is just to produce correct debugging information.

	/* gdb apparently expects the vtable field to be named
	   "_vptr$...." (stabsread.c) Otherwise, the debugger gives
	   lots of annoying error messages.  C++ appends the class
	   name of the first base witht that field after the '$'. */
	/* update: annoying messages might not appear anymore after making
	   other changes */
	// Add the virtual table pointer
	tree decl = build_decl(FIELD_DECL, get_identifier("_vptr$"), /*vtbl_type*/d_vtbl_ptr_type_node);
	agg_layout.addField( decl, 0 ); // %% target stuff..
	
	if (inherited) {
	    vfield = copy_node( decl );
	    DECL_ARTIFICIAL( decl ) = DECL_IGNORED_P( decl ) = 1;
	} else {
	    vfield = decl;
	}
	DECL_VIRTUAL_P( vfield ) = 1;
	TYPE_VFIELD( rec_type ) = vfield; // This only seems to affect debug info

	if (! sym->isInterfaceDeclaration()) {
	    DECL_FCONTEXT( vfield ) = obj_rec_type;

	    // Add the monitor
	    // %% target type
	    decl = build_decl(FIELD_DECL, get_identifier("_monitor"), ptr_type_node);
	    DECL_FCONTEXT( decl ) = obj_rec_type;
	    DECL_ARTIFICIAL( decl ) = DECL_IGNORED_P( decl ) = inherited;
	    agg_layout.addField( decl, 4); // %% target stuff...

	    // Add the fields of each base class
	    agg_layout.go();
	} else {
	    ClassDeclaration * p = sym;
	    while (p->baseclasses.dim) {
		p = ((BaseClass *) p->baseclasses.data[0])->base;
	    }
	    DECL_FCONTEXT( vfield ) = TREE_TYPE( p->type->toCtype() );
	}

	agg_layout.finish();

	// Create BINFO even if debugging is off.  This is needed to keep
	// references to inherited types. There could be a more effecient way,
	// however.
	// if (write_symbols != NO_DEBUG) {
	
	gen.addAggMethods(rec_type, sym);

	if ( ! sym->isInterfaceDeclaration() )
	    TYPE_BINFO( rec_type ) = binfo_for(NULL_TREE, sym);
	else {
	    unsigned offset = 0;
	    BaseClass bc;
	    bc.base = sym;
	    TYPE_BINFO( rec_type ) = intfc_binfo_for(NULL_TREE, sym, offset);
	}
	
	// }
	
	gen.declareType(rec_type, sym);
    }
    return ctype;
}

void
ClassDeclaration::toDebug()
{
}

void
LabelStatement::toIR(IRState* irs)
{
    FuncDeclaration * func = irs->getCurrentFunction(); 
    LabelDsymbol * label = isReturnLabel ? func->returnLabel : func->searchLabel(ident);
    tree t = irs->getLabelTree( label  );

    if (t) {
	expand_label( t );

	if (isReturnLabel && func->fensure)
	    func->fensure->toIR(irs);
	else if (statement) // %% really else?
	    statement->toIR(irs);
    }
    // else, there was an error
}

void
GotoStatement::toIR(IRState* irs)
{
    gen.doLineNote(loc); /* This makes the 'undefined label' error show up on the correct line...
			    The extra doLineNote in doJump shouldn't cause a problem. */
    tree t = irs->getLabelTree( label );
    if (t)
	gen.doJump(this, t);
    // else, there was an error
}

void
GotoCaseStatement::toIR(IRState*)
{
    // assumes cblocks have been set in SwitchStatement::toIR
    gen.doJump(this, cs->cblock);
}

void
GotoDefaultStatement::toIR(IRState*)
{
    // assumes cblocks have been set in SwitchStatement::toIR
    gen.doJump(this, sw->sdefault->cblock);
}

void
SwitchErrorStatement::toIR(IRState* irs)
{
    expand_expr_stmt_value( irs->assertCall(loc, LIBCALL_SWITCH_ERROR),
	0, 1);
}

void
VolatileStatement::toIR(IRState* irs)
{
    irs->pushVolatile(); 
    statement->toIR( irs );
    irs->popVolatile();
}

void
ThrowStatement::toIR(IRState* irs)
{
    ClassDeclaration * class_decl = exp->type->isClassHandle();
    // Front end already checks for isClassHandle
    InterfaceDeclaration * intfc_decl = class_decl->isInterfaceDeclaration();
    
    tree arg = exp->toElem(irs);

    if (intfc_decl) {
	if ( ! intfc_decl->isCOMclass()) {
	    arg = irs->convertTo(arg, exp->type, irs->getObjectType());
	} else {
	    error("cannot throw COM interfaces");
	}
    }
    
    gen.doLineNote(loc);
    expand_expr_stmt_value( irs->libCall(LIBCALL_THROW, 1, & arg), 0, 1 );
    // %%TODO: somehow indicate flow stops here?
}

void
TryFinallyStatement::toIR(IRState * irs)
{
    // %% doc: this is not the same as a start_eh/end_eh_cleanup sequence
    tree tf = build(TRY_FINALLY_EXPR, void_type_node,
	gen.makeStmtExpr(body, irs),
	gen.makeStmtExpr(finalbody, irs));
    // TREE_SIDE_EFFECTS(tf) = 1; // probably not needed
    gen.doLineNote(loc); // %% really?
    expand_expr_stmt_value( tf, 0, 1 );
}

void
TryCatchStatement::toIR(IRState * irs)
{
    gen.doLineNote(loc); // %% really?
    
    expand_eh_region_start();
    if (body)
	body->toIR(irs);
    expand_start_all_catch();
    if (catches) {
	for (unsigned i = 0; i < catches->dim; i++) {
	    Catch * a_catch = (Catch *) catches->data[i];
		
	    expand_start_catch( a_catch->type->toCtype() );
	    
	    gen.doLineNote(a_catch->loc);
	    irs->startScope();

	    if ( a_catch->var ) {
		tree exc_obj = irs->convertTo(irs->exceptionObject(),
		    irs->getObjectType(), a_catch->type);
		// need to override initializer...
		// set DECL_INITIAL now and emitLocalVar will know not to change it
		DECL_INITIAL( a_catch->var->toSymbol()->Stree ) = exc_obj;
		irs->emitLocalVar(a_catch->var);
	    }

	    a_catch->handler->toIR(irs);
	    irs->endScope();
	    expand_end_catch();
	}
    }
    expand_end_all_catch();
}

void
WithStatement::toIR(IRState * irs)
{
    if (wthis) {
	irs->startScope();
	irs->emitLocalVar(wthis);
    }
    body->toIR(irs);
    if (wthis) {
	irs->endScope();
    }
}

void
SynchronizedStatement::toIR(IRState * irs)
{
    if (exp) {
	InterfaceDeclaration * iface;
	
	tree decl = build_decl(VAR_DECL, NULL_TREE, IRState::getObjectType()->toCtype());
	pushdecl(decl); /* Only doing this for GC purposes, otherwise should start a new
			   block and place it there */
	DECL_ARTIFICIAL( decl ) = 1;
	DECL_IGNORED_P( decl ) = 1;
	DECL_CONTEXT( decl ) = irs->getDeclContext();
	tree cleanup = irs->libCall(LIBCALL_MONITOREXIT, 1, & decl);
	// assuming no conversions needed
	tree init_exp;

	assert(exp->type->toBasetype()->ty == Tclass);
	iface = ((TypeClass *) exp->type->toBasetype())->sym->isInterfaceDeclaration();
	if (iface) {
	    if (! iface->isCOMclass()) {
		init_exp = irs->convertTo(exp, irs->getObjectType());
	    } else {
		error("cannot synchronize on a COM interface");
		init_exp = error_mark_node;
	    }
	} else {
	    init_exp = exp->toElem(irs);
	}
	
	DECL_INITIAL(decl) = init_exp;

	gen.doLineNote(loc);

	expand_start_target_temps();
	//irs->startBindings();
	expand_decl(decl);
	expand_decl_init(decl);
	expand_expr_stmt_value(irs->libCall(LIBCALL_MONITORENTER, 1, & decl), 0, 1);
	expand_decl_cleanup(decl, cleanup);
	if (body)
	    body->toIR( irs );
	expand_end_target_temps();

	//expand_expr_stmt_value(irs->libCall(LIBCALL_MONITOREXIT, 1, & sync_exp), 0, 1);
    } else {
#ifndef D_CRITSEC_SIZE
#define D_CRITSEC_SIZE 64
#endif
	static tree critsec_type = 0;
	static unsigned critsec_idx;

	if (! critsec_type ) {
	    critsec_type = irs->arrayType(Type::tuns8, D_CRITSEC_SIZE);
	}

	char buf[64]; // name is only used to prevent ICEs
	snprintf(buf, sizeof(buf), "__critsec%u", ++critsec_idx);
	tree critsec_decl = build_decl(VAR_DECL, get_identifier(buf), critsec_type);
	tree critsec_ref = irs->addressOf(critsec_decl); // %% okay to use twice?
	dkeep(critsec_decl);

	TREE_STATIC(critsec_decl) = 1;
	TREE_PRIVATE(critsec_decl) = 1;
	DECL_ARTIFICIAL(critsec_decl) = 1;
	DECL_IGNORED_P(critsec_decl) = 1;

	rest_of_decl_compilation(critsec_decl, NULL, 1, 0);

	expand_eh_region_start();	
	expand_expr_stmt_value(irs->libCall(LIBCALL_CRITICALENTER, 1, & critsec_ref), 0, 1);
	if (body)
	    body->toIR( irs );
	expand_expr_stmt_value(irs->libCall(LIBCALL_CRITICALEXIT, 1, & critsec_ref), 0, 1);
	expand_eh_region_end_cleanup(irs->libCall(LIBCALL_CRITICALEXIT, 1, & critsec_ref));
    }
}

void
ContinueStatement::toIR(IRState* irs)
{
    gen.doLineNote(loc);
#ifdef OLD
    expand_continue_loop( irs->getLoopForLabel( ident ) );
#endif
    irs->continueLoop(ident);
}

void
BreakStatement::toIR(IRState* irs)
{
    gen.doLineNote(loc);
#ifdef OLD    
    if (ident)
	expand_exit_loop( irs->getLoopForLabel( ident ) );
    else
	expand_exit_something();
#endif
    irs->exitLoop(ident);
}

void
ReturnStatement::toIR(IRState* irs)
{
    if (exp && exp->type->toBasetype()->ty != Tvoid) { // %% == Type::tvoid ?
	FuncDeclaration * func = irs->getCurrentFunction();
	Type * ret_type = func->type->next;
	tree result_decl = DECL_RESULT( current_function_decl ); // %% global -- go through irs?
	tree result_assign = build ( MODIFY_EXPR,
	    TREE_TYPE( result_decl ), result_decl,
	    // %% convert for init -- if we were returning a reference,
	    // would want to take the address...
	    cur_irs->convertForAssignment(exp, (Type*)ret_type) ); /*exp->toElem( irs )*/
	gen.doLineNote(loc);
	
	expand_return(result_assign);
    } else {
	expand_null_return();
    }
}

static tree
make_case_label(Loc loc) {
    tree label_decl = build_decl(LABEL_DECL, NULL_TREE, void_type_node);
    DECL_CONTEXT( label_decl ) = current_function_decl;
    DECL_MODE( label_decl ) = VOIDmode;
    // Not setting this doesn't seem to cause problems (unlike VAR_DECLs)
    if (loc.filename)
	IRState::setDeclLoc( label_decl, loc );
    

    return label_decl;
}

void
DefaultStatement::toIR(IRState * irs)
{
    tree dummy;

    pushcase( NULL_TREE, convert, cblock, & dummy);
    statement->toIR( irs );
}

void
CaseStatement::toIR(IRState * irs)
{
    tree dummy;
    tree case_value;

    if ( exp->type->isscalar() )
	case_value = exp->toElem(irs);
    else
	case_value = irs->integerConstant(index, Type::tuns32); // %% target type
    pushcase( case_value, convert, cblock, & dummy);
    statement->toIR( irs );
}

void
SwitchStatement::toIR(IRState * irs)
{
    tree cond_tree;
    // %% also what about c-semantics doing emit_nop() ?
    gen.doLineNote( loc );

    cond_tree = condition->toElem( irs );

    Type * cond_type = condition->type->toBasetype();
    if (cond_type->ty == Tarray) {
	Type * elem_type = cond_type->next->toBasetype();
	LibCall lib_call;
	switch (elem_type->ty) {
	case Tchar:  lib_call = LIBCALL_SWITCH_STRING; break;
	case Twchar: lib_call = LIBCALL_SWITCH_USTRING; break;
	case Tdchar: lib_call = LIBCALL_SWITCH_DSTRING; break;
	default:
	    ::error("switch statement value must be an array of some character type, not %s", elem_type->toChars());
	    abort();
	}

	// Apparently the backend is supposed to sort and set the indexes
	// on the case array
	// have to change them to be useable
	cases->sort(); // %%!!
	
	// %% check char or wchar -- don't have support for dchar
	// okay to output this now?
	ArrayMaker a(elem_type, STCstatic | STCconst);
	for (unsigned case_i = 0; case_i < cases->dim; case_i++) {
	    CaseStatement * case_stmt = (CaseStatement *) cases->data[case_i];
	    tree elem = case_stmt->exp->toElem(cur_irs);
	    TREE_STATIC( elem ) = 1;
	    TREE_CONSTANT( elem ) = 1;
	    a.add( elem );
	    
	    case_stmt->index = case_i;
	}

	tree table = a.finish();

	tree args[2] = {
	    irs->darrayVal(cond_type->arrayOf()->toCtype(), cases->dim,
		irs->addressOf(table)),
	    cond_tree };
	
	cond_tree = irs->libCall(lib_call, 2, args);
    } else if (! cond_type->isscalar()) {
	::error("cannot handle switch condition of type %s", cond_type->toChars());
	abort();
    }

    // Build LABEL_DECLs now so they can be refered to by goto case
    if (cases) {
	for (unsigned i = 0; i < cases->dim; i++) {
	    CaseStatement * case_stmt = (CaseStatement *) cases->data[i];
	    case_stmt->cblock = make_case_label(case_stmt->loc);
	}
	if (sdefault)
	    sdefault->cblock = make_case_label(sdefault->loc);
    }
    
    genrtl_do_pushlevel(); // %% see where else this is needed(is it?) // %% C does this in other places
    gen.doLineNote(loc);
    expand_start_case( 1, cond_tree, TREE_TYPE( cond_tree ), "switch statement" );
    irs->beginLoop(this, NULL);
    if (body)
	body->toIR( irs );
    expand_end_case( cond_tree );
    irs->endLoop();
}


void
Statement::toIR(IRState*)
{
    ::error("Statement::toIR: don't know what to do (%s)", toChars());
    abort();
}

void
IfStatement::toIR(IRState * irs)
{
    genrtl_do_pushlevel (); //%%
    gen.doLineNote( loc );
    expand_start_cond( irs->convertForCondition( condition ), 0 );
    if (ifbody)
	ifbody->toIR( irs );
    if ( elsebody ) {
	expand_start_else();
	elsebody->toIR ( irs );
    }
    expand_end_cond();
}

void
ForeachStatement::toIR(IRState* irs)
{
    // %% better?: set iter to start - 1 and use result of increment for condition?

    // side effects?

    Type * agg_type = aggr->type->toBasetype();
    Type * elem_type = agg_type->next->toBasetype();
    tree iter_decl;
    tree bound_expr;
    tree iter_init_expr;
    tree aggr_expr = irs->maybeMakeTemp( aggr->toElem(irs) );

    assert(value);

    irs->startScope();
    irs->startBindings(); /* Variables created by the function will probably
			     end up in a contour created by emitLocalVar.  This
			     startBindings call is just to be safe */
    gen.doLineNote( loc );

    Loc default_loc;
    if (loc.filename)
	default_loc = loc;
    else
	default_loc = Loc(getCurrentModule(), 1); // %% fix
    
    if (! value->loc.filename)
	gen.setDeclLoc( value->toSymbol()->Stree, default_loc );
    
    irs->emitLocalVar(value, true);

    if (key) {
	if (! key->loc.filename)
	    gen.setDeclLoc( key->toSymbol()->Stree, default_loc );
	if (! key->init)
	    DECL_INITIAL( key->toSymbol()->Stree ) =
		key->type->defaultInit()->toElem(irs);
	irs->emitLocalVar(key); // %% getExpInitializer causes uneeded initialization
    }

    if (elem_type->ty != Tbit) {
	if (value->isOut()) {
	    iter_decl = value->toSymbol()->Stree;
	} else {
	    iter_decl = build_decl(VAR_DECL, NULL_TREE,
		build_pointer_type(elem_type->toCtype()) );
	    pushdecl(iter_decl);
	    DECL_CONTEXT( iter_decl ) = irs->getDeclContext();
	    DECL_ARTIFICIAL( iter_decl ) = 1;
	    expand_decl(iter_decl);
	}

	if ( agg_type->ty == Tsarray) {
	    bound_expr = ((TypeSArray *) agg_type)->dim->toElem(irs);
	    iter_init_expr = irs->addressOf( aggr_expr );
	    // Type needs to be pointer-to-element to get pointerIntSum
	    // to work
	    TREE_TYPE(iter_init_expr) = agg_type->next->pointerTo()->toCtype();
	} else {
	    bound_expr = irs->darrayLenRef( aggr_expr );
	    iter_init_expr = irs->darrayPtrRef( aggr_expr );
	}
	iter_init_expr = save_expr( iter_init_expr );
	bound_expr = irs->pointerIntSum(iter_init_expr, bound_expr);
	// aggr. isn't supposed to be modified, so...
	bound_expr = save_expr( bound_expr );

	tree condition = build(LT_EXPR, boolean_type_node, iter_decl, bound_expr);

	expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, iter_decl, iter_init_expr),
	    0, 1);

	nesting * new_loop = expand_start_loop_continue_elsewhere(1);
	irs->beginLoop(this, new_loop);
	expand_exit_loop_if_false(new_loop, condition);
	if ( iter_decl != value->toSymbol()->Stree ) {
	    // %% check..
	    expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, value->toSymbol()->Stree,
				       build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(iter_decl)), iter_decl)),
		0, 1);
	}
	if (body)
	    body->toIR( irs );
	expand_loop_continue_here();

	expand_expr_stmt_value(	build(MODIFY_EXPR, void_type_node, iter_decl,
			    build(PLUS_EXPR, TREE_TYPE(iter_decl), iter_decl,
				size_int(elem_type->size()))),
	    0, 1);
	if (key) {
	    tree key_decl = key->toSymbol()->Stree;
	    expand_expr_stmt_value(	build(MODIFY_EXPR, void_type_node, key_decl,
				build(PLUS_EXPR, TREE_TYPE(iter_decl), key_decl,
				    irs->integerConstant(1, TREE_TYPE(key_decl)))),
		0, 1);
	}
    } else {
	// %% could be more effecient by operating on a word at a time
	tree aggr_expr = irs->maybeMakeTemp( aggr->toElem(irs) );
	tree iter_decl;
	Type * iter_type;
	
	if (key) {
	    iter_decl = key->toSymbol()->Stree;
	    iter_type = key->type;
	} else {
	    iter_type = Type::tindex; // %% should be unsigned?
	    iter_decl = build_decl(VAR_DECL, NULL_TREE, iter_type->toCtype());
	    pushdecl(iter_decl);
	    DECL_CONTEXT( iter_decl ) = irs->getDeclContext();
	    DECL_ARTIFICIAL( iter_decl ) = 1;
	    expand_decl(iter_decl);
	}
	
	expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, iter_decl,
				   irs->integerConstant(0, iter_type)),
	    0, 1);//%%redundant after key->toobjfile... unless typedef..
	
	nesting * new_loop = expand_start_loop_continue_elsewhere(1);
	irs->beginLoop(this, new_loop);

	IndexExp * elem_expr = new IndexExp(loc,
	    new WrappedExp(loc, aggr->op/* %%check */, aggr_expr, aggr->type),
	    new WrappedExp(loc, TOKvar, iter_decl, iter_type));
	elem_expr->type = elem_type;
	if ( agg_type->ty == Tsarray) {
	    bound_expr = ((TypeSArray *) agg_type)->dim->toElem(irs);
	} else {
	    bound_expr = irs->darrayLenRef( aggr_expr );
	}
	tree condition = build(LT_EXPR, boolean_type_node, iter_decl, bound_expr);
	

	expand_exit_loop_if_false(new_loop, condition);

	AssignExp * a_exp = new AssignExp(loc, new VarExp(loc, value), elem_expr);
	a_exp->type = value->type; // tvoid, maybe? -- okay for MODIFY_EXPR?
	expand_expr_stmt_value( a_exp->toElem(irs), 0, 1 );
	
	if (body)
	    body->toIR( irs );
	expand_loop_continue_here();

	// %% could use preincrement on other case, too
	//tree t = build(PREINCREMENT_EXPR, void_type_node, iter_decl, integer_one_node);
	//TREE_SIDE_EFFECTS( t ) = 1;
	tree t = build(MODIFY_EXPR, void_type_node, iter_decl,
	    build(PLUS_EXPR, TREE_TYPE(iter_decl), iter_decl,
		irs->integerConstant(1, TREE_TYPE(iter_decl))));
	expand_expr_stmt_value( t, 0, 1);
    }
    expand_end_loop();
    irs->endLoop();

    irs->endBindings(); // not really needed
    irs->endScope();
}

void
ForStatement::toIR(IRState * irs)
{
    gen.doLineNote( loc );
    
    // %% what about scope?
    if (init)
	init->toIR( irs );
    nesting * new_loop = expand_start_loop_continue_elsewhere(1);
    irs->beginLoop(this, new_loop);
    expand_exit_loop_if_false(new_loop, irs->convertForCondition( condition ));
    
    if (body)
	body->toIR( irs );
    expand_loop_continue_here();
    if (increment) {
	// Dup'd from ExpStatement::toIR
	tree exp_tree = increment->toElem(irs);
	TREE_SIDE_EFFECTS(exp_tree) = 1; // for now.  not setting this can make it not be emitted..(details unknown)


	gen.doLineNote(increment->loc); //%%really?
	expand_expr_stmt_value(exp_tree, 0, 1 ); // %% want flag...
    }
    expand_end_loop();
    irs->endLoop();
}

void
DoStatement::toIR(IRState * irs)
{
    gen.doLineNote( loc );

    nesting * new_loop = expand_start_loop_continue_elsewhere(1);

    irs->beginLoop(this, new_loop);
    if (body)
	body->toIR( irs );
    // %% else expand_*_null_loop?
    expand_loop_continue_here();
    gen.doLineNote(condition->loc); // %% really?
    if (! expand_exit_loop_if_false(new_loop, irs->convertForCondition( condition )) ) {
	abort();
    }
    expand_end_loop();    
    irs->endLoop();
}

void
WhileStatement::toIR(IRState* irs)
{
    gen.doLineNote( loc );

    nesting * new_loop = expand_start_loop(1);

    irs->beginLoop(this, new_loop);

    //expand_loop_continue_here();
    if (! expand_exit_loop_top_cond(new_loop, irs->convertForCondition( condition )) ) {
	abort();
    }
    if (body)
	body->toIR( irs );
    // %% expand_end_null_loop
    expand_end_loop();
    
    irs->endLoop();
}

void
ScopeStatement::toIR(IRState* irs)
{
    if (statement) {
	irs->startScope();
	statement->toIR( irs );
	irs->endScope();
    }
}

void
CompoundStatement::toIR(IRState* irs)
{
    if (statements) {
	for (unsigned i = 0; i < statements->dim; i++) {
	    Statement * statement = (Statement *) statements->data[i];
	    
	    if (statement)
		statement->toIR(irs);
	}
    }
}

void
ExpStatement::toIR(IRState * irs)
{
    if (exp) {
	gen.doLineNote(loc);

	tree exp_tree = exp->toElem(irs);
	if (irs->inVolatile())
	    TREE_THIS_VOLATILE(exp_tree) = 1; // don't know if toplevel is good enough
	expand_expr_stmt_value(exp_tree, 0, 1 );
    } else {
	// nothing
    }
}

void
AsmStatement::toIR(IRState *)
{
    sorry("assembler statements are not supported yet");
}

void
EnumDeclaration::toDebug()
{
    
}

int
Dsymbol::cvMember(unsigned char*)
{
    return 0;
}
int
EnumDeclaration::cvMember(unsigned char*)
{
    return 0;
}
int
FuncDeclaration::cvMember(unsigned char*)
{
    return 0;
}
int
VarDeclaration::cvMember(unsigned char*)
{
    return 0;
}
int
TypedefDeclaration::cvMember(unsigned char*)
{
    return 0;
}

rtx
#if D_GCC_VER == 33
d_expand_expr(tree exp, rtx target , enum machine_mode tmode, int modifier)
#else
d_expand_expr(tree exp, rtx target , enum machine_mode tmode, int modifier, rtx *)
#endif
{
    if ( TREE_CODE(exp) == (enum tree_code) D_STMT_EXPR ) {
	IRState * irs;
	Statement * stmt;

	gen.retrieveStmtExpr(exp, & stmt, & irs);
	// need push_temp_slots()?

	push_temp_slots (); // will this work? maybe expand_start_binding
	tree rtl_expr = expand_start_stmt_expr(1);
	// preserve_temp_slots as in c-common.c:c_expand_expr
	
	stmt->toIR(irs);

	expand_end_stmt_expr (rtl_expr);
	rtx result = expand_expr (rtl_expr, target, tmode, (enum expand_modifier) modifier);
	pop_temp_slots();
	return result;
    } else if ( TREE_CODE(exp) == (enum tree_code) D_ARRAY_SET_EXPR ){
	// %% if single byte element, expand to memset
	
	assert( POINTER_TYPE_P( TREE_TYPE( TREE_OPERAND( exp, 0 ))));
	assert( INTEGRAL_TYPE_P( TREE_TYPE( TREE_OPERAND( exp, 2 ))));
	// assuming unsigned source is unsigned

	push_temp_slots (); // will this work? maybe expand_start_binding
	tree rtl_expr = expand_start_stmt_expr(1);
	
	//expand_start_bindings(0);
	cur_irs->startBindings(); // %%maybe not
	
	tree count_var = build_decl(VAR_DECL, NULL_TREE, Type::tsize_t->toCtype()); // %% same type as count?
	tree ptr_var = build_decl(VAR_DECL, NULL_TREE, TREE_TYPE(TREE_OPERAND(exp, 0)));
	pushdecl(count_var); // for GC, shouldn't be seen in debugger
	pushdecl(ptr_var);   // // for GC, shouldn't be seen in debugger
	DECL_CONTEXT(count_var) = cur_irs->getDeclContext(); // %% should use make/retrieveStmtExpr..
	DECL_CONTEXT(ptr_var) = cur_irs->getDeclContext();
	DECL_INITIAL(count_var) = TREE_OPERAND(exp, 2);
	DECL_INITIAL(ptr_var) = TREE_OPERAND(exp, 0);
	expand_decl(count_var);
	expand_decl_init(count_var);
	expand_decl(ptr_var);
	expand_decl_init(ptr_var);
	//d_mark_addressable(count_var);
	//d_mark_addressable(ptr_var);
	nesting * loop = expand_start_loop (1);
	
	expand_exit_loop_top_cond(loop,
	    build(NE_EXPR, boolean_type_node, integer_zero_node, count_var));

	static tree t;
	t = build(MODIFY_EXPR, void_type_node,
	    build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(ptr_var)), ptr_var),
	    TREE_OPERAND(exp, 1));
	
	expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node,
				   build1(INDIRECT_REF, TREE_TYPE(TREE_TYPE(ptr_var)), ptr_var),
				   TREE_OPERAND(exp, 1)), 0, 1);
	//expand_expr_stmt_value(build(PREDECREMENT_EXPR, void_type_node, count_var, integer_one_node), 0, 1);
	//expand_expr_stmt_value(build(PREINCREMENT_EXPR, void_type_node, ptr_var,
	// TYPE_SIZE_UNIT(TREE_TYPE(TREE_TYPE(ptr_var)))), 0, 1);
	//expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, count_var, integer_zero_node), 0, 1);

	expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, ptr_var,
				   build(PLUS_EXPR, TREE_TYPE(ptr_var), ptr_var,
				       TYPE_SIZE_UNIT(TREE_TYPE(TREE_TYPE(ptr_var))))), 0, 1);
	expand_expr_stmt_value(build(MODIFY_EXPR, void_type_node, count_var,
				   build(MINUS_EXPR, TREE_TYPE(count_var), count_var, integer_one_node)), 0, 1);
	expand_end_loop ();
	
	//expand_end_bindings(NULL_TREE, 1, 1);
	cur_irs->endBindings(); // %%maybe not
	
	expand_end_stmt_expr(rtl_expr);
	rtx result = expand_expr(rtl_expr, target, tmode, (enum expand_modifier) modifier);
	pop_temp_slots ();
	return result;
    } else {
	abort();
    }
}

static tree
d_build_eh_type_type(tree type)
{
    TypeClass * d_type = (TypeClass *) IRState::getDType(type);
    assert(d_type);
    d_type = (TypeClass *) d_type->toBasetype();
    assert(d_type->ty == Tclass);
    return IRState::addressOf( d_type->sym->toSymbol()->Stree );
}

tree d_void_zero_node;
#if D_GCC_VER == 33
tree boolean_type_node;
tree boolean_true_node;
tree boolean_false_node;
#endif

tree d_null_pointer;
tree d_vtbl_ptr_type_node;

void
gcc_d_backend_init()
{
    // %% need this here to add the type decls ...
    init_global_binding_level();

    // This is required or we'll crash pretty early on. %%log
    build_common_tree_nodes (flag_signed_char);

    // This is also required (or the manual equivalent) or crashes
    // will occur later
    // %% configure, target, flags, etc.
    size_type_node = unsigned_intSI_type_node;
    // c was: TREE_TYPE (identifier_global_value (get_identifier (SIZE_TYPE)));
    //signed_size_type_node = c_common_signed_type (size_type_node);
    // %%doc Early_type_list in stor-layout.c is an f'd up way of doing things...
    // Also, if this is called after the next statements, you'll get an ICE.
    set_sizetype(size_type_node);


    // need this for void.. %% but this crashes... probably need to impl
    // some things in dc-lang.cc
    build_common_tree_nodes_2 (0 /* %% support the option */);

    // Specific to D (but so far all taken from C)
    d_void_zero_node = build_int_2 (0, 0);
    TREE_TYPE (d_void_zero_node) = void_type_node;
#if D_GCC_VER == 33
    /* For GCC 3.3.x, the C front end uses integer_type_node, but this conflicts
       with bit.size==1.  GCC 3.4 uses an 8bit value like this.  The GCC 3.4 code
       is copied here. */
    boolean_type_node = make_unsigned_type(BOOL_TYPE_SIZE);
    TREE_SET_CODE (boolean_type_node, BOOLEAN_TYPE);
    TYPE_MAX_VALUE (boolean_type_node) = build_int_2 (1, 0);
    TREE_TYPE (TYPE_MAX_VALUE (boolean_type_node)) = boolean_type_node;
    TYPE_PRECISION (boolean_type_node) = 1;
    
    boolean_false_node = TYPE_MIN_VALUE(boolean_type_node);// convert(boolean_type_node, integer_zero_node);
    boolean_true_node = TYPE_MAX_VALUE(boolean_type_node); //convert(boolean_type_node, integer_one_node);
    // * /
#endif
    // (else) %%TODO: we are relying on default boolean_type_node being 8bit / same as Tbit

    d_null_pointer = convert(ptr_type_node, integer_zero_node);
    

    //%% fix dbxout_init expects integer_type_node and others to have a TYPE_NAME..
    // and it must be a TYPE_DECL, not an IDENTIFIER_NODE
    // GCC 3.3 only?

    TYPE_NAME( integer_type_node ) = build_decl(TYPE_DECL, get_identifier("int"), integer_type_node);
    TYPE_NAME( char_type_node ) = build_decl(TYPE_DECL, get_identifier("cchar"), char_type_node); // "char?"

    REALSIZE = int_size_in_bytes(long_double_type_node);
    PTRSIZE = int_size_in_bytes(ptr_type_node);
    switch (int_size_in_bytes(size_type_node)) {
    case 4:
	Tsize_t = Tuns32;
	break;
    case 8:
	Tsize_t = Tuns64;
	break;
    default:
	abort();
    }
    switch ( PTRSIZE ) {
    case 4:
	assert(POINTER_SIZE == 32);
	Tptrdiff_t = Tint32;
	break;
    case 8:
	assert(POINTER_SIZE == 64);
	Tptrdiff_t = Tint64;
	break;
    default:
	abort();
    }

    d_init_builtins();
    
    if (flag_exceptions) {
	eh_personality_libfunc = init_one_libfunc(d_using_sjlj_exceptions()
	    ? "__gdc_personality_sj0" : "__gdc_personality_v0");
	lang_eh_runtime_type = d_build_eh_type_type;
	using_eh_for_cleanups ();
	// lang_proctect_cleanup_actions = ...; // no need? ... probably needed for autos
    }

    /* copied and modified from cp/decl.c; only way for vtable to work in gdb... */
    // or not, I'm feeling very confused...
    if (1) {
	/* Make sure we get a unique function type, so we can give
	   its pointer type a name.  (This wins for gdb.) */
	tree vfunc_type = make_node (FUNCTION_TYPE);
	TREE_TYPE (vfunc_type) = Type::tint32->toCtype(); // integer_type_node; messed up built in types?
	TYPE_ARG_TYPES (vfunc_type) = NULL_TREE;
	layout_type (vfunc_type);

	tree vtable_entry_type = build_pointer_type (vfunc_type);
	d_vtbl_ptr_type_node = build_pointer_type(vtable_entry_type);
	layout_type (d_vtbl_ptr_type_node);// %%TODO: check if needed
    }

    // This also allows virtual functions to be called, but when vtbl entries,
    // are inspected, function symbol names do not appear.
    // d_vtbl_ptr_type_node = Type::tvoid->pointerTo()->pointerTo()->toCtype();

    // This is the C main, not the D main
    main_identifier_node = get_identifier ("main");

    bitConfig.setType(Type::tuns8);
}

void
gcc_d_backend_term()
{
}
