/* GDC -- D front-end for GCC
   Copyright (C) 2004 David Friedman
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "d-gcc-includes.h"
#include "d-lang.h"
#include "total.h"
#include "attrib.h"
#include "template.h"
#include "symbol.h"
#include "d-codegen.h"

Array d_bi_types;
Array d_bi_funcs;

static Type * gcc_type_to_d_type(tree t);

Type * d_gcc_builtin_va_list_d_type;

void
d_bi_init(int nt, int nb)
{
    d_bi_types.setDim(nt);
    d_bi_types.zero();
    d_bi_funcs.setDim(nb);
    d_bi_funcs.zero();

    // assumes va_list_type_node already built
    d_gcc_builtin_va_list_d_type = gcc_type_to_d_type(va_list_type_node);
    if (! d_gcc_builtin_va_list_d_type) {
	// fallback to array of byte of the same size?
	error("cannot represent built in va_list type in D");
	abort();
    }
    TYPE_MAIN_VARIANT( d_gcc_builtin_va_list_d_type->toCtype() ) =
	TYPE_MAIN_VARIANT( va_list_type_node );
}

/*
  Set ctype directly for complex types to save toCtype() the work.
  For others, it is not useful or, in the cast of (C char) -> (D char),
  will cause errors.  This also means char* ...
*/

static Type *
gcc_type_to_d_type(tree t)
{
    Type *d;
    switch (TREE_CODE(t)) {
    case POINTER_TYPE:
	d = gcc_type_to_d_type(TREE_TYPE(t));
	if (d) {
	    d = d->pointerTo();
	    return d;
	}
	break;
    case REFERENCE_TYPE:
	d = gcc_type_to_d_type(TREE_TYPE(t));
	if (d) {
	    // Want to assign ctype directly so that the REFERENCE_TYPE
	    // code can be turned into an InOut argument below.  Can't use
	    // pointerTo(), because that Type is shared.
	    d = new TypePointer(d);
	    d->ctype = t;
	    return d;
	}
	break;
    case INTEGER_TYPE:
	{
	    // DMD integer types come before char and bit...
	    unsigned sz = tree_low_cst( TYPE_SIZE_UNIT( t ), 1 );
	    bool unsgn = TREE_UNSIGNED( t );

	    // Make sure we get char instead of byte.  Otherwise, printf and
	    // the like won't be able to take char* arguments
	    if (sz == 1 && ! unsgn)
		return Type::tchar;
	    for (int i = 0; i < (int) TMAX; i++) {
		d = Type::basic[i];
		if (d && d->isintegral() && d->size() == sz &&
		    (d->isunsigned()?true:false) ==  unsgn) {
		    return d;
		}
	    }
	}
	break;
    case REAL_TYPE:
	{
	    unsigned sz = tree_low_cst( TYPE_SIZE_UNIT( t ), 1 );
	    for (int i = 0; i < (int) TMAX; i++) {
		d = Type::basic[i];
		if (d && d->isfloating() && ! d->iscomplex() && ! d->isimaginary() &&
		    d->size() == sz) {
		    return d;
		}
	    }
	}
	break;
    case COMPLEX_TYPE:
	{
	    unsigned sz = tree_low_cst( TYPE_SIZE_UNIT( t ), 1 );
	    for (int i = 0; i < (int) TMAX; i++) {
		d = Type::basic[i];
		if (d && d->iscomplex() && d->size() == sz) {
		    return d;
		}
	    }
	}
	break;
    case VOID_TYPE:
	return Type::tvoid;
    case ARRAY_TYPE:
	d = gcc_type_to_d_type(TREE_TYPE(t));
	if (d) {
	    tree index = TYPE_DOMAIN (t);
	    tree ub = TYPE_MAX_VALUE (index);
	    tree lb = TYPE_MIN_VALUE (index);
	    tree length	    
		= size_binop (PLUS_EXPR, size_one_node,
				 convert (sizetype,
					  fold (build (MINUS_EXPR,
						       TREE_TYPE (lb),
						       ub, lb))));
	    d = new TypeSArray(d, new IntegerExp(tree_low_cst(length, 1)));
	    d->ctype = t;
	    return d;
	}
	break;
    case RECORD_TYPE:
	// cheat by making a bye array of the same size -- this should
	// be okay for the builtin functions.
	d = new TypeSArray(Type::tuns8,
	    new IntegerExp( tree_low_cst( TYPE_SIZE_UNIT( t ), 1 )));
	d->ctype = t;
	return d;
    default:
	break;
    }
    return NULL;
}

void 
d_bi_builtin_type(int ti, tree t)
{
    Type * d = gcc_type_to_d_type(t);
    if (d) {
	d_bi_types.data[ti] = d;
    } else {
	// warning("cannot create builtin type %d\n", ti);
    }
}

void
d_bi_builtin_func_type(int ti, tree t, int ri, int ai1, int ai2, int ai3, int ai4, int va)
{
    Array * args = new Array;
    Type * rd = (Type *) d_bi_types.data[ri];
    if (! rd)
	return;
    
    int aia[4] = { ai1, ai2, ai3, ai4 };
    for (int i = 0; i < 4; i++) {
	Type * ad;
	int ai = aia[i];
	enum InOut io = In;
	
	if (ai == -1)
	    break;
	
	ad = (Type *) d_bi_types.data[ai];
	if (! ad)
	    return;

	if ( TREE_CODE( ad->toCtype() ) == REFERENCE_TYPE ) {
	    ad = ad->next;
	    io = InOut;
	}
	
	args->push( new Argument(io, ad, NULL, NULL) );
    }

    // DMD doesn't like this.  Nothing usable actually uses it, though
    if (args->dim == 0 && va)
	return;
    
    TypeFunction * dtf = new TypeFunction(args, rd, va, LINKc);
    dtf->ctype = t;
    // hope this is legit..    
    TYPE_LANG_SPECIFIC( t ) = build_d_type_lang_specific(dtf);
    d_bi_types.data[ti] = dtf;
}

void
d_bi_builtin_func(int bi, const char *name, int ti, tree decl)
{
    TypeFunction * df = (TypeFunction *) d_bi_types.data[ti];
    if (! df) {
	// warning("cannot create builtin %s\n", name);
	return;
    }
    FuncDeclaration * func = new FuncDeclaration(0, 0,
	Lexer::idPool(name), STCextern, df);
    func->csym = new Symbol;
    func->csym->Stree = decl;

    d_bi_funcs.data[bi] = func;
}

// std.stdarg is different: it expects pointer types (i.e. _argptr)
/*
  We can make it work fine as long as the argument to va_varg is _argptr,
  we just call va_arg on the hidden va_list.  As long _argptr is not
  otherwise modified, it will work. */

static void
d_gcc_magic_stdarg_module(Module *m, bool is_c_std_arg)
{
    Array * members = m->members;
    Identifier * id = Lexer::idPool("va_arg");
    Identifier * id_start = Lexer::idPool("va_start");
    for (unsigned i = 0; i < members->dim; i++) {
	TemplateDeclaration * td =
	    ((Dsymbol *) members->data[i])->isTemplateDeclaration();
	if (td) {
	    if (td->ident == id) {
		if (is_c_std_arg)
		    IRState::setCStdArgArg(td);
		else
		    IRState::setStdArg(td);
	    } else if (td->ident == id_start && is_c_std_arg) {
		IRState::setCStdArgStart(td);
	    } else
		continue;
	    
	    if ( TREE_CODE( TREE_TYPE( va_list_type_node )) == ARRAY_TYPE ) {
		/* For GCC, a va_list can be an array.  D static arrays are
		   automatically passed by reference, but the 'inout'
		   modifier is not allowed. */
		assert(td->members);
		for (unsigned j = 0; j < td->members->dim; j++) {
		    FuncDeclaration * fd =
			((Dsymbol *) td->members->data[j])->isFuncDeclaration();
		    if (fd && fd->ident == id) {
			TypeFunction * tf;

			// Should have nice error message instead of ICE in case some tries
			// to tweak the file.
			assert( ! fd->parameters );
			tf = (TypeFunction *) fd->type;
			assert( tf->ty == Tfunction &&
			    tf->arguments && tf->arguments->dim == 1 );
			((Argument*) tf->arguments->data[0])->inout = In;

			fd->isFuncDeclaration();
		    }
		}
	    }
	}
    }
}

static void
d_gcc_magic_builtins_module(Module *m)
{
    Array * funcs = new Array;

    for (unsigned i = 0; i < d_bi_funcs.dim; i++) {
	void * p = d_bi_funcs.data[i];
	if (p)
	    funcs->push( p );
    }

    Type * d = gcc_type_to_d_type(va_list_type_node);
    if (d) {
	funcs->push( new AliasDeclaration(0,
			   Lexer::idPool("__builtin_va_list"), d) );
    }

    // These four types are an experiment.  This allows Phobos (deh.d/unwind.d) to
    // be compiled correctly on 64-bit machines in a 32-bit environment

    // Don't know if 'long integer' is really the ABI word size -- it's a best guess
    d = gcc_type_to_d_type(long_integer_type_node);
    if (d)
	funcs->push( new AliasDeclaration(0,
			 Lexer::idPool("__builtin_abi_int"), d) );
    d = gcc_type_to_d_type(long_unsigned_type_node);
    if (d)
	funcs->push( new AliasDeclaration(0,
			 Lexer::idPool("__builtin_abi_uint"), d) );
    d = gcc_type_to_d_type(d_type_for_size(UNITS_PER_WORD*BITS_PER_UNIT, 0));
    if (d)
	funcs->push( new AliasDeclaration(0,
			 Lexer::idPool("__builtin_machine_int"), d) );
    d = gcc_type_to_d_type(d_type_for_size(UNITS_PER_WORD*BITS_PER_UNIT, 1));
    if (d)
	funcs->push( new AliasDeclaration(0,
			 Lexer::idPool("__builtin_machine_uint"), d) );

    m->members->push( new LinkDeclaration(LINKc, funcs) );
}


void d_gcc_magic_module(Module *m)
{
    ModuleDeclaration * md = m->md;
    if (md && md->packages && md->id) {
	if (md->packages->dim == 1 &&
	    ! strcmp( ((Identifier *) md->packages->data[0])->string, "gcc" ) &&
	    ! strcmp( md->id->string, "builtins" )) {
	    d_gcc_magic_builtins_module(m);
	} else if (md->packages->dim >= 1 &&
	    ! strcmp( ((Identifier *) md->packages->data[0])->string, "std" )) {

	    if (! strcmp( md->id->string, "stdarg" )) {
		if (md->packages->dim == 1) {
		    d_gcc_magic_stdarg_module(m, false);
		} else if (md->packages->dim == 2 &&
		    ! strcmp( ((Identifier *) md->packages->data[1])->string, "c" )) {
		    d_gcc_magic_stdarg_module(m, true);
		}
	    } else if (! strcmp( md->id->string, "intrinsic") &&
		md->packages->dim == 1) {
		IRState::setIntrinsicModule(m);
	    }
	}
    }
}
