/*
   CIPE - encrypted IP over UDP tunneling

   lebf.c - glue between our Blowfish implentation and the crypto API

   Copyright 2001 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: lebf.c,v 1.9 2004/08/03 08:21:17 olaf81825 Exp $ */

#include "cipe.h"
#include "bf.h"

int lebf_cra_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
    printk(KERN_DEBUG "lebf_cra_setkey: ctx=%p\n", ctx);
    Blowfish_ExpandUserKey(key, keylen, ctx);
    return 0;
}

int lebf_cit_setkey(struct crypto_tfm *tfm,
                    const u8 *key, unsigned int keylen)
{
    printk(KERN_DEBUG "lebf_cit_setkey: tfm=%p ctx=%p\n", tfm, crypto_tfm_ctx(tfm));
    Blowfish_ExpandUserKey(key, keylen, crypto_tfm_ctx(tfm));
    return 0;
}

void lebf_encrypt(void *ctx, u8 *dst, const u8 *src) {
    L_Blowfish_Encrypt((void *)src, dst, ctx);
}

void lebf_decrypt(void *ctx, u8 *dst, const u8 *src) {
    L_Blowfish_Decrypt((void *)src, dst, ctx);
}


/* This holds the bare minimum needed for CIPEs internal purposes.
   NOT for registering with the crypto API. */
static struct crypto_alg LEBF_ALG = {
    .cra_flags = CRYPTO_ALG_TYPE_CIPHER,
    .cra_blocksize = 8,
    .cra_ctxsize = sizeof(Blowfish_Key),
    .cra_name = "cipe-le-blowfish-internal",
    .cra_cipher = {
    .cia_setkey = lebf_cra_setkey,
    .cia_encrypt = lebf_encrypt,
    .cia_decrypt = lebf_decrypt,
    },
};
static struct cipher_tfm LEBF_CIPHER = {
    .cit_ivsize = 8,
    .cit_setkey = lebf_cit_setkey,
};

struct crypto_tfm *cipe_alloc_tfm(const char *alg_name, u32 tfm_flags)
{
    printk(KERN_DEBUG "cipe_alloc_tfm '%s'\n", alg_name);
    if (!alg_name || !*alg_name || !strcmp(alg_name, "blowfish-internal")) {
        struct crypto_tfm *tfm;
        tfm=kmalloc(sizeof(*tfm)+sizeof(Blowfish_Key), GFP_KERNEL);
        if (tfm) {
            tfm->crt_cipher = LEBF_CIPHER;
            tfm->__crt_alg = &LEBF_ALG;
        }
        printk(KERN_DEBUG "cipe_alloc_tfm: tfm=%p\n", tfm);
        return tfm;
    }
#ifdef USE_CRYPTOAPI
    return crypto_alloc_tfm(alg_name, tfm_flags);
#else
    return NULL;
#endif
}

void cipe_free_tfm(struct crypto_tfm *tfm)
{
    if (tfm->__crt_alg == &LEBF_ALG) {
        memset(tfm, 0, sizeof(*tfm)+sizeof(Blowfish_Key));
        kfree(tfm);
    } else {
#ifdef USE_CRYPTOAPI
        crypto_free_tfm(tfm);
#else
        BUG();
#endif
    }
}

/* Fill a block of length blockSize with strong random numbers.
   Used for generating IVs. */
void cipe_cryptpad(unsigned char *buf, int len)
{
    static int padcnt=MAXBLKS;
    static Blowfish_Key padkey;

    if (++padcnt>MAXBLKS) {
	/* make a new random key */
	Blowfish_UserKey k;
	dprintk(DEB_CRYPT, (KERN_INFO "%s: re-keying cryptpad\n", DEVNAME));
	cipe_prnpad((unsigned char*)k, sizeof(k));
	Blowfish_ExpandUserKey(k, sizeof(k), padkey);
	padcnt=0;
    }
    *(int *)(buf)=padcnt;
    cipe_prnpad(buf+sizeof(int), len-sizeof(int));
    _N_Blowfish_Encrypt(buf, buf, padkey);
    /* it doesn't matter if we use big or little endian here */
}

