/*
   CIPE - encrypted IP over UDP tunneling

   cryptoapi.h - cryptographic API glue

   Copyright 2001 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: cryptoapi.h,v 1.6 2004/08/03 08:21:17 olaf81825 Exp $ */

#ifndef _CRYPTOAPI_H_
#define _CRYPTOAPI_H_

#ifndef _CIPE_H_
#error "This file must be included by cipe.h only."
#endif

#ifdef HAVE_CRYPTO_H

#include <linux/crypto.h>

#else

/* Crypto API not available in the kernel. Emulate just enough of it
   to satisfy our needs. Support our Blowfish module only.
*/

#define CRYPTO_ALG_TYPE_MASK		0x000000ff
#define CRYPTO_ALG_TYPE_CIPHER		0x00000001
#define CRYPTO_TFM_MODE_CBC		0x00000002
#define CRYPTO_MAX_ALG_NAME		64

struct cipher_alg {
    int (*cia_setkey)(void *ctx, const u8 *key,
                      unsigned int keylen, u32 *flags);
    void (*cia_encrypt)(void *ctx, u8 *dst, const u8 *src);
    void (*cia_decrypt)(void *ctx, u8 *dst, const u8 *src);
};

struct crypto_alg {
    u32 cra_flags;
    unsigned int cra_blocksize;
    unsigned int cra_ctxsize;
    const char cra_name[CRYPTO_MAX_ALG_NAME];
    struct cipher_alg cra_cipher;
};

struct crypto_tfm;

struct cipher_tfm {
    unsigned int cit_ivsize;
    int (*cit_setkey)(struct crypto_tfm *tfm,
                      const u8 *key, unsigned int keylen);
};

struct crypto_tfm {
    struct cipher_tfm crt_cipher;
    struct crypto_alg *__crt_alg;
};

struct crypto_tfm *crypto_alloc_tfm(const char *alg_name, u32 tfm_flags);
void crypto_free_tfm(struct crypto_tfm *tfm);

static inline u32 crypto_tfm_alg_type(struct crypto_tfm *tfm)
{
	return tfm->__crt_alg->cra_flags & CRYPTO_ALG_TYPE_MASK;
}

static inline unsigned int crypto_tfm_alg_ivsize(struct crypto_tfm *tfm)
{
	BUG_ON(crypto_tfm_alg_type(tfm) != CRYPTO_ALG_TYPE_CIPHER);
	return tfm->crt_cipher.cit_ivsize;
}

static inline unsigned int crypto_tfm_alg_blocksize(struct crypto_tfm *tfm)
{
	return tfm->__crt_alg->cra_blocksize;
}

static inline int crypto_cipher_setkey(struct crypto_tfm *tfm,
                                       const u8 *key, unsigned int keylen)
{
	BUG_ON(crypto_tfm_alg_type(tfm) != CRYPTO_ALG_TYPE_CIPHER);
	return tfm->crt_cipher.cit_setkey(tfm, key, keylen);
}

#endif

static inline void *crypto_tfm_ctx(struct crypto_tfm *tfm)
{
    return (void *)&tfm[1];
}

static inline void crypto_cipher_encrypt_blk(struct crypto_tfm *tfm,
                                             u8 *dst,
                                             const u8 *src)
{
    BUG_ON(crypto_tfm_alg_type(tfm) != CRYPTO_ALG_TYPE_CIPHER);
    tfm->__crt_alg->cra_cipher.cia_encrypt(crypto_tfm_ctx(tfm), dst, src);
}

static inline void crypto_cipher_decrypt_blk(struct crypto_tfm *tfm,
                                             u8 *dst,
                                             const u8 *src)
{
    BUG_ON(crypto_tfm_alg_type(tfm) != CRYPTO_ALG_TYPE_CIPHER);
    tfm->__crt_alg->cra_cipher.cia_decrypt(crypto_tfm_ctx(tfm), dst, src);
}

struct crypto_tfm *cipe_alloc_tfm(const char *alg_name, u32 tfm_flags);
void cipe_free_tfm(struct crypto_tfm *tfm);

#endif
