/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#include "gStuff.h"
#include "eGrid.h"
#include "eWall.h"
#include "math.h"
#include "gCycle.h"
#include "rTexture.h"
#include "eTimer.h"
#include "gGame.h"
#include "rScreen.h"
#include "gWall.h"
#include "rRender.h"
#include "eCamera.h"
#include "tConfiguration.h"
#include "gExplosion.h"
#include "tMath.h"

/* **********************************************
   Wall
   ********************************************** */

#ifndef DEDICATED

//texture ArmageTron_invis_eWall("eWall2.png",1,0);
static rTexture gWallRim_text(rTEX_WALL,"textures/rim_wall.png",1,1);
//static rTexture gWallRim_text_moviepack(rTEX_WALL,"moviepack/gWallRim2.png",1,0);
static rTexture gWallRim_a(rTEX_WALL,"moviepack/rim_wall_a.png",0,0);
static rTexture gWallRim_b(rTEX_WALL,"moviepack/rim_wall_b.png",0,0);
static rTexture gWallRim_c(rTEX_WALL,"moviepack/rim_wall_c.png",0,0);
static rTexture gWallRim_d(rTEX_WALL,"moviepack/rim_wall_d.png",0,0);

static rTexture *gWallRim_mp[4]={&gWallRim_a,&gWallRim_b,
								 &gWallRim_c,&gWallRim_d};

static rTexture dir_eWall(rTEX_WALL,"textures/dir_wall.png",1,0);
static rTexture dir_eWall_moviepack(rTEX_WALL,"moviepack/dir_wall.png",1,0);

#endif

static REAL mp_eWall_stretch=4;
static tSettingItem<REAL> mpws
("MOVIEPACK_WALL_STRETCH",mp_eWall_stretch);

#ifndef DEDICATED
static void dir_eWall_select(){
	if (sg_MoviePack()){
		TexMatrix();
		IdentityMatrix();
		ScaleMatrix(1/mp_eWall_stretch,1,1);
		dir_eWall_moviepack.Select();
	}
	else
		dir_eWall.Select();
}


#endif

/* **********************************************
   RimWall
   ********************************************** */

gWallRim::gWallRim(eGrid *grid, bool backface_cull,REAL h)
	:eWallRim(grid, backface_cull, h){}

gWallRim::~gWallRim(){}
  
bool gWallRim::Splittable() const{return 1;}
void gWallRim::Split(eWall *& w1,eWall *& w2,REAL){
	w1=tNEW(gWallRim(grid));
	w2=tNEW(gWallRim(grid));
}

// from display.C
extern REAL lower_height,upper_height;

#ifndef DEDICATED


static void gWallRim_helper(eCoord p1,eCoord p2,REAL tBeg,REAL tEnd,REAL h,
							REAL Z_SCALE,bool sw){

	if (sg_MoviePack()){
		int t=int(floor((tBeg+tEnd)/2));
		tBeg-=t;
		tEnd-=t;
		t=t%4;
		gWallRim_mp[t]->Select();
	}

	if (sw){
		Swap(p1,p2);
		Swap(tBeg,tEnd);
	}

	if (h>1000){
		if (sr_lowerSky || sg_MoviePack()) h=lower_height;
		if (sr_upperSky && !sg_MoviePack()) h=upper_height;
	}

	if (h<100 || !sr_infinityPlane){
		BeginQuads();

		IsEdge(false);
		TexVertex(p1.x, p1.y, 0,
				  tBeg      , 1);
    
		IsEdge(true);
		TexVertex(p1.x, p1.y, h,
				  tBeg,       1-h/Z_SCALE);
    
		IsEdge(false);
		TexVertex(p2.x, p2.y, h,
				  tEnd,       1-h/Z_SCALE);

		TexVertex(p2.x, p2.y, 0,
				  tEnd      , 1);

		RenderEnd();
	}

	else{
		BeginTriangles();
    
		IsEdge(false);
		TexVertex(p1.x, p1.y, 0,
				  tBeg,       1);
    
		IsEdge(true);
		TexCoord(0,-1/REAL(Z_SCALE),0,0);
    
#ifndef WIN32
		Vertex(0,0,1,0);
#else  
		Vertex(0.001f,0.001f,1,0); // Windows OpenGL has problems with
		// infitite points perpenticular to the viewing direction
#endif
    
		TexVertex(p2.x, p2.y, 0,
				  tEnd,       1);
		RenderEnd();
	}
}

void gWallRim::RenderReal(){
	if ( Edge() ){
		const eCoord *p1=&EndPoint(0);
		const eCoord *p2=&EndPoint(1);

   
		if(bf_cull)
			glDisable(GL_CULL_FACE);

		if (fabs(p1->x - p2->x) < .001)
			Color(.7, .7, .7);
		else
			Color(1, 1, 1);

		REAL X_SCALE=100;
		REAL Z_SCALE=100;


		if (sg_MoviePack()){
			X_SCALE=50;
			Z_SCALE=50;
		}



		REAL tBeg=(p1->x+p1->y)/X_SCALE;
		REAL tEnd=(p2->x+p2->y)/X_SCALE;
		eCoord P1=*p1;
		eCoord P2=*p2;


		if (sg_MoviePack()){
			bool sw=false;
      
			if (tBeg>tEnd){
				Swap(P1,P2);
				Swap(tBeg,tEnd);
				//sw=true;
			}
      

			REAL ta=tBeg;
			eCoord ca=P1;
			for(int i=int(ceil(tBeg));i<tEnd;i++){
				eCoord cb=P1+(P2-P1)*((i-tBeg)/(tEnd-tBeg));
				gWallRim_helper(ca,cb,ta,i,height,Z_SCALE,sw);
				ca=cb;
				ta=i;
			}
			gWallRim_helper(ca,P2,ta,tEnd,height,Z_SCALE,sw);
		}
		else{
			gWallRim_text.Select();
			gWallRim_helper(*p1,*p2,tBeg,tEnd,height,Z_SCALE,false);
		}

		//eWall::Render_helper(edge,(p1->x+p1->y)/SCALE,(p2->x+p2->y)/SCALE,40,height);

		if(bf_cull)
			glEnable(GL_CULL_FACE);
	}
}

#endif

/* **********************************************
   PlayerWall
   ********************************************** */

#ifdef DEBUG
#define CHECKWALL this->Check();
#else
#define CHECKWALL 
#endif

gPlayerWall::gPlayerWall(gNetPlayerWall*w, gCycle *p)
	:eWall(p->grid),cycle_(p),netWall_(w),begAlpha_(0.0f),endAlpha_(1.0f)
{
	CHECKWALL;

	if (cycle_)
		windingNumber_ = cycle_->WindingNumber();

#ifdef DEBUG
	if (!cycle_)
	{
//		st_Breakpoint();
	}
#endif
}

gPlayerWall::~gPlayerWall(){
	CHECKWALL;
}
  
//ArmageTron_eWalltype gPlayerWall::type(){return ArmageTron_PLAYERWALL;}
  
void gPlayerWall::Flip(){
	CHECKWALL;

	eWall::Flip();
	Swap( this->begAlpha_, this->endAlpha_ );

	CHECKWALL;
}
  
bool gPlayerWall::Splittable() const{return 1;}

bool gPlayerWall::Deletable() const{
	CHECKWALL;

	// a wall without a cycle can clearly be deleted
	if ( !cycle_ )
		return true;

	// is the player dead?
	if ( gCycle::WallsStayUpDelay() >= 0 )
	{
		if ( !cycle_->Alive() && se_GameTime() - cycle_->deathTime > 1 + gCycle::WallsStayUpDelay() )
			return true;
	}

	// is the wall behind the wall end?
	if ( gCycle::WallsLength() > 0 )
	{
		REAL maxDist = cycle_->distance - gCycle::WallsLength();
		if ( maxDist > EndPos()  && maxDist > BegPos() )
			return true;
	}

	return false;
}

/*
static void S_Mix( const gPlayerWallCoord source[2], REAL alpha, gPlayerWallCoord& target )
{
	REAL diff  = ( source[1].Alpha - source[0].Alpha );
	REAL ralpha = 0;
	if ( diff > 0 )
		ralpha     = ( alpha - source[0].Alpha ) / diff;

	target.Alpha = alpha;
	target.Pos   = source[0].Pos  + ralpha * ( source[1].Pos  - source[0].Pos  );
	target.Pos   = source[0].Time + ralpha * ( source[1].Time - source[0].Time );
}
*/

void gPlayerWall::Split(eWall *& w1,eWall *& w2,REAL a){
	CHECKWALL;

	gPlayerWall *W1, *W2;

	W1=tNEW(gPlayerWall(netWall_,cycle_));
	W2=tNEW(gPlayerWall(netWall_,cycle_));
	W1->windingNumber_ = windingNumber_;
	W2->windingNumber_ = windingNumber_;
	W1->begAlpha_ = begAlpha_;
	W2->endAlpha_ = endAlpha_;
	W1->endAlpha_ = W2->begAlpha_ = begAlpha_ + ( endAlpha_ - begAlpha_ ) * a;

/*
	int divindex = IndexPos( mp );
	int i;

	// transfer front points
	W1->coords_.SetLen( divindex + 2 );
	for ( i = divindex+1; i>=0; --i )
		W1->coords_(i) = coords_(i);

	W1->coords_(divindex+1).Pos  = mp;
	W1->coords_(divindex+1).Time = mt;

	// transfer rear points
	W2->coords_.SetLen( coords_.Len() - divindex );
	for ( i = coords_.Len() - divindex - 1 ; i>=0; --i )
		W2->coords_(i) = coords_( divindex + i );

	W2->coords_(0).Pos  = mp;	
	W2->coords_(0).Time = mt;

	if ( flipped )
		Swap( W1, W2 );
*/

	// store wall pointers
	w1 = W1;
	w2 = W2;

#ifdef DEBUG
	W1->Check();
	W2->Check();
#endif

	CHECKWALL;
}

//#define gBEG_LEN 2
//#define gBEG_LEN .25
//#define gCYCLE_LEN .95
//#define gCYCLE_LEN 1.6
#define gCYCLE_LEN 1.5
#define gBEG_OFFSET .25
#define gBEG_LEN 2
//#define gCYCLE_LEN 3.8
//#define gBEG_OFFSET 1

#ifndef DEDICATED
void gPlayerWall::Render(){
	if (!cycle_)
		return;
	RenderList(true);
}

void gNetPlayerWall::Render(){
	if (!cycle_)
		return;
	RenderList(true);
}

void gPlayerWall::RenderList(bool list)
{
	netWall_->RenderList( list );
}

void gNetPlayerWall::RenderList(bool list){
	if (displayList_ && cycle_->Alive() && list)
		glCallList(displayList_);
	else 
	{  
		dir_eWall_select();

		REAL r,g,b;
		if (cycle_){
			r=cycle_->trailColor_.r;
			g=cycle_->trailColor_.g;
			b=cycle_->trailColor_.b;
		}
		else
			r=g=b=1;

		eCoord P1=EndPoint(0);
		eCoord P2=EndPoint(1);
    
		if (fabs(P1.x - P2.x) < .5*fabs(P1.y - P2.y)){
			r*=REAL(.7);
			g*=REAL(.7);
			b*=REAL(.7);
		}

		REAL a=1;

#define SEGLEN 2.5
		//REAL ta=startTime*3;
		//REAL te=endTime*3;
		for ( int i = coords_.Len()-2; i>=0; --i )
		{
			const gPlayerWallCoord* coord = &coords_(i);

			if ( !coord[0].IsDangerous )
				continue;
			
			REAL pa = coord[0].Pos;
			REAL pe = coord[1].Pos;

			REAL aa = Alpha( pa );
			REAL ae = Alpha( pe );

			eCoord p1 = P1 + ( P2 - P1 ) * aa;
			eCoord p2 = P1 + ( P2 - P1 ) * ae;

			REAL ta=pa/SEGLEN;
			REAL te=pe/SEGLEN;
			//REAL shift=REAL(floor((ta+te)/20)*10);
    
			//REAL time=ArmageTronTimer*3;
			REAL time;
			if (cycle_)
			{
				time=cycle_->distance/SEGLEN;
				if ( !cycle_->Alive() )
					time += se_GameTime() - cycle_->deathTime;
			}
			else
				time=0;

		//ta-=shift;
		//te-=shift;
		//time-=shift;
    
			if (ta>te){
				Swap(ta,te);
				Swap(p1,p2);
				Swap(pa,pe);
			}

			// cut the end of the wall
			if ( bool(cycle_) && gCycle::WallsLength() > 0 )
			{
				REAL cut = (cycle_->distance - gCycle::WallsLength() - pe) / ( pa - pe );
				if ( cut < 0 )
					continue;
				if ( cut < 1 )
				{
					p1 = p2 + (p1-p2)*cut;
					ta = te + (ta-te)*cut;
				}
			}

			if (te+gBEG_LEN<=time){
				/*      if (!ePlayer->Alive())
						RenderNormal(p1,p2,ta,te,r,g,b,a);
						else if (list){
						displayList=glGenLists(1);
						glNewList(displayList,GL_COMPILE_AND_EXECUTE);
						RenderNormal(p1,p2,ta,te,r,g,b,a);
						glEndList();
						}
						else */
				RenderNormal(p1,p2,ta,te,r,g,b,a);
			}
    
			else{ // complicated
				if (ta+gBEG_LEN>=time){
					RenderBegin(p1,p2,ta,te,
								1+(ta-time)/gBEG_LEN,
								1+(te-time)/gBEG_LEN,
								r,g,b,a);
				}
				else{
					REAL s=((time-gBEG_LEN)-ta)/(te-ta);
					eCoord pm=p1+(p2-p1)*s;
					RenderBegin(pm,p2,
								ta+(te-ta)*s,te,0,
								1+(te-time)/gBEG_LEN,
								r,g,b,a);
					RenderNormal(p1,pm,ta,ta+(te-ta)*s,r,g,b,a);
				}
			}
		}
	}
}
	

bool upperlinecolor(REAL r,REAL g,REAL b){
	if (TextureMode[rTEX_WALL]<0)
		glColor3f(1,1,1);
	else{
		/*
		  REAL upperline_alpha=fabs(se_cameraRise*2);
		  upperline_alpha-=1;
		  if (upperline_alpha>.5)
		  upperline_alpha=1;
		  if (upperline_alpha<=.5)
		  return false;
		  glColor4f(r,g,b,upperline_alpha);
		*/
		//glDisable(GL_TEXTURE);
		glDisable(GL_TEXTURE_2D);
		glColor3f(r,g,b);
	}
	return true;
}

void gNetPlayerWall::RenderNormal(const eCoord &p1,const eCoord &p2,REAL ta,REAL te,REAL r,REAL g,REAL b,REAL a){
	REAL hfrac=1;

	if (bool(cycle_) && !cycle_->Alive() && gCycle::WallsStayUpDelay() >= 0 ){
		REAL dt=(se_GameTime()-cycle_->deathTime-gCycle::WallsStayUpDelay())*2;
		if (dt>1) dt=1;
		if (dt>=0)
		{
			REAL ca=REAL(.5/(dt+.5));
			REAL alpha=1-dt;
			if (alpha>1) alpha=1;
			hfrac=1-dt;

			r+=ca;
			b+=ca;
			g+=ca;

			a*=alpha;
		}
	}      
	REAL h=1;


	if (hfrac>0){
		if(upperlinecolor(r,g,b)){
      
			// draw additional upper line
			BeginLines();
      
			sr_DepthOffset(true);
			glVertex3f(p1.x,p1.y,h*hfrac);
			glVertex3f(p2.x,p2.y,h*hfrac);
			sr_DepthOffset(false);
      
			RenderEnd();
			glDisable(GL_POLYGON_OFFSET_LINE);
		}
    
		//glColor4f(r,g,b,a);
    
		dir_eWall_select();
    
		glColor3f(r,g,b);
  
#ifdef XDEBUG
		REAL extrarise = 0;
		if ( this->id >= 0 )
		{
			extrarise = 1;
		}
#else
 		static const REAL extrarise = 0;
#endif 
		BeginQuads();
		glEdgeFlag(GL_FALSE);
		glTexCoord2f(ta,hfrac);
		glVertex3f(p1.x,p1.y,extrarise);
    
		glEdgeFlag(GL_TRUE);
		glTexCoord2f(ta,0);
		glVertex3f(p1.x,p1.y,extrarise + h*hfrac);
    
		glEdgeFlag(GL_FALSE);
		glTexCoord2f(te,0);
		glVertex3f(p2.x,p2.y,extrarise + h*hfrac);
    
		glTexCoord2f(te,hfrac);
		glVertex3f(p2.x,p2.y,extrarise);
		RenderEnd();
	}
}

static inline REAL hfunc(REAL x){return 1-(x*x)/2;}
//static inline REAL hfunc(REAL x){return 1-(x*x);}
static inline REAL cfunc(REAL x){return (x*x);}
//static inline REAL afunc(REAL x){return 1-(x*x)/2;}
static inline REAL afunc(REAL x){return 1-(x*x);}
static inline REAL sfunc(REAL x){return (x*x);}
//static inline REAL xfunc(REAL x){return (x+x*x)/2;}
static inline REAL xfunc(REAL x){return REAL((x*.2+x*x)/2);}

void gNetPlayerWall::RenderBegin(const eCoord &p1,const eCoord &pp2,REAL ta,REAL te,REAL ra,REAL re,REAL r,REAL g,REAL b,REAL a){
	REAL hfrac=1;

	eCoord p2 = pp2;

	if (re > 1){
		if (re > 2)
			return;

		REAL ratio = (1-ra)/(re-ra);
		p2 = p1 + (pp2-p1)*ratio;
		te = ta + (te-ta)*ratio;
		re= 1;
	}

	if (bool(cycle_) && !cycle_->Alive()){
		REAL dt=(se_GameTime()-cycle_->deathTime-gCycle::WallsStayUpDelay())*2;
		if (dt>1) dt=1;
		if (dt>0)
		{
			REAL ca=REAL(.5/(dt+.5));
			REAL alpha=1-dt;
			if (alpha>1) alpha=1;
			hfrac=1-dt;

			r+=ca;
			b+=ca;
			g+=ca;
		}
		//a*=alpha;
	}      

	REAL h=1;

	eCoord ppos=cycle_->pos-cycle_->dir*REAL(gCYCLE_LEN);

	if (hfrac>0 && upperlinecolor(r,g,b)){
		sr_DepthOffset(true);
		//REAL H=h*hfrac;
#define segs 5
		BeginLineStrip();
    
		for(int i=0;i<=segs;i++){
			REAL frag=i/float(segs);
			REAL rat=ra+frag*(re-ra);
			REAL x=(p1.x+frag*(p2.x-p1.x))*(1-xfunc(rat))+ppos.x*xfunc(rat);
			REAL y=(p1.y+frag*(p2.y-p1.y))*(1-xfunc(rat))+ppos.y*xfunc(rat);
      
			REAL H=h*hfrac*hfunc(rat);
			glVertex3f(x+H*cycle_->skew*sfunc(rat)*cycle_->dir.y,
					   y-H*cycle_->skew*sfunc(rat)*cycle_->dir.x,
					   H);//+se_cameraZ*.005);
		}
		RenderEnd();
		sr_DepthOffset(false);
	}
    
	dir_eWall_select();
  
  
	BeginQuadStrip();
    
    
    
    //REAL H=h*hfrac;
    
    //ppos=ePlayer->pos-ePlayer->dir*gCYCLE__LEN;
    
	for(int i=0;i<=segs;i++){
		REAL frag=i/float(segs);
		REAL rat=ra+frag*(re-ra);
		REAL x=(p1.x+frag*(p2.x-p1.x))*(1-xfunc(rat))+ppos.x*xfunc(rat);
		REAL y=(p1.y+frag*(p2.y-p1.y))*(1-xfunc(rat))+ppos.y*xfunc(rat);

		// bottom
		glEdgeFlag(GL_FALSE);
		glColor4f(r+cfunc(rat),g+cfunc(rat),b+cfunc(rat),a*afunc(rat));
		glTexCoord2f(ta+(te-ta)*frag,hfrac);
		glVertex3f(x,y,0);

		// top

		glEdgeFlag(GL_TRUE);
		//glTexCoord2f(ta+(te-ta)*frag,hfrac*(1-hfunc(rat)));
		glTexCoord2f(ta+(te-ta)*frag,0);
		REAL H=h*hfrac*hfunc(rat);
		glVertex3f(x+H*cycle_->skew*sfunc(rat)*cycle_->dir.y,
				   y-H*cycle_->skew*sfunc(rat)*cycle_->dir.x,
				   H);
	}
	RenderEnd();
}
#endif

void gNetPlayerWall::SetEndTime(REAL t){
	CHECKWALL;

	REAL BegTime = coords_( coords_.Len() -2 ).Time;
	if ( t < BegTime )
	{
		t = BegTime;
	}

	coords_(coords_.Len()-1).Time = t;

	CHECKWALL;

}

void gNetPlayerWall::SetEndPos(REAL ep){
	CHECKWALL;

	REAL BegPos = coords_( coords_.Len() -2 ).Pos;
	if ( ep < BegPos )
	{
		ep = BegPos;
	}

	coords_(coords_.Len()-1).Pos = ep;

	CHECKWALL;
}

REAL gPlayerWall::BlockHeight() const{
	if (bool(cycle_) && cycle_->Alive()==1)
		return 4;
	else
		return 0;
}

REAL gPlayerWall::SeeHeight() const{
	return BlockHeight();
}


gCycle *gPlayerWall::Cycle() const {return cycle_;}
gNetPlayerWall *gPlayerWall::NetWall() const {return netWall_;}

void gPlayerWall::Insert()
{
	CHECKWALL;

	eWall::Insert();
}

void gPlayerWall::Check() const
{
//	netWall_->Check();
}

REAL gPlayerWall::LocalToGlobal( REAL a ) const
{
	tASSERT( good( a ) );

	REAL ret = this->begAlpha_ + ( this->endAlpha_ - this->begAlpha_ ) * a;

	tASSERT( good( ret ) );

	return ret;
}

REAL gPlayerWall::GlobalToLocal( REAL a ) const
{
	tASSERT( good( a ) );

	REAL div = ( this->endAlpha_ - this->begAlpha_ );
	if ( div == 0 )
	{
		return .5f;
	}
	else
	{
		REAL ret = ( a - begAlpha_ ) / div;

		tASSERT( good( ret ) );

		return ret;
	}
}

REAL gPlayerWall::Time(REAL a) const
{
	tASSERT( good( a ) );

	return netWall_->Time( LocalToGlobal( a ) );
}

REAL gPlayerWall::Pos(REAL a) const
{
	CHECKWALL;

	tASSERT( good( a ) );

	return netWall_->Pos( LocalToGlobal( a ) );
}

REAL gPlayerWall::Alpha(REAL pos) const
{
	CHECKWALL;

	return GlobalToLocal( netWall_->Alpha( pos ) );
}

bool gPlayerWall::IsDangerous( REAL a ) const
{
	CHECKWALL;

	return netWall_->IsDangerous( LocalToGlobal( a ) );
}

REAL gPlayerWall::EndPos() const
{
	CHECKWALL;

	return netWall_->Pos( LocalToGlobal( this->endAlpha_ ) );
}

REAL gPlayerWall::BegPos() const
{
	CHECKWALL;

	return netWall_->Pos( LocalToGlobal( this->begAlpha_ ) );
}

REAL gPlayerWall::EndTime() const
{
	CHECKWALL;

	return netWall_->Time( LocalToGlobal( this->endAlpha_ ) );
}

REAL gPlayerWall::BegTime() const
{
	CHECKWALL;

	return netWall_->Time( LocalToGlobal( this->begAlpha_ ) );
}

void gPlayerWall::BlowHole	( REAL beg, REAL end )
{
	CHECKWALL;
	
	this->netWall_->BlowHole( beg, end );
}

/*
void gPlayerWall::Clamp	(  )
{
	tASSERT( coords.Len() >= 2 );

	// clamp beginning
	int begin = IndexAlpha(0.0f);
	gPlayerWallCoord* bcoord = &coords[begin];
	S_Mix( bcoord, 0.0f, bcoord[0] );

	// clamp end
	int end = IndexAlpha(1.0f);
	gPlayerWallCoord* ecoord = &coords[end];
	S_Mix( ecoord, 1.0f, ecoord[1] );
	
	// useful coordinates now lie between begin and end+1

	// throw away junk at the beginning
	if ( begin > 0 )
	{
		for ( int i = 0; i <= end + 1 - begin; ++i )
			coords[i] = coords[ i + begin ];
	}

	// throw away junk at the end
	coords.SetLen( end - begin + 2 );
}
*/

// ************************************************
// ************************************************


tList<gNetPlayerWall> sg_netPlayerWalls;
tList<gNetPlayerWall> gridded_sg_netPlayerWalls;


void gNetPlayerWall::CreateEdge()
{
	if ( this->edge_ )
		return;

	if (this->cycle_)
	{
		gPlayerWall* w = tNEW(gPlayerWall)(this,
										   this->cycle_);

		this->edge_=tNEW(eTempEdge)(beg,
									end,
									w );
	}
	else
	{
		this->edge_ = NULL;
		return;
	}
}

void gNetPlayerWall::InitAfterCreation()
{
	nNetObject::InitAfterCreation();
	MyInitAfterCreation();
}

void gNetPlayerWall::InitArray()
{
	REAL ep = dbegin+sqrt((beg-end).NormSquared());
	REAL sp = dbegin;

	if ( ep < sp )
	{
		ep = sp;
	}

	if ( tEnd < tBeg )
	{
		tEnd = tBeg;
	}

	coords_.SetLen(2);
	coords_[0].Pos   		= sp;
	coords_[0].Time  		= tBeg;
	coords_[0].IsDangerous   = true;
	coords_[1].Pos   		= ep;
	coords_[1].Time  		= tEnd;
	coords_[1].IsDangerous   = true;
}

void gNetPlayerWall::MyInitAfterCreation()
{
	//w=
#ifdef DEBUG
	if (!finite(end.x) || !finite(end.y))
		st_Breakpoint();

	if (!finite(beg.x) || !finite(beg.y))
		st_Breakpoint();
#endif

	if ( coords_.Len() < 2 )
	{
		InitArray();
	}

	displayList_ = 0;

	CreateEdge();

	id=-1;
	griddedid=-1;
	sg_netPlayerWalls.Add(this,id);

	if ( !Wall() )
		return; 
	tASSERT( Wall()->Splittable() );

	for(int i=MAX_VIEWERS-1;i>=0;i--)
		Wall()->SetVisHeight(i,0);

	Wall()->Remove();
}



gNetPlayerWall::gNetPlayerWall(gCycle *cyc,
							   const eCoord &begi,const eCoord &d,
							   REAL tBegi, REAL dbeg)
	:nNetObject(-1),
	 id(-1),griddedid(-1),
	 cycle_(cyc),dir(d),dbegin(dbeg),
	 beg(begi),end(begi),tBeg(tBegi),tEnd(tBegi),
	 inGrid(false){
	dir=dir*REAL(1/sqrt(dir.NormSquared()));
	preliminary=(sn_GetNetState()==nCLIENT);
	MyInitAfterCreation();
}

/*
void gNetPlayerWall::Update(REAL Tend,REAL dend){
	if (!inGrid){
		tEnd=Tend;
		end=beg + dir*(dend-dbegin);

#ifdef DEBUG
		if (!finite(end.x) || !finite(end.y))
			st_Breakpoint();
#endif

		if (e)
			e->Coord(1) = end;

		gPlayerWall *w = Wall();
		if (w){
			w->SetEndTime(tEnd);
			w->SetEndPos(dend);
			w->CalcLen();
		}
	}
}
*/

void gNetPlayerWall::Update(REAL Tend,const eCoord &pend)
{
	if (!inGrid && ( preliminary || sn_GetNetState() != nCLIENT ) )
	{
		real_Update( Tend, pend, false );
	}
}

void gNetPlayerWall::real_Update(REAL Tend,const eCoord &pend, bool force )
{
	tEnd=Tend;
	end=pend;

	// make sure the wall points forward
	REAL forward = eCoord::F( end-beg, dir );
	if ( forward < 0 )
	{
		beg = end;
		tBeg = tEnd;
	}

#ifdef DEBUG
	if (!finite(end.x) || !finite(end.y))
		st_Breakpoint();
#endif

	eCoord odir=dir.Turn(0,1);
	REAL x=eCoord::F(odir,(end-beg));
	beg=beg+odir*x;

	if (bool( this->edge_ ) && this->edge_->Point(0) && this->edge_->Point(1)){
		this->edge_->Coord(1) = end;
		this->edge_->Coord(0) = beg;
	}

	gPlayerWall *w = Wall();

	SetEndTime(tEnd);

	if ( bool( this->cycle_ ) && !force )
	{
		SetEndPos(this->cycle_->distance);
	}
	else
	{
		SetEndPos(REAL(dbegin+sqrt((beg-end).NormSquared())));
	}

	if ( w )
		w->CalcLen();
}

void gNetPlayerWall::CopyIntoGrid(bool force){
	if (!inGrid && (force ||
					(sn_GetNetState()!=nCLIENT || preliminary))){
		inGrid=true;
		gridding=REAL(se_GameTime()+.5);
		if (sn_GetNetState()==nSERVER)
			RequestSync();
		else
			gridding=se_GameTime()+2*sn_Connections[0].ping+.5;
	}
}

void gNetPlayerWall::real_CopyIntoGrid(eGrid *grid){
	//  con << "Gridding " << ID() << " : ";
	//con << "from " << *e->Point(0) << " to " << *e->Point(1) << '\n';

#ifdef DEBUG
	grid->Check();
#endif

	if(griddedid<0){
		if ( this->cycle_ )
		{
			tASSERT( static_cast< bool >(this->edge_) );
			tASSERT(Wall());
			tASSERT(Wall()->Splittable());

			if (preliminary){
				//delete this; // get rid of it
				tControlledPTR< nNetObject > bounce( this );

				sg_netPlayerWalls.Remove(this,id);
				gridded_sg_netPlayerWalls.Add(this,griddedid);
				Wall()->Insert();
				this->ReleaseData();
			}
			else{
				gridded_sg_netPlayerWalls.Add(this,griddedid);
				sg_netPlayerWalls.Remove(this,id);
				if ( this->edge_ ){
					Wall()->Insert();
					this->edge_->CopyIntoGrid(this->cycle_->Grid());
					this->edge_ = NULL;
				}
			}
		}
	}

#ifdef DEBUG
	grid->Check();
#endif

}

void gNetPlayerWall::s_CopyIntoGrid()
{
#ifdef DEBUG
	static int maxw=20;
	if (sg_netPlayerWalls.Len()>maxw)
		con << "Many walls: " << (maxw=sg_netPlayerWalls.Len()) << '\n';
#endif

	for(int i=sg_netPlayerWalls.Len()-1;i>=0;i--){
		gNetPlayerWall *w=sg_netPlayerWalls(i);
		if (w->inGrid && w->griddedid<0 && se_GameTime()>w->gridding)
			w->real_CopyIntoGrid(w->cycle_->Grid());
	}
}

void gNetPlayerWall::RealWallReceived( gNetPlayerWall* realwall )
{
	if (this->cycle_ )
	{
		tASSERT( realwall );
		tASSERT( preliminary && !realwall->preliminary );

		REAL thresh = 10.0f;

		if ( tBeg + thresh < realwall->tEnd )
		{
			this->real_CopyIntoGrid( this->cycle_->Grid() );
		}
	}
}


void gNetPlayerWall::WriteCreate(nMessage &m)
{
	tASSERT( this->cycle_ );

	nNetObject::WriteCreate(m);
	m.Write(this->cycle_->ID());
	m << beg;
	m << dir;
	m << dbegin;
	m << tBeg;
	m << static_cast<int>(preliminary);
}

gNetPlayerWall::gNetPlayerWall(nMessage &m)
	:nNetObject(m),
	 id(-1),griddedid(-1),
	 cycle_(NULL),edge_(NULL),//w(NULL),
	 dir(0,0),dbegin(0),
	 beg(0,0),end(0,0),
	 tBeg(0),tEnd(0),
	 inGrid(0)
{
	unsigned short cid;
	m.Read(cid);
	cycle_=static_cast<gCycle *>(Object(cid));

	m >> beg;
	end=beg;
	m >> dir;
	m >> dbegin;

	m >> tBeg;
	m >> reinterpret_cast<int &>(preliminary);

	this->InitArray();
}

void gNetPlayerWall::ReleaseData()
{
	if (this->cycle_){
		if (this->cycle_->currentWall==this)
			this->cycle_->currentWall=NULL;
		if (this->cycle_->lastWall==this)
			this->cycle_->lastWall=NULL;
	}

	// tDESTROY(w);
  
	if (this->edge_)
	{
		if ( this->edge_->Wall() )
			this->edge_->Wall()->Insert();

		this->edge_ = NULL;  // w will be deleted with e
		//    tDESTROY_PTR(p1); 
		//    tDESTROY_PTR(p2); 
	}

	this->cycle_=NULL;
	this->edge_=NULL;
	//w=NULL;

	sg_netPlayerWalls.Remove(this,id);
	gridded_sg_netPlayerWalls.Remove(this,griddedid);
}

gNetPlayerWall::~gNetPlayerWall()
{
  ReleaseData();

#ifndef DEDICATED
	if(this->displayList_!=0)
		glDeleteLists(this->displayList_,1);
#endif
}

bool gNetPlayerWall::ActionOnQuit()
{
  if ( sn_GetNetState() == nSERVER )
	{
	  TakeOwnership();
	  return false;
	}
  else
	{
	  ReleaseData();
  
	  return true;
	}
}

bool gNetPlayerWall::ClearToTransmit(int user) const{
#ifdef DEBUG
	if (nNetObject::DoDebugPrint() && bool( this->cycle_ ) )
    {
		if (!GridIsReady(user))
			con << "Not transfering gNetPlayerWall " << ID()
				<< " for user " << user << " because the grid is not ready yet.\n";
		else if (!this->cycle_)
			con << "Not transfering gNetPlayerWall " << ID()
				<< " for user " << user << " because it has no cycle!\n";
		else if (!this->cycle_->HasBeenTransmitted(user))
		{
			tString s;
			s << "No transfering gNetPlayerWall " << ID()
			  << " for user " << user << " because ";
			this->cycle_->PrintName(s);
			s << " has not been transmitted.\n";
			con << s;
		}
    }
#endif

	return GridIsReady(user) && nNetObject::ClearToTransmit(user)
		&& bool(this->cycle_) && this->cycle_->HasBeenTransmitted(user) && inGrid;
}

void gNetPlayerWall::WriteSync(nMessage &m){
	nNetObject::WriteSync(m);

	if (inGrid){
		m << end; // the far end of the eWall
		m << tEnd; // the endTime
	}
	else{
		m << beg;
		m << tBeg;
	}
	m.Write(inGrid);

	if ( coords_.Len() > 2 || !coords_(0).IsDangerous || !coords_(1).IsDangerous )
	{
		unsigned short len = coords_.Len();
		m.Write( len );
		for ( int i = len-1; i>=0; --i )
		{
			const gPlayerWallCoord& coord = coords_(i);
			m << coord.IsDangerous;
			m << coord.Pos;
			m << coord.Time;
		}
	}
}

bool gNetPlayerWall::SyncIsNew(nMessage &m)
{
//	return (nNetObject::SyncIsNew(m) && !inGrid);
	return nNetObject::SyncIsNew(m);
}

static bool sg_ServerSentHoles = false;

void gNetPlayerWall::ReadSync(nMessage &m){
	nNetObject::ReadSync(m);

	REAL tEnd_new;
	eCoord end_new;

	m >> end_new;
	m >> tEnd_new;

	if ( tEnd_new < tBeg )
	{
		tEnd_new = tBeg;
	}

	real_Update(tEnd_new,end_new, true);

	unsigned short new_inGrid;
	m.Read(new_inGrid);

	CreateEdge();

	if ( ! m.End() )
	{
		unsigned short len;
		m.Read( len );

		coords_.SetLen( len );

		for ( int i = len-1; i>=0; --i )
		{
			gPlayerWallCoord& coord = coords_(i);
			m >> coord.IsDangerous;
			m >> coord.Pos;
			m >> coord.Time;
		}

		sg_ServerSentHoles = true;
	}

	if(Wall() && new_inGrid && !inGrid)
	{
/*
		if ( ( beg - end ).NormSquared() > 0.01f )
		{
			gExplosion::OnNewWall( Wall() );
		}
*/

		CopyIntoGrid(true);

		if (!preliminary)
		{
			// inform preliminary walls
			for (int i=sg_netPlayerWalls.Len()-1;i>=0;i--)
			{
				gNetPlayerWall *o=sg_netPlayerWalls[i];
				if ( o != this && o->preliminary && o->cycle_ == this->cycle_ && o->tBeg - 10.0f < tEnd )
				{
					o->RealWallReceived( this );
				}
			}
		}
	}
	else
	{
//		st_Breakpoint();
	}
}

static nNOInitialisator<gNetPlayerWall> gNetPlayerWall_init(300,"gNetPlayerWall");

nDescriptor &gNetPlayerWall::CreatorDescriptor() const
{
	return gNetPlayerWall_init;
}

void gNetPlayerWall::PrintName(tString &s) const
{
  s << "gNetPlayerWall nr. " << id;
  if ( this->cycle_ )
	{
	  s	<< " owned by ";
	  this->cycle_->PrintName( s );
	}
}


void gNetPlayerWall::Clear()
{
	//	if( nCLIENT == sn_GetNetState() )
	//	return;

	int i;
	for(i=sg_netPlayerWalls.Len()-1;i>=0;i--){
		// sg_netPlayerWalls(i)->owner=sn_myNetID;
		//delete sg_netPlayerWalls(i);
		gNetPlayerWall* w = sg_netPlayerWalls(i);
		tControlledPTR< nNetObject > bounce( w );
		w->ReleaseData();

		sg_netPlayerWalls.Remove( w, w->id );

		if ( w->edge_ )
			w->edge_->Wall()->Insert();

	}
	for(i=gridded_sg_netPlayerWalls.Len()-1;i>=0;i--){
		// gridded_sg_netPlayerWalls(i)->owner=sn_myNetID;
		gNetPlayerWall* w = gridded_sg_netPlayerWalls(i);
		tControlledPTR< nNetObject > bounce( w );
		w->ReleaseData();

		gridded_sg_netPlayerWalls.Remove( w, w->griddedid );
	}
}


void gNetPlayerWall::Check() const
{
#ifdef DEBUG	
	int i;
	for ( i = coords_.Len() -2 ; i>=0; --i )
	{
		gPlayerWallCoord* coords = &( coords_( i ) );
		tASSERT( coords[0].Pos <= coords[1].Pos );
		tASSERT( coords[0].Time <= coords[1].Time );
	}

	for ( i = coords_.Len() -1 ; i>=0; --i )
	{
		gPlayerWallCoord* coords = &( coords_( i ) );
		tASSERT( finite( coords[0].Pos ) );
		tASSERT( finite( coords[0].Time ) );
	}
#endif
}

int gNetPlayerWall::IndexPos(REAL d) const
{
	CHECKWALL;

	
	// get the first coord with smaller alpha than a
	int i = coords_.Len() - 2;
	while ( i >= 1 && coords_(i).Pos >= d)
		--i;

#ifdef DEBUG
	if (!( i >= 0 && i < coords_.Len() - 1 ))
	{
		st_Breakpoint();
	}
#endif

	return i;
}

int gNetPlayerWall::IndexAlpha(REAL a) const
{
	CHECKWALL;

	REAL d = Pos( a );

	return IndexPos( d );
}

REAL gNetPlayerWall::Time(REAL a) const
{
	tASSERT( good( a ) );

	CHECKWALL;

	const gPlayerWallCoord* coord = &coords_(IndexAlpha(a));
	REAL div = ( coord[1].Pos - coord[0].Pos );
	REAL alpha = 0.0f;
	if ( div > 0 )
	{
		alpha = ( Pos(a) - coord[0].Pos ) / div;
	}

	REAL ret = coord[0].Time + alpha*(coord[1].Time-coord[0].Time);

	tASSERT( good( ret ) );

	return ret;
}

REAL gNetPlayerWall::Pos(REAL a) const
{
	CHECKWALL;

	tASSERT( good( a ) );

	REAL ret = BegPos() + a * ( EndPos()  - BegPos() );

	tASSERT( good( ret ) );

	return ret;
}

REAL gNetPlayerWall::Alpha(REAL pos) const
{
	CHECKWALL;

	REAL diff = ( EndPos()  - BegPos() );
	REAL a = pos - BegPos();

	if ( diff > 0 )
		a /= diff;

	tASSERT ( -.001 < a );
	tASSERT ( 1.001 > a );

	return a;
}

bool gNetPlayerWall::IsDangerous( REAL a ) const
{
	CHECKWALL;

	const gPlayerWallCoord* coord = &coords_(IndexAlpha(a));
	return coord->IsDangerous;
}

REAL gNetPlayerWall::EndPos() const
{
	CHECKWALL;

	return coords_(coords_.Len()-1).Pos;
}

REAL gNetPlayerWall::BegPos() const
{
	CHECKWALL;

	return coords_(0).Pos;
}

REAL gNetPlayerWall::EndTime() const
{
	CHECKWALL;

	return coords_(coords_.Len()-1).Time;
}

REAL gNetPlayerWall::BegTime() const
{
	CHECKWALL;

	return coords_(0).Time;
}

void gNetPlayerWall::BlowHole	( REAL beg, REAL end )
{
	CHECKWALL;

	// don't touch anything if the server concluded it is his business
	if ( sn_GetNetState() != nSERVER && sg_ServerSentHoles && !preliminary )
	{
		return;
	}

#ifdef DEBUG
	tASSERT (coords_.Len() < 1000 );
#endif

	// find the last index that will stay before the hole:
	int begind = IndexPos( beg );

	// find the last index in the hole:
	int endind = IndexPos( end );

	if ( beg < BegPos() )
	{
		begind = -1;

		beg = BegPos();
	}

	if ( end > EndPos() )
	{
		if ( EndPos() < cycle_->distance-10 || this != cycle_->currentWall )
			endind = coords_.Len() - 1;

		end = EndPos();
	}

	// out of range
	if ( end < beg )
	{
		return;
	}

	if ( sn_GetNetState() != nCLIENT )
	{
		this->RequestSync();
	}

	// find the alpha at the hole begin and end:
	REAL begalph = Alpha( beg );
	REAL endalph = Alpha( end );

	// find the time at the hole begin and end:
	REAL begtime = Time( begalph );
	REAL endtime = Time( endalph );

	int insert = begind + 2 - endind;

#ifdef DEBUG
	tASSERT (insert < 40 );
#endif

	// remove positions inside the hole:
	if ( insert < 0 )
	{
		for ( int i = begind+1; i - insert < coords_.Len(); ++i )
			coords_(i) = coords_( i - insert );
		coords_.SetLen( coords_.Len() + insert );
	}

	// make room for the new points of the hole:
	else if ( insert > 0 )
	{
		coords_.SetLen( coords_.Len() + insert );

		for ( int i = coords_.Len() - 1; i >= begind + insert && i >= insert ; --i )
			coords_( i ) = coords_( i - insert );
	}

	// clamp times
	{
		if ( begind >= 0 )
		{
			REAL beforetime = coords_(begind).Time;
			if ( begtime < beforetime )
			{
				begtime = beforetime;
			}	
		}

		if ( begind +3 < coords_.Len() )
		{
			REAL afttime = coords_(begind + 3).Time;
			if ( endtime > afttime )
			{
				endtime = afttime;
			}	
		}
	}

	// enter the hole
	coords_(begind+1).IsDangerous = false;
	coords_(begind+1).Time        = begtime;
	coords_(begind+1).Pos         = beg;
	coords_(begind+2).Time        = endtime;
	coords_(begind+2).Pos         = end;

	CHECKWALL;
}

static void login_callback(){
	sg_ServerSentHoles = false;
}

static nCallbackLoginLogout sg_LoginLogout(&login_callback);

