/* intel.c

    icdprog - an open source PIC programmer for use with the Microchip ICD(1)
    Copyright (C) 2001-2004  Geir Thomassen.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "intel.h"

HEXFILE * open_hexfile (char *filename, char *mode)
{
	HEXFILE *fp;

	if ((fp = (HEXFILE *) malloc (sizeof (HEXFILE))) == NULL)
		return NULL;

	fp->adr = 0;
	fp->segment = 0;
	fp->checksum = 0;
	fp->dp = fp->dat;
	fp->ep = fp->dat;
	
	if (*mode == 'r')
		fp->mode = 'r';
	else
		fp->mode = 'w';
	
	if ((fp->fp = fopen (filename, mode)) == NULL) {
		free (fp);
		return NULL;
	}
	
	return fp;
}

void close_hexfile (HEXFILE * fp)
{
	fclose (fp->fp);
	free (fp);
}

int read_hexfile (HEXFILE * fp, int *data, unsigned long *adr)
{
	char ch;
	int n, x;
	int checksum, recordtype;
	unsigned long off;
	
	if (!(fp->dp < fp->ep)) {       /* Data in buffer ?? */
		/* No, fetch data */
		ch = fgetc (fp->fp);
		if (ch != ':')		/* Read ':' */
			return HEX_ERR;
		
		if (fscanf (fp->fp, "%02x%04lx%02x", &n, &off, &recordtype) != 3) /* read header */
			return HEX_ERR;
		
		checksum = (n + (off & 0xFF) + ((off >> 8) & 0xFF) + recordtype) & 0xFF;
		
		switch (recordtype) {
		case 0:		/* data record */
			fp->dp = fp->ep = fp->dat;	/* reset buffer */
			
			while (n--) {
				if (fscanf (fp->fp, "%02x", &x) != 1)	/* read data */
					return HEX_ERR;
				
				*fp->ep++ = x;	/* store data */
				checksum += x;
				checksum &= 0xFF;
			}
			
			if (fscanf (fp->fp, "%02x", &x) != 1)	/* read checksum */
				return HEX_ERR;
			
			checksum += x;
			
			if ((checksum & 0xFF) != 0)
				return HEX_ERR;
			
			while(isspace(ch = fgetc (fp->fp))) /* Discard \n */
				;
			
			ungetc(ch,fp->fp);
			
			fp->adr = fp->segment + off;
			break;
			
		case 1:		/* End of file record */
			if (fscanf (fp->fp, "%02x", &x) != 1)	/* checksum should be 0xFF */
				return HEX_ERR;
			return (x == 0xFF) ? HEX_EOF : HEX_ERR;
			break;
			
		case 2:		/* Extended address record */
		case 4:		/* Extended linear address record */
			
			if (fscanf (fp->fp, "%04lx%02x", &fp->segment, &x) != 2)
				return HEX_ERR;
			
			checksum += (fp->segment & 0xFF) + (((fp->segment) >> 8) & 0xFF) + x;
			
			if ((checksum & 0xFF) != 0)
				return HEX_ERR;
			
			if (recordtype == 2)
				fp->segment <<= 4;
			else
				fp->segment <<= 16;
			
			while(isspace(ch = fgetc (fp->fp))) /* Discard \n */
				;
			
			ungetc(ch,fp->fp);
			
			return read_hexfile (fp, data, adr);   /* I bet you didn't expect that ... */
			break;
			
		default:		/* unknown record type */
			return HEX_ERR;
			break;
		}
	}
	
	*adr = fp->adr++;
	*data = *fp->dp++;
	return HEX_OK;
}
