/*
 * GQradio
 * (C) 2005 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqradio.h"

#include "display.h"
#include "io_radio.h"
#include "mixer.h"
#include "preset.h"
#include "rcfile.h"
#include "ui2_editor.h"
#include "ui2_main.h"
#include "ui2_util.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"
#include "window.h"


#define RADIO_SLOW_SEEK 4
#define RADIO_SLOW_THRESHHOLD 4

static gint main_loop_id = -1;

static gint auto_seeking = 0;
static gint auto_seek_signal = -1;
static guint32 auto_seek_freq = 0;

static gint seeking = 0;
static gint seek_cnt = 0;
static gint seek_trips = 0;

static gint preset_mode = FALSE;
static gint preset_mode_point = 0;
static gint preset_mode_seek = 0;

static gint preset_seeking = 0;

static gint preset_scan = FALSE;
static guint32 preset_frequency = 0;
static gint preset_muted = FALSE;

static gint vol_btn_down = 0;
static gint bal_btn_down = 0;


static void auto_seek_reset(void);

static void preset_seek_increment(gint direction_up);
static void preset_auto_scan_end(void);

static void adjust_volume_by(gint n);
static void adjust_balance_by(gint n);


/*
 *-----------------------------------------------------------------------------
 * main loop!
 *-----------------------------------------------------------------------------
 */


static gint main_loop(gpointer data)
{
	static gint status_cnt = 0;
	static gint flash = 0;
	static gint mixer_counter = 0;

	flash++;
	if (flash > RADIO_STATUS_INTERVAL / 2) flash = 0;

	if (vol_btn_down !=0) adjust_volume_by((vol_btn_down > 0) ? 3 : -3);
	if (bal_btn_down !=0) adjust_balance_by((bal_btn_down > 0) ? -3 : 3);

	/* update mixer (volume/balance controls) every 4 seconds */
	mixer_counter++;
	if (mixer_counter > RADIO_STATUS_INTERVAL * 4)
		{
		mixer_counter = 0;
		display_set_volume();
		display_set_balance();
		}

	if (preset_scan)
		{
		display_increment_scanner(FALSE);
		display_set_preset_scan(TRUE);
		if (flash == 0) display_update_preset(TRUE, -1);

		if (frequency + RADIO_SCAN_INCREMENT > radio_limit_get_upper())
			{
			preset_auto_scan_end();
			}
		}

	if (auto_seeking != 0)
		{
		if (!radio_status(&stereo, &signal_strength))
			{
			auto_seek_reset();
			}
		else if (auto_seek_signal < 0)
			{
			if (signal_strength < RADIO_SCAN_SENSITIVITY) auto_seek_signal = 0;
			seek_increment(RADIO_SCAN_INCREMENT, (auto_seeking > 0));
			display_increment_scanner(FALSE);
			}
		else if (auto_seek_signal == 0)
			{
			if (signal_strength >= RADIO_SCAN_SENSITIVITY)
				{
				auto_seek_signal = 1;
				auto_seek_freq = frequency;
				}
			seek_increment(RADIO_SCAN_INCREMENT, (auto_seeking > 0));
			display_increment_scanner(FALSE);
			}
		else
			{
			if (signal_strength < RADIO_SCAN_SENSITIVITY)
				{
				guint32 new_freq;

				new_freq = (auto_seek_freq + (frequency - RADIO_SCAN_INCREMENT)) / 2;
				new_freq = radio_freq_clamp_to_increment(new_freq, freq_step);

				if (debug_mode) printf("auto scan using %d\n", new_freq);

				auto_seek_reset();

				if (preset_scan)
					{
					gint p;

					p = preset_list_find_free_slot();
					if (p >= 0)
						{
						preset_set(p, new_freq, NULL);
						display_preset_list_update(p);
						}
					if (debug_mode) printf("Auto seeked station %s as preset %d\n", radio_freq_to_text(new_freq), p);

					auto_seeking = 1;
					}
				else
					{
					seek_to_freq(new_freq);
					mute_set(FALSE);
					}
				}
			else
				{
				seek_increment(RADIO_SCAN_INCREMENT, (auto_seeking > 0));
				display_increment_scanner(FALSE);
				}
			}

		display_update_status();
		return TRUE;
		}

	status_cnt++;
	if (status_cnt > RADIO_SCANS_PER_SECOND / RADIO_STATUS_INTERVAL)
		{
		if (radio_status(&stereo, &signal_strength))
			{
			display_update_status();
			}

		status_cnt = 0;
		}

	if (seeking != 0 || preset_mode_seek != 0 || preset_seeking != 0)
		{
		seek_cnt++;
		if (seek_cnt > RADIO_SCANS_PER_SECOND / RADIO_SLOW_SEEK)
			{
			seek_cnt = 0;
			if (seek_trips <= RADIO_SLOW_THRESHHOLD) seek_trips++;
			}

		if (seek_cnt == 0 || seek_trips > RADIO_SLOW_THRESHHOLD)
			{
			if (preset_mode_seek != 0)
				{
				preset_seek_increment((preset_mode_seek > 0));
				flash = 1;
				}
			else if (seeking != 0)
				{
				seek_increment(freq_step, (seeking > 0));
				}
			else
				{
				if (preset_seeking > 0)
					{
					preset_next();
					}
				else
					{
					preset_prev();
					}
				}
			}
		}
	else
		{
		seek_trips = 0;
		seek_cnt = 0;

		if (preset_mode && flash == 0)
			{
			display_update_preset(TRUE, preset_mode_point);
			}
		}

	return TRUE;
}

void main_loop_init()
{
	if (main_loop_id == -1)
		{
		main_loop_id = g_timeout_add(1000 / RADIO_SCANS_PER_SECOND,
					     main_loop, NULL);
		}
}

/*
 *-----------------------------------------------------------------------------
 * seek
 *-----------------------------------------------------------------------------
 */

void seek_to_freq(guint32 freq)
{
	gchar *buf;
	gchar *number;
	const gchar *desc;

	CLAMP(freq, radio_limit_get_lower(), radio_limit_get_upper());

	frequency = freq;

	if (!muted) radio_set_mute(TRUE);
	radio_set_freq(frequency);
	if (!muted) radio_set_mute(FALSE);

	if (preset < 0 || !preset_is_freq(preset, frequency))
		{
		preset = preset_find_freq(frequency);
		}

	display_set_frequency(frequency);
	if (!preset_scan) display_update_preset(FALSE, -1);

	number = radio_freq_to_text(frequency);
	desc = preset_get_description(preset);
	if (!desc)
		{
		buf = number;
		number = NULL;
		}
	else
		{
		buf = g_strdup_printf("%s %s", number, desc);
		}
	window_main_set_title(buf);
	g_free(buf);
	g_free(number);
}

void seek_increment(guint32 step, gint direction_up)
{
	guint32 old;
	guint32 l, h;

	l = radio_limit_get_lower();
	h = radio_limit_get_upper();

	/* sane steps */
	if (step < 10000) step = 10000;

	old = frequency;
	frequency = radio_freq_clamp_to_increment(frequency, step);

	if (direction_up)
		{
		if (frequency <= old) frequency += step;
		if (frequency > h) frequency = l;
		}
	else
		{
		if (frequency >= old) frequency -= step;
		if (frequency < l) frequency = h;
		}

	seek_to_freq(frequency);
}

void seek_start(gint direction_up)
{
	gint d;

	if (preset_scan) return;

	if (direction_up)
		{
		d = 1;
		}
	else
		{
		d = -1;
		}
	seek_cnt = 0;
	seek_trips = 0;

	if (preset_mode)
		{
		preset_mode_seek = d;
		preset_seek_increment(preset_mode_seek > 0);
		return;
		}

	if (seek_mode == SEEK_MODE_PRESET)
		{
		auto_seek_reset();

		if (direction_up)
			{
			preset_next();
			}
		else
			{
			preset_prev();
			}

		preset_seeking = d;
		return;
		}

	if (seek_mode == SEEK_MODE_AUTO)
		{
		if (!auto_seek_nomute) mute_set(TRUE);
		seek_increment(RADIO_SCAN_INCREMENT, direction_up);

		if (auto_seeking != 0)
			{
			if (auto_seeking == d)
				{
				auto_seeking = 0;
				mute_set(FALSE);
				}
			else
				{
				auto_seek_reset();
				auto_seeking = d;
				}
			}
		else
			{
			auto_seek_reset();
			auto_seeking = d;
			}
		return;
		}

	/* SEEK_MODE_MANUAL */

	auto_seek_reset();
	seek_increment(freq_step, direction_up);
	seeking = d;
}

void seek_start_by_mode(SeekMode mode, gint direction_up)
{
	SeekMode orig_mode;

	orig_mode = seek_mode;

	seek_mode = mode;
	seek_start(direction_up);
	seek_mode = orig_mode;
}

void seek_stop(void)
{
	if (preset_scan) return;

	seeking = 0;
	seek_cnt = 0;
	seek_trips = 0;

	preset_mode_seek = 0;
	preset_seeking = 0;
}

static void auto_seek_reset(void)
{
	auto_seeking = 0;
	auto_seek_signal = -1;
	auto_seek_freq = 0;
	if (!preset_scan) display_increment_scanner(TRUE);
}

/*
 *-----------------------------------------------------------------------------
 * modes, etc.
 *-----------------------------------------------------------------------------
 */

void mute_set(gint set)
{
	if (muted == set) return;
	if (preset_scan) return;

	muted = set;
	radio_set_mute(muted);
	display_update_status();
}

void mute_toggle(void)
{
	mute_set(!muted);
}

void mode_set(SeekMode mode)
{
	if (preset_scan) return;

	seek_mode = mode;
	display_set_mode(TRUE);
}

void mode_toggle(void)
{
	SeekMode mode;

	if (seek_mode == SEEK_MODE_MANUAL)
		{
		mode = SEEK_MODE_PRESET;
		}
	else if (seek_mode == SEEK_MODE_PRESET)
		{
		mode = SEEK_MODE_AUTO;
		}
	else
		{
		mode = SEEK_MODE_MANUAL;
		}

	mode_set(mode);
}

/*
 *-----------------------------------------------------------------------------
 * presets, etc.
 *-----------------------------------------------------------------------------
 */

void preset_select(gint p)
{
	if (p < 0 || p >= PRESET_LIST_SIZE || preset == p) return;

	if (preset_scan) return;

	if (preset_mode)
		{
		preset_mode_point = p;
		preset_button_click();
		return;
		}

	if (!preset_is_set(p)) return;

	auto_seek_reset();

	preset = p;
	seek_to_freq(preset_get_freq(p));
	mute_set(FALSE);
}

void preset_next(void)
{
	gint p, start_p;

	if (preset_is_freq(preset, frequency))
		{
		p = preset + 1;
		}
	else
		{
		p = -1;
		}

	start_p = p;
	if (p < 0) p = 0;

	while (p < PRESET_LIST_SIZE && !preset_is_set(p)) p++;

	if (p >= PRESET_LIST_SIZE && start_p >= 0)
		{
		p = 0;
		while (p < start_p && !preset_is_set(p)) p++;
		if (start_p == p) return;
		}

	preset_select(p);
}

void preset_prev(void)
{
	gint p, start_p;

	if (preset_is_freq(preset, frequency))
		{
		p = preset - 1;
		}
	else
		{
		p = -1;
		}

	start_p = p;
	if (p < 0) p = PRESET_LIST_SIZE - 1;

	while (p >= 0 && !preset_is_set(p)) p--;

	if (p < 0 && start_p >= 0)
		{
		p = PRESET_LIST_SIZE - 1;
		while (p > start_p && !preset_is_set(p)) p--;
		if (p < 0) return;
		}

	preset_select(p);
}

static void preset_seek_increment(gint direction_up)
{
	if (direction_up)
		{
		preset_mode_point++;
		if (preset_mode_point >= PRESET_LIST_SIZE) preset_mode_point = 0;
		}
	else
		{
		preset_mode_point--;
		if (preset_mode_point < 0) preset_mode_point = PRESET_LIST_SIZE - 1;
		}

	display_update_preset(TRUE, preset_mode_point);
}

void preset_button_click(void)
{
	if (preset_scan) return;

	if (preset_mode)
		{
		preset_mode = FALSE;
		preset_set(preset_mode_point, frequency, NULL);
		preset = preset_mode_point;
		display_preset_list_update(preset_mode_point);
		display_update_preset(FALSE, 0);
		}
	else
		{
		preset_mode = TRUE;
		display_update_preset(TRUE, preset_mode_point);
		display_set_description(_("Select preset"));
		}
}

void preset_cancel(void)
{
	if (preset_scan)
		{
		preset_auto_scan_end();
		}
	else if (preset_mode)
		{
		preset_mode = FALSE;
		display_update_preset(FALSE, preset);
		}
}

gint preset_mode_get(void)
{
	return (preset_mode || preset_scan);
}

void preset_clear_one(gint p)
{
	if (p < 0) return;

	preset_clear(p);

	if (preset == p) preset = -1;
	display_preset_list_update(p);
	display_update_preset(FALSE, preset);
}


static GenericDialog *preset_clear_dlg = NULL;


static void preset_clear_cancel_cb(GenericDialog *gd, gpointer data)
{
	generic_dialog_close(gd);
	preset_clear_dlg = NULL;
}

static void preset_clear_ok_cb(GenericDialog *gd, gpointer data)
{
	generic_dialog_close(gd);
	preset_clear_dlg = NULL;

	preset_list_clear();
	preset = -1;
	display_preset_list_update(-1);
	display_update_preset(FALSE, preset);
}

void preset_clear_all(void)
{
	if (preset_clear_dlg) return;

	preset_clear_dlg = generic_dialog_new(_("Clear presets - GQradio"),
					      "GQradio", "preset_clear",
					      NULL, FALSE,
					      preset_clear_cancel_cb, NULL);
	generic_dialog_add_button(preset_clear_dlg, GTK_STOCK_CLEAR, NULL,
				  preset_clear_ok_cb, TRUE);

	generic_dialog_add_message(preset_clear_dlg, GTK_STOCK_DIALOG_QUESTION,
				   _("Clear the preset list?"), NULL);

	gtk_widget_show(preset_clear_dlg->dialog);
}

static void preset_auto_scan_end(void)
{
	auto_seek_reset();

	preset_scan = FALSE;
	muted = preset_muted;

	seek_to_freq(preset_frequency);
	display_set_preset_scan(FALSE);
	display_increment_scanner(TRUE);
}


static GenericDialog *preset_scan_dlg = NULL;


static void preset_scan_cancel_cb(GenericDialog *gd, gpointer data)
{
	generic_dialog_close(gd);
	preset_scan_dlg = NULL;
}

static void preset_scan_ok_cb(GenericDialog *gd, gpointer data)
{
	generic_dialog_close(gd);
	preset_scan_dlg = NULL;

	if (!preset_scan)
		{
		preset_cancel();

		preset_muted = muted;
		if (!auto_seek_nomute) mute_set(TRUE);

		preset_scan = TRUE;
		preset_frequency = frequency;

		seek_mode = SEEK_MODE_AUTO;
		auto_seek_reset();
		auto_seeking = 1;

		display_set_mode(TRUE);
		seek_to_freq(radio_limit_get_lower());
		display_increment_scanner(FALSE);
		display_set_preset_scan(TRUE);
		}
}

void preset_auto_scan_click(void)
{
	if (preset_scan)
		{
		preset_auto_scan_end();
		}
	else if (!preset_scan_dlg)
		{
		preset_scan_dlg = generic_dialog_new(_("Auto Scan - GQradio"),
						     "GQradio", "preset_scan",
						     NULL, FALSE,
						     preset_scan_cancel_cb, NULL);
		generic_dialog_add_button(preset_scan_dlg, GTK_STOCK_FIND, _("Scan"),
					  preset_scan_ok_cb, TRUE);

		generic_dialog_add_message(preset_scan_dlg, GTK_STOCK_DIALOG_QUESTION,
					   _("Scan frequencies to fill the preset list?"), NULL);

		gtk_widget_show(preset_scan_dlg->dialog);
		}
}

/*
 *-----------------------------------------------------------------------------
 * mixer, volume and balance
 *-----------------------------------------------------------------------------
 */

static void adjust_volume_by(gint n)
{
	gint vol = get_volume();
	vol += n;
	if (vol < 0) vol = 0;
	if (vol > 100) vol = 100;
	set_volume(vol);
	display_set_volume();
}

void volume_adjust(float pos)
{
	set_volume((gint)(pos * 100));
}

void btn_volume_up_pressed(void)
{
	vol_btn_down = 1;
	adjust_volume_by(3);
}

void btn_volume_down_pressed(void)
{
	vol_btn_down = -1;
	adjust_volume_by(-3);
}

void btn_volume_released(void)
{
	vol_btn_down = 0;
}

/*
 *-----------------------------------------------------------------------------
 * balance callbacks
 *-----------------------------------------------------------------------------
 */

static void adjust_balance_by(gint n)
{
	gint bal = get_balance();
	bal += n;
	if (bal < 0) bal = 0;
	if (bal > 100) bal = 100;
	if (bal > 50 - abs(n) && bal < 50 + abs(n)) bal = 50; /* basic balance auto-centering */
	set_balance(bal);
	display_set_balance();
}

void balance_adjust(float pos)
{
	gint p = pos * 100;
	if (p > 44 && p < 56) p = 50; /* to help balance 'lock' on center */
	set_balance(p);
}

void btn_balance_left_pressed(void)
{
	bal_btn_down = 1;
	adjust_balance_by(-3);
}

void btn_balance_right_pressed(void)
{
	bal_btn_down = -1;
	adjust_balance_by(3);
}

void btn_balance_released(void)
{
	bal_btn_down = 0;
}

void mixer_run(void)
{
	if (mixer_command)
		{
		gchar *command = g_strconcat(mixer_command, " &", NULL);
		system(command);
		g_free(command);
		}
}





