use criterion::*;
use editdistancek::edit_distance;
use editdistancek::edit_distance_bounded;
use triple_accel::levenshtein::levenshtein_simd_k as levenshtein_simd_bounded;
use triple_accel::levenshtein::levenshtein as levenshtein_simd;

fn edit_distance_benchmark(c: &mut Criterion) {
    let test_pairs = vec![
        ("kitten", "sitting"),
        ("rosettacode", "raisethysword"),
        // add more test pairs here
    ];

    c.bench_function("edit_distance", |b| {
        b.iter(|| {
            for &(s1, s2) in &test_pairs {
                black_box(edit_distance(black_box(s1.as_bytes()), black_box(s2.as_bytes())));
            }
        })
    });
    c.bench_function("edit_distance from triple_accel", |b| {
        b.iter(|| {
            for &(s1, s2) in &test_pairs {
                black_box(levenshtein_simd(black_box(s1.as_bytes()), black_box(s2.as_bytes())));
            }
        })
    });
}

fn edit_distance_benchmark_long(c: &mut Criterion) {
    let test_tuples =
        vec![
            ("ACATNAAAGNGCTGAGGCGCTGGCACGGCNTATGAAGCCAGGCATGTGTTNTCTATGTGACANTCNCTNGATTAACNTNTCAATCCCCNCCNATCCNCATAGNTGCNCGAACGNGCCTATGAGTAANNGGCNGAGNTTNCTTAGCTCGCNAANGGNATCTNTGTTCTAGANNCGGCTNGNNCGACNNTTGGTGACGGNNTGNGCATCAGGCCCNTCNTAAGGNNTGACATGACCTTGGNGGNTCACNGCTCCTCTCACTCGCGNCGANCTATGTCATNATAATCCTTATANCTCGNCTNNCGNCNCATATGNTAAACACAACNCNGACATTATCTTNGNGGAGGNTNGGAGGCTGGAAGNAATGCAGNTTACTCCCGAGNNGAACCCTCTCTAACCAGANTANTCAAANTGNTGTCGTACNCAATACCTGNCNGCNCTCTCTTGTTAGTATNCGTAGTATNAAGGCGGCACGGTTNAAGCTNCCNTNNCNTGGTTTGAGATCGACTGCNCGNNATCTGNGNNNAACACAAAACTANCNCCATNCTNANTGNNNNNTCNTACGCNNANNTCCGGNNTNACCCTACTGNGGCNAGAACNAAGGGTTATGNTTTAGTANTCACGNGATNACAGNAGGNGTGCTTNGAGNAATTTNGGCANTCNGCACCACCCNGGGNTNNNGCGCNNAAATGTCGGTCNNTNNATTCCGCCNNNTTNAANCCNCGTNTCNNGTNANGCTCCNGGGTANANNTTAATGGNAGNAANANCGANCGGACGCGTCCATNGGTAANAANCGGANNNTANTCAGNCGNACCTNTAANAGNGCTGGTCTAATGGTNNNNATGCTTNGCANGGCACGATNTNNAAAATTGAGTTNTTAAGGCCNNCCNANTTTAGGNCTGTGTTNGTCCNNATGNNGCGCNNNCTNTTAANCCANGNATTGGGTTTANCANCCTANGTCATGGNCTANAANNNAGCCCANATAANNGNGCANNTNCTANTC", "ACATNAAAGNGCTGAGGCGCTGGCACGGCNTATGAAGCCAGGCATGTGTTNTCTATGTGACANTCNCTNGATTAACNTNTCAATCCCCNCCNATCCNCATAGNTGCNCGAACGNGCCTATGAGTAANNGGCNGAGNTTNCTTAGCTCGCNAANGGNATCTNTGTTCTAGANNCGGCTNGNNCGACNNTTGGGGACGGNNTGNGCATCAGGCCCNTCNTAAGGNNTGACATGACCTTGGNGGNTCACNGCTCCTCTCACTCGCGNCGANCTATGTCATNATAATCCTTATANCTCGNCTNNCGNCNCATATGNTAAACACAACNCNGACATTATCTTNGNGGAGGNTNGGAGGCTGGAAGNAATGCAGNTTACTCCCGAGNNGAACCCTCTCTAACCAGANTANTCAAANTGNTGTCGTACNCAATACCTGNCNGCNCTCTCTTGTTAGTATNCGTAGTATNAAGGCGGCACGGTTNAAGCTNCCNTNNCNTGGTTTGAGATCGACTGCNCGNNATCTGNGNNNAACACAAAAGTANCNCCATNCTNANTGNNNNNTCNTAGCGCNNANNTCCGGNNTNACCCTACTGNGGCNAGACCNAAGGGTTATGNTTTAGTANTCACGNGATNACAGNAGGNGTGCTTNGAGNAATTTNGGCANTCNGCACCACCCNGGGNTNNNGCGCNNAAATGTCGGTCNNTNNATTCCGCCNNNTTNAANCCNCGTNTCNNGTNANGCTCCNGGGTANANNTTAATGGNAGNAANANCGANCGGACGCGTCCATNGGTAANAANCGGANNNTANCTCAGNCGNACCTNTAANAGNGCTGGTCTAATGGTNNNNATGCTTNGCANGGCACGATNTNNAAAATTGAGTTNTTAAGGCCNNCCNANTTTAGGNCTGTGTTNGTCCNNATGNNGCGCNNNCTNTTAANCCANGNATTGGGTTTANCANCCTANGTCATGGNCTANAANNNAGCCCANATAANNGNGCANNTNCTANTC", 5usize),
            ("NNGCTACCTNANGGCCTNGTTTANGCCANTCNTGNCCGTGNGTTANGCCCTATNCTNGGTNCNNGNANGGNGGATGTTACGNGGGCATCACAGACGNNGGCCCNGCTAGNGTGATACGATNNGCATTANGTGNTTTGCTCGACNTTAAGGAANGNGTNNACNGATTNTCAANNGACAGAGGATCGGCTATCTTCGTGGGGGTCGCCTGCGGAAANNGNGGTNCNACAGTGAGGNTGTTACCANTNATAGCGTCNTCATACNGTNTCNTTCTGCTTNNTNTTCTAAANCTTCATNNNGTNGTAGCNCATTGTAAACCGNTAACAGCNNTANTNGTGTCGGNTNTTGGCNAGANAAGGCNAATCACGANGNCTAGTCCCAGNCCCGNCCGCTGANATNNTTTATANCCTACATNCGTCGTATTTTTCCTNCAANGCANCTTANACGCGTGCCTTTTCCAGGCGCNAATANTTCAGANTAATGANGCAACNTGGNGTCTCACCNTATANGTNNCANTCNACGNGCTACNANTAGAGAAGNNNGTNACCANGCNANANCCACCNTNCCCGCCNGTATNAGCGNGGCTCNAAAANNTCNNTCNACTANGNGANTNNCAGCTGNCTANTGGCNTTCGGTNTGGTTCACCTCTGAACCTCATNTGAAATAAGANNGTCNGTNCTCGGTNNGCGTNNTGNAGCTTCNAGTCTCCNCCGAAATTNTTTNCACANNAATNNGTTCCGAGGCCAATGGNAAGCCCCCTANNCAGATNTGNNGTGGACTATCCGTANCNTATCGTATANAGACAGNNTANGGACNCTAATTGGCTAACAATNNACCGCCGGANNNCCCTCNAAGGTNCANGTNCATCGNTGTACANCNTCGTTNCCCACANAGCNNNNNNTTGNCTGNCNCTTCTCNNGTTCNNCCAANGTANTANACAGTCNGNNACCTNNCNCNCGCAAAAAGATANTAANGGCGAGGTGNCNAANGNTGNCTGGCGCCA", "NNGCTACCTNANGGCCTNGTTTANGCCANTCNTGNCCGTGNGTTANGCCCTATNCTNGGTNCNNGNANGGNGGATGTTACGNGGGCATCACAGACGNNGGCCCNGCAGNGTGATACGATNNGCATNANGTGNTTTGCTCGACNTTAAGGAANGNGTNNACNGATTNTCAANNGACAGAGGATCGGCTATCTTCGTGGGGGTCGCCTGCGGAAANNGNGGTNCNACAGTGAGGNTGTTACCANTNATAGCGTCNTCATACNGTNTCCTTCTGCTTNNTNTTCTAAANCTTCATNNNGTNGTAGCNCATTGTAAACCGNTAACAGCNNTANTNGTGTTCGGNTNTTGGCNAGANAAGGCNAATCACGANGNCTAGTCCCAGNCCCGNCCGCTGANATNNTTTATANCCTACATNCGTCGTATTTTTCCTNCAANGCANCTTANACGCGTGCCTTTTCCAGGCGCNAATANTTCAGANTAATGANGCAACNTGGNGTCTCACCNTATANGTNNCANTCNACGNGCTACNANTAGAGAAGNNNGTNACCANGCNANANCCACCNTNCCCGCCNGTATNAGCGNGGCTCNAAAANNTCNNTCNACTANGNGANTNNCAGCTGNCTANTGGCNTTCGGTNTGGTTCACCTCTGAACNCTCATNTGAAATAAGANNGTCNGTNCTCGGTNNGCGTNNCTGNAGCTTCNAGTCTCCNCCGAAATTNTTTNCACANNAATNNGTTCCGAGGCCAATGGNAAGCCCCCTANNCAGATNTGNNGTGGACTATCCGTANNTATCGTATANAGACAGNNTANGGACNCTAATTGGCTAACAATNNACCGCCGANNNCCCTCNAAGGTNCANGTNCATCGNTGTACANCNTCGTTNCCCACANAGCNNNNNNTTGNCTGNCNCCTTCTCNNGTTCNNCCAANGTANTANACAGTCNGNNACCTNNCNCNCGCAAAAAGATANTAANGGCGAGGTGNCNAANGNTGNCTGGCGCCA", 10usize),
            ("AGCTTGAATCTGTAGTTAAGACATGTAAAGTGNCNTGCGNNGGTAANTGTCCTGAGNGANCTGGGANGNTCCTCTNNGTGAGATAGNGCCNTANCGCNNNGAGCGGAGTACGCNCAGNACCCGAATATNAGNGNGANNNNGACAACAGANGGCGTTNCGATGGAGGNTCAGNNNNNGNNAGTTTCGCTCGCGCGTCGAGCNGTTAGCGCAGCGGNNNNTGGTGACNTCNNCGCTANNGTTGTGANCNACGNGGGATCACTATTNCANNCANGAAGGGATGGAACNNNTTTNAGNNNTNNNANTGNCAACNATNCAGNAGTGGTTTACNNACTNNTGCTNGNNGTTTTTANTACNNCGTTGGGANATTCACNTNGNTTNCAGNAGANACGCCNACNCGGCGANGTACCCGANGGGCCCTCGAGNNTCAAANTGGGGNGTTCCCNCCCANCAATAANGNCATAATTTTNCNGNAATTNGGATTGTCCACCCAGAGTAGACATNAACTGTACGGAATGTGCGCTGGAANCTACNGNCCTTTCTTGCNAAAATCNAGNTGNTCCCATNGCTNNACGNNACTGGNTCACTGCANAAAGNAATNAGANANANTGNTTACNTNGGNTGCGNNNNTGACGNTGACCTNGAGGTACNAANCTANNNAGGNTTAGCTGATACNANCCTNAGACAGTGCACNCCNGACCACGGCTGCGTTAACNAGGCGAATAAGNTANGCCATCGANCCCCCTGTTAGCNNACANANNCNCAAANCANNAAGGGAGTANGAGCNTNGCATCANNNATGGCAGATNATNGACTNNAGGGCAAANANTANNTAAACGTNTTNAAGTNAGGAAGGTCNGNNCNTTAGGTNGTGCNAGAACTNAATATACNCGNNANGANTNGNGAGCTCNAGCTCTCNNAANANGTNCGNAATGGNNCNGTCNNTCTGNGNANCCAANGTNTTNANGGNNTACGTNCGCGCATTTNANAGNAGNNATAGGNNA", "AGCTTGAATCTGTAGTTAAGACATGTAAAGTGNCNTGCGNNGGTAANTGTCCTGAGNGANCTGGGANGNTCCTCTNNGTGAGATAGNGACNTANCGCNNNGAGCGGAGTACGCNCAGNACCCGAATATNAGNGNGANNNNGACAACAGANGGCGTTNCGATGGAGGNTCAGNNNNNGNNAGTTTCGCTCGCGCGTCGAGCNGTTAGCGCAGCGGNNNNTGGTGACNTCNNCGCTANNGTTGTGANCAACGNGGGATCACTATTNCANNCANGAAGNGATGCAACNNNTTTNAGNNNTNNNANTGNCAACNATNCGNAGTGGTTTACNNACTNNTGCTNGNNGTTTTTANTACNNCGTTGGGANATTCACNTNCGNTTNCAGNAGANACGCCNACNCGGCGANGTACCCGANGGGCCCTCGAGNNTCAAANTGGGGNGTTCCCNCCCANCAATAANGNCATAATTTATNCNGNAATTNGGATTGTCCACCCAGAGTAGACATNAACTGTACGGAACTGTGCGCTGGAANCTACNGGNCCTTTCTTGCNAAAATCNAGNTGNTCCCATNGCTNNACGNNACTGGNTCACTGCANAAAGNAATNAGANANANTGNTTACNTNGGNTGCGNNNNTGACGNTGACCTNGAGGTACNAANCTANNNAGGNTTAGCTGATACNANCCTNAGACAGTGCACNCCNNGANCCACGGCTGCGTTAACNAGGCGAATAAGNTANGCCATCGANCCCCCTGTTAGCNNACANANNCNCAAANCANNAAGGGAGTANGAGCNTNGCATCANNNATGGCAGATNATNGACTNNAGGGCAAANANTANTAAACGTNTTNAAGTNAGGAAGTCNGNNCNTTAGGTNGTGCNAGAACTNAATATACNCGNNANGANTNGNGANGCTCNAGCTCTCNNAANAGTNCGNAATGGNNCNGTCNNTCTGNGAANCCAANGTNTTNNGGNNTACGTNCGCGCATTTNANAGNAGNNATAGGNNA", 20usize),
            ("NNCGTGGGCGTCNTNTTAAGGGGCCCTTGNCATTACGATTAATNACATNNCCAAGTGNATGTTNNTGGGTNNGNGGATGCNCGAATTNCCNAATAANAGTAGAATAGTTTTNAGAAANGCNNGGAATNNNTCNGATGTANGCAGTCGANGAAGANNTANGANANGTTNAAACGTANAGGCTNTNCNAGTAATGNTTGNNCCAGGGCCGNCACCCGNACTCAATNTGNACGGAGCCTGGTNTTAANTNAGNTTNCCTATGGNCTGTANTCCGATTGCNNNANTANNCTCATNGAAGCNAAGTAGCAGACNGCNNGTAAGCCACCNAGTAGTAGGCAGGGCCATNGCGCCTNCCNACGCANCNNGTACTGATCGAGCGCCTCAAGACNTNACAGGNTTAANNCTCCNTGGCGGNANTTNANANGTTNANNCGAAANCGCANNCCNACGTCCCNGCTNGNAGAGNCNNCCCNNTCCNNNCGANNCATTTCTGANNGCNTGTTANTTCTTCCCTNNAGAAGTGNGTNCGNTGTANGACCAANCAGGANCCGAGNTCANTGATTTATGATANAANGACCTTTCATAACTGTCANCACAGNGAAANTACGATCNNCNCCANCACCNACCCCTCTTCANNGAGAGGTGTGGATTGCTTCTCATCCGACGGGGTNTGCGAAGTNACTNNCCGNTGTATACATNTNNGTCGCGNNGGNTNCGTNTCAATCANTNATNTTTCGTCGNNNTANGACCTTAGNCNCNGATCNGNNTNCNACGTCNCNNTGANATCNGCACACAGTNCAGAAAGACAGTACTCGTCTAAATGCTTTGGTACNCTTCAACCAGTGCTGANACTGNAAATTNATCGAGGCTTNGCTGGCTTTNNATAGTNGAANAACGNNNTGCATNNNATNCCCNNCGGACNNANCGCNTGTCGNNTTNNTCAGGCCGCTNNTTNCCATTGCGCNGNCAGACNNNGTTATGTTAGTCTANNGCATNNCTCGGAA", "NNCGTGGGCGTCNTNTTAAGGGGCCTTNGNCATTACNGATTAATNACATNNCCAAGTGCNCTGTTNNTGGGTNNGNGGATGCNCGAATTNCCNAATAANAGTAGAATAGTTTTNAGAAANGCNNGGAATNNNTCNGATGTANGTCAGTCGANGAAAGANNTANGANANGTTNAAACGTAAGGCTNTNCNAGTAATGNTTGNNCCAGGGCCGNCACCCNACTCAATNTGNACGGAGCCTGGTNTTAANTNAGNTTNCCTATGGNCTGTTCCGATTCGCNNNANTANNCTCATNGAGCNAAGTAGCAGACNGCNNGGTAAGCCACCNCAGTAGTAGGCAGGGCCATNGCGCCTNCCNACGCANCNNGTACTGATCGAGCGCCTCAAGACCNTNACAGGNTTAANNCTCCNTGGCGGNANTTNANANGTTNANNCGAAANCGCANNCCNACGTCCCNGCTNGNAGAGNCNNCCCNNTCCNNNCGANNCATTTCTGANNGCNTGTTANTTCTCCCTNNAGAAGTGNGTNCGNTGTANGACCAANCAGGANCCGAGNTCANTGATTTATGATANAANGACCTTTCATAACTGCANCACAGNGAAANTACGATCNNCNCCANCACCNACCCCTCTTCANNGAGAGGTGTGGATTGCTTTCATCCGACGGGGTNTGCGAAGTNACTNNCCGNTGTATACATNTNNGTCGCGNGGGNTNCGTNTCAATCNANTNATNTTTCGTCGNNNTANGACCTTAGNCNACNGACTCNGNNTNCNACGCNCANTGGANATCNGCACACAGTNCAGAAAGACAGTACTCGTCTAATGCTTTCGGTACNCTTCCACCAGTGCTGANATGNAAATTNATCGAGGCTTNGCTGGCTTTNNATAGTNGAANAACGNNNTGCATNNNATNCCCNNCGGACNNANCGCNTGTCGNNTNNTGAGGCCGCTNNTTNCCATTGCGCNGNCAGACNNNGTCTGTTGTCTANNGCATNNCTCGGAA", 40usize),
            ("TCCACCTCANGGTNCACANTGTNAAANNNCCTACANTAAATGGNNCTTTATACANGTTCTNNCAGTGTTGCATTATCNGTNGTGCTNACCTTANGNATGNTCATCCATANCCGGTTGNGTGCTTGCCNACTGGATANACANNTCANCCTTCAGTNCTCNGGTAANGCGCNNGCTCAGTTGGGANTCNGTAGAGANCCTTTNTNCTACAANGAAGANCACTGTACTCTNCGACNANGAAANATCAGCNGNGNGGAAACCNTTNTTTCTCCAACAAACTANTNCTNGTNANGACNTTAGGTCCCCNNGANACGTGNTGTGAANNNATNTAGGTTCNAGCNNTCNACNTNGGCANCGNTCCACANNTCTTATGTAGGAGTATCNAGGNTANCCNCTTNTACATNTCCCTCTGCTANCTANTTCNGCAAGCNNCAANGNGCCCTACCNACTCTATTNACNNCGNNNTAANTTGTACCCCCACCCCNNATNGATAGAGCTGCGAATNAGGCNGGNCAAGNANGGCCGCGNNGANAAATGTNATACGNNCGGGTCATANAATGACTATGNNCCACCGTCCNCTGNATATTNAACCCNATCCATGAAGACNCCCCNACNCTCTTTTCAAAGCGGANGNAGNCCTNCGCNCNCGTTTTNNCGNTNCGNNTTCACAGANTTTGAGGATAGATGAAATTAGATANANGGTATTATAGANANNCCGGTTGGGTNAGGGTNAATCNANNCTGGANNNAGCTNNTTNGTACGTTGGANATATCACAGANTNAGCCCACTCGNAGCCNNCGGGNGCCAANGAGAGAAACCTATGGAAAGTCTNGCANGATNNTCGAGGGATATTCNATNCAACCANTGCACGGNTGGCAGTGACTNTAAGCANNANGTGNCGCCGCCTGANTTTGTNCGTTGTNTNTGANGGCTCANATNCTANNNGGCTNNTAANCACCNAGAGGTNGNNTANAGTGTTGNAGGNTGGCNTGTGNTACNNCTN", "TCCACCTCANGGTNCACANTGTNAAANNCCACANTAACTGGNNCTTTTATACANGTTCTNNCAGTGTTGCATTATCNGTNGTGGTNACCTTANGNANTGNTCATCCATANCCGGTTGNGTGNTNTGCCNACTGGATANACANNCANCCTTCAGTNCTGNGGTAANGCGCNNGCTNCAGTTGGGANTCNGTAGAGANCCTTTNTNCTACAANGAANANCACTGTACTCATNCTACNANGAAANATCAGCNGNGNGGAAACCNTNTTTCTCCAACAAACTANTCTNGTNANGACNTTAGTCCCCCNNGANGCTGNTGTNGAANNNATNTAGGTNCNGCNNCNACNGTNGGCANCGNTCCACANNTNCTTTGTAGGAGTATCNAGGNTANCCNCTTNTACATNTNCCCTGCTANCTANCTTCTGCAAGCNGCAANGNGCCCTACCNACTCTATTANCNNCGNNNTAANTTGTACCNCCACCCCANATNGATAGAGCTGCGAATNAGGCNGGNCANAGTNANGGCCGCGNTANAAATGTNATACGNNAGGGTCATANAATGACTATGNNCCACCGTCCNCTGAATTNACCCNATCCATGAAGANCNCCCCNANCTCTTTTCTAAAGCGGANGNAGNCCTNCGTNCNCGTTTNNNCGNTNNGNNTTCACAGANTTTAGATAGATGAAATTAGNATANANGGTATTATAGANANNCTCGGCNTGGGTNAGGGTNAATCNANNCTGGANNNCAGCTNNTTNGTACGTTGGANATATCACAGANTNAGCCCACTCGNAGCCNNCGGGNGGCCCAANGAGAGGAAACCTATGGCAAGTCNTGCANGATNNTCGAGGGATAGTCNATNCAACCATTGCACCGGNTGGCATGACTNTAAGCANNANGTNCGCCGCCTGANTTTGTNCGTTGGTNTNTAGANGGCTCANATNCTANNNGGCTNNTAANCACCGAGAGGTNGNNTANGTGTTGNAGGNTGGCNTGTGNTTCNNCTN", 80usize),
            ("NGAGNTTAGNNTAGCNCGNTNCTNAGNGNATNNTCTCGCNCGCGAACCNCCTCNACAGAGATTTNNTACNTCTNTTTCATTACCCACCTGTCGACCTNNTACCCCAGANNCTAATAANGGACATCTCNGNANNCAAGNCTAANATACTCCNATNATATGCCNTNNACGTANNGGTANTNCACNNATTGAAATCGCGGCAGAATCGNTCGCATGCACCGNGAANGTANAAAAGATANNTTNNNGCCNANAATANNTTCGCATATGAGNANNTAGTANAGGCAAGNCTNNACNNCCTCNNGNATGACNANGNATNGAGGCGGCGAANAGTGCGCTGGGGNAAGNNTTNCGANACGNANNGCNCTNGCCGNCNTCTTAGANCTCACACANTNGATGANCTNATGNCGNAACGGNNGTATANTGTGACTACANNGGGANGCTANNACNTANCTNTCATNAANNCTCNANNTCCCCCGTCACNACANCCCNCCTACCATTANGTGANCTCGTGGTCTNGNGNNAACCTNAGTGATCNAGNCAACTNNNNNCAANNAGAGCGACTNNNACANTGANCNACGCNCNTANTCGTAGGTCTNACANNGTNGCATTGGAGGCCTGANNTAAGGATANNCCCCTAACNCAGCGCCNATTCAGAAGACNCCNANNCNTATACCNCGCTACANCCNCTGATNNACCGNTCNNCTTTACCTGNNCAAAGGNANAACTGCTCCCAAGNTACCTTACTTAAAGAGCGCAAATNGCCTTNTCGTGCGCNTATTTGGACGGGTCACTCATANCATATGGTCTTGTCTANGTATTGGGGNNGGGNTGANAANNNTTNGCCCTANANCCTCNTTAAGANACNCGGNCGTTNCCGTCGTCTCCTCAGTATNATNTTGCTGCGANACGNCCCGTTNACCTGGCNTCCCGNAATNCAGNANACTGACANNGTGGAGACGACCNGTGGNCANNACCNNTTTGGCACNTCCTCNTNCCNGAGTCC","CNGAGNTTAAGNNTCGCNCGANTNCTNAGAGNATNNTCTNCGCNCGCGAACCGCCTCNCAAGAGATTTNNTTTCNTTNCTTCATTACCCACCTGTCGACCTNNTACCCCAGANGNCTAATAANGGAGATCTCNGNANGCAAGNCAANATACTCCCNATNATNTGCCNTNACGTANNGGATANTNCACNNATTGAACGCGGCAGAATCGNGCTATGCACCGNGAANGTANAAAAATANNTTNNNGCCNANAATANNTTCGCATATGAGNTANNTAGTANAGGCAAGNCTNNACNCCCTCNNGNTGCNANGNANGGCGGCGAAAGTGCGCTGGGGNAAGNNTTNCGANACGNANNCNTCTNGCCGNCNTCCNAAAGANCTCACACANTNGATGANCTNATGNCGNACGGNNGTATAGNTGTGGACTACANNGGGTANGCTANNACNTANCTNTCATNAANNCCNANNTCCCCCGTCACNACANCGCCTACCATANGTGANCTCGTGGTCTTNGNGNNAACCTNAGTGNATCNAGNCAANCNNNNNTAANNAGAGCGACTNNNACANTGANCNACGCNCCTANTGTAGGTCTNACANNGTNNGCATTGGAGGNCTANNACGGATANNCCCTAACNCAGCNCCNATTCAGAAGACTNCCNNNCNTAAACCNCGCTACANCCNCTGATNTACCGNTTCNNCTTACCTGNNCAAAGGNANAACTGCTCCCAAGNTACCCTTACTTAAGGGCGCAAATNGCCTTNTCGTGCGGGATTNTGGACGGGTCACTCCTANCATATGGTCTTGTCTANGTATTGGGGNGGGNTGANAANNNTTNGCCCTANANCCTCNTTAAGANACNCGGNCGTTNCCGCGTCTCCTCAGTATNATNTTGCTGGANACGNCCCGTCTNCACCTGGCNTCCCGAATNCAGNANACTGACANNGTGGATGACGACCNGTGGNANCACCNTTTGGCTACNTCCTCNTNCCNGAGTCC",100),
                ("AGTCNNANANNGAATTCTAATTNTNNCCCNGNCAAACATTTNGTTTGGATCCGTAAAACCACGNGGNCNTNNTTGATNGNTCCTTGNATANACTTCTTTCTCTTNCGAAANNNTCACNNCGAGCTGNNCNANGAGNNTCCNGTAGAACGCGTCACNCTGCATTGTTNNANGAAGTTTAGANGNTGNATNTGCGNCACCTNCCTGNGCNANCTTGGTGCAAAANCCTGGNAANCACGAACCCNTNGCCTCGGATGGAAACNTCTACNGCNGGNGGGTNNAGCCGAANCTACTGCNACCAAAANCCCANAGCNGAGNCCTNATANANGTNGCNATNCCNNNGGAGNCGCAGNNTCCCNTGTNCNNGCNTNTNNGGCCCTNATTACANTGAGCNGAGNCGAAATGGCNCCNCNCAAGTCGACCGNTCCTTGTCTNTCAATAAACNNAGNCCACAATNANAANATNCAGNNCGANTTTNNGNGNGTGCTTNAGTTGANNANNCTGGTTCCNCCTGNAGTNGGCTATTAGNNCACGANCTTTTNGNATCATATCTCTATNCNAANTCNTNATNAATNGAAAAGTGNTGNCAGNGGNAATCNNTANCCCCACNNGTCNCCCNNANGATGAATTTAGGATNGANNAGTTNATCNAACNTGNTGTTGNATGNCTTCTTCGACTTNCTGAGTGTATCCTTCNNATNCCANCGTGCGGGTNANANCNACTTNAGGCTTGTNTGATNANNTGGNATACNNNNCTGCCNGTNTAAAGAGTAGGGNCNNATTNNNNGTTNGGAGCTGANACGTTNNCNGCGGCTCTTGGCCTGNGCATTCGNCTGCGNTGCATAGAGANCCCTGANACCCNNNNANGNTTANGGACTNAGGANTNNAGGNNTANGNTATGGATGCCNNGCNGGNATTANCACTTCNAGNCTATATCACGATAGACGTNTCTATGTCTNNNCTNCNGNGAGCATCANNCCNGGGNGTNNTCNNAANTGNATAGA","AGTTNNANANNTGAATTGCAAATTNANNCTCNGNCAAAATTTNGNTTTGGATCCGTTAAANACCACGNGGNCNTNTTGAATNNTCCTTGNATACATTCTTTCTCGATTNACNAGAANCNNTCACCGAGCTGNNCNANGAGNTNCTCCNGTAAGNAACGCCGCCACCNCTGCATTGTNANGAACGTTTAGANGNTGNAANTGGNCACCNCCCGNNGCNANCTTGTGTGCAAAANCCTGGNAACNCACGAATCCNTNGCCTCGGATGAAACNTCGGACNGCNGGNAGGGTNNAGNCNANAANCTATGCNACCAAAANCCANAGCNGAGCCTNCAANANGTNGGCCNATNNCCNNNGGANCGTAGNTTCCCNTGTGCNNGCNTNTNNGTCCTNNATTACANTGAGCNGNNGNCGNAATGGCNCCNCNCAGTCGACCNNCCTTGTCTNTCAAATAGNAACNNNNGNCCAGCAAACTNNGTANAANATNCAGNCGANTTNNGGNGTGCTTNATTGACNNANNACTGTTACCNTCTNAGTNGGCTTGTTAGNACACGANCTTTNGATCATANCTCTANCNAANTCTNTCAAATNGAAAAGTGNTGNCAGNGGNAATCNNTCCCCCCNCNTCNCANCNNANGATGAATTTAGGATNGANAGTTNATCNAACNTCGNTTGTTGNTGNCCTTCGTACTCCGAGGTATCCCTTCNNATNCCCANGTGCGGGTNANACCNACCTTNAGGCTTGTNTGACNANNTGGNAACNNANNCGTCCNGTNTAAAGAGTCGGGCCNNATTNNNGTTNGAGCTTCACGTNNCNGCGGCTCTGGCCTGNGCATATCGACTGCGNTGCATAGACANCCCTGNAACCCNNNNTANGNTTANGGACTNAGANTNNAGGNNTANGNTACTNGATGCCNGCTNGNCATTANCTCTTCNAGNCTATATTCACGATAGANGTNTCTATTTCTNNNCTCNTNGAGACTCGNNCNGGNNGTNNTCNNAANTGNATAGA",200),
        ];
    for (s, t, k) in &test_tuples {
        c.bench_function(&format!("string length 1000 with distance {}, edit distance from editdistancek", k), |b| {
            b.iter(|| {
                black_box(edit_distance_bounded(black_box(s.as_bytes()), black_box(t.as_bytes()), *k));
            })
        });
        c.bench_function(&format!("string length 1000 with distance {}, edit distance from triple_accel", k), |b| {
            b.iter(|| {
                black_box(levenshtein_simd_bounded(black_box(s.as_bytes()), black_box(t.as_bytes()), *k as u32));
            })
        });
    }

    for (s, t, k) in &test_tuples {
        c.bench_function(&format!("string length 1000 with distance {} and 2x threshold, edit distance from editdistancek", k), |b| {
            b.iter(|| {
                black_box(edit_distance_bounded(black_box(s.as_bytes()), black_box(t.as_bytes()), 2 * *k));
            })
        });
        c.bench_function(&format!("string length 1000 with distance {} and 2x threshold, edit distance from triple_accel", k), |b| {
            b.iter(|| {
                black_box(levenshtein_simd_bounded(black_box(s.as_bytes()), black_box(t.as_bytes()), 2 * *k as u32));
            })
        });
    }
}



criterion_group!(benches, edit_distance_benchmark, edit_distance_benchmark_long);
criterion_main!(benches);