﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SageMaker {
namespace Model {

/**
 * <p>Provides information about the location that is configured for storing model
 * artifacts. </p> <p>Model artifacts are outputs that result from training a
 * model. They typically consist of trained parameters, a model definition that
 * describes how to compute inferences, and other metadata. A SageMaker container
 * stores your trained model artifacts in the <code>/opt/ml/model</code> directory.
 * After training has completed, by default, these artifacts are uploaded to your
 * Amazon S3 bucket as compressed files.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sagemaker-2017-07-24/ModelArtifacts">AWS
 * API Reference</a></p>
 */
class ModelArtifacts {
 public:
  AWS_SAGEMAKER_API ModelArtifacts() = default;
  AWS_SAGEMAKER_API ModelArtifacts(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API ModelArtifacts& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The path of the S3 object that contains the model artifacts. For example,
   * <code>s3://bucket-name/keynameprefix/model.tar.gz</code>.</p>
   */
  inline const Aws::String& GetS3ModelArtifacts() const { return m_s3ModelArtifacts; }
  inline bool S3ModelArtifactsHasBeenSet() const { return m_s3ModelArtifactsHasBeenSet; }
  template <typename S3ModelArtifactsT = Aws::String>
  void SetS3ModelArtifacts(S3ModelArtifactsT&& value) {
    m_s3ModelArtifactsHasBeenSet = true;
    m_s3ModelArtifacts = std::forward<S3ModelArtifactsT>(value);
  }
  template <typename S3ModelArtifactsT = Aws::String>
  ModelArtifacts& WithS3ModelArtifacts(S3ModelArtifactsT&& value) {
    SetS3ModelArtifacts(std::forward<S3ModelArtifactsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_s3ModelArtifacts;
  bool m_s3ModelArtifactsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
