// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

package ebpfcommon

import (
	"bufio"
	"bytes"
	"errors"
	"io"
	"log/slog"
	"net"
	"os"
	"strings"
	"sync"
	"time"
	"unsafe"

	"github.com/cilium/ebpf"
	"github.com/cilium/ebpf/asm"
	"github.com/cilium/ebpf/link"
	lru "github.com/hashicorp/golang-lru/v2"
	"github.com/hashicorp/golang-lru/v2/expirable"
	"github.com/hashicorp/golang-lru/v2/simplelru"

	"go.opentelemetry.io/obi/pkg/appolly/app/request"
	"go.opentelemetry.io/obi/pkg/config"
	"go.opentelemetry.io/obi/pkg/ebpf/common/dnsparser"
	"go.opentelemetry.io/obi/pkg/internal/ebpf/kafkaparser"
	"go.opentelemetry.io/obi/pkg/internal/ebpf/ringbuf"
	"go.opentelemetry.io/obi/pkg/pipe/msg"
)

//go:generate $BPF2GO -cc $BPF_CLANG -cflags $BPF_CFLAGS -target amd64,arm64 -type http_request_trace_t -type sql_request_trace_t -type http_info_t -type connection_info_t -type http2_grpc_request_t -type tcp_req_t -type kafka_client_req_t -type kafka_go_req_t -type redis_client_req_t -type tcp_large_buffer_t -type otel_span_t -type mongo_go_client_req_t -type dns_req_t Bpf ../../../bpf/common/common.c -- -I../../../bpf

// HTTPRequestTrace contains information from an HTTP request as directly received from the
// eBPF layer. This contains low-level C structures for accurate binary read from ring buffer.
type (
	HTTPRequestTrace     BpfHttpRequestTraceT
	SQLRequestTrace      BpfSqlRequestTraceT
	BPFHTTPInfo          BpfHttpInfoT
	BPFConnInfo          BpfConnectionInfoT
	TCPRequestInfo       BpfTcpReqT
	GoSaramaClientInfo   BpfKafkaClientReqT
	GoRedisClientInfo    BpfRedisClientReqT
	GoKafkaGoClientInfo  BpfKafkaGoReqT
	TCPLargeBufferHeader BpfTcpLargeBufferT
	GoOTelSpanTrace      BpfOtelSpanT
	GoMongoClientInfo    BpfMongoGoClientReqT
	DNSInfo              BpfDnsReqT
)

const (
	EventTypeSQL            = 5  // EVENT_SQL_CLIENT
	EventTypeKHTTP          = 6  // HTTP Events generated by kprobes
	EventTypeKHTTP2         = 7  // HTTP2/gRPC Events generated by kprobes
	EventTypeTCP            = 8  // Unknown TCP protocol to be classified by user space
	EventTypeGoSarama       = 9  // Kafka client for Go (Shopify/IBM Sarama)
	EventTypeGoRedis        = 10 // Redis client for Go
	EventTypeGoKafkaGo      = 11 // Kafka-Go client from Segment-io
	EventTypeTCPLargeBuffer = 12 // Dynamically sized TCP buffers
	EventOTelSDKGo          = 13 // OTel SDK manual span
	EventTypeGoMongo        = 14 // Go MongoDB spans
	EventTypeFailedConnect  = 15 // Failed Connections
	EventTypeDNS            = 16 // DNS events
)

// Kernel-side classification
const (
	ProtocolTypeUnknown uint8 = iota
	ProtocolTypeMySQL
	ProtocolTypePostgres
)

var IntegrityModeOverride = false

var ActiveNamespaces = make(map[uint32]uint32)

// ProbeDesc holds the information of the instrumentation points of a given
// function/symbol
type ProbeDesc struct {
	// Required, if true, will cancel the execution of the eBPF Tracer
	// if the function has not been found in the executable
	Required bool

	// The eBPF program to attach to the symbol as a uprobe (either to the
	// symbol name or to StartOffset)
	Start *ebpf.Program

	// The eBPF program to attach to the symbol either as a uretprobe or as a
	// uprobe to ReturnOffsets
	End *ebpf.Program

	// Optional offset to the start of the symbol
	StartOffset uint64

	// Optional list of the offsets of every RET instruction in the symbol
	ReturnOffsets []uint64
}

type Filter struct {
	io.Closer
	Fd int
}

type SockOps struct {
	io.Closer
	Program       *ebpf.Program
	AttachAs      ebpf.AttachType
	SockopsCgroup link.Link
}

type SockMsg struct {
	io.Closer
	Program  *ebpf.Program
	MapFD    int
	AttachAs ebpf.AttachType
}

type Iter struct {
	Program *ebpf.Program
	Link    link.Link
}

type MisclassifiedEvent struct {
	EventType int
	TCPInfo   *TCPRequestInfo
}

type EBPFParseContext struct {
	protocolDebug              bool
	h2c                        *lru.Cache[uint64, h2Connection]
	redisDBCache               *simplelru.LRU[BpfConnectionInfoT, int]
	largeBuffers               *expirable.LRU[largeBufferKey, *largeBuffer]
	mongoRequestCache          PendingMongoDBRequests
	mysqlPreparedStatements    *simplelru.LRU[mysqlPreparedStatementsKey, string]
	postgresPreparedStatements *simplelru.LRU[postgresPreparedStatementsKey, string]
	postgresPortals            *simplelru.LRU[postgresPortalsKey, string]
	kafkaTopicUUIDToName       *simplelru.LRU[kafkaparser.UUID, string]
	payloadExtraction          config.PayloadExtraction
	dnsEvents                  *expirable.LRU[dnsparser.DNSId, *request.Span]
}

type EBPFEventContext struct {
	CommonPIDsFilter ServiceFilter
	SharedRingBuffer *ringBufForwarder
	EBPFMaps         map[string]*ebpf.Map
	RingBufLock      sync.Mutex
	MapsLock         sync.Mutex
	LoadLock         sync.Mutex
}

var MisclassifiedEvents = make(chan MisclassifiedEvent)

func ptlog() *slog.Logger { return slog.With("component", "ebpf.ProcessTracer") }

func NewEBPFParseContext(cfg *config.EBPFTracer, spansChan *msg.Queue[[]request.Span], filter ServiceFilter) *EBPFParseContext {
	var (
		err                        error
		protocolDebug              bool
		redisDBCache               *simplelru.LRU[BpfConnectionInfoT, int]
		mysqlPreparedStatements    *simplelru.LRU[mysqlPreparedStatementsKey, string]
		postgresPreparedStatements *simplelru.LRU[postgresPreparedStatementsKey, string]
		postgresPortals            *simplelru.LRU[postgresPortalsKey, string]
		kafkaTopicUUIDToName       *simplelru.LRU[kafkaparser.UUID, string]
		mongoRequestCache          PendingMongoDBRequests
		payloadExtraction          config.PayloadExtraction
		dnsEvents                  *expirable.LRU[dnsparser.DNSId, *request.Span]
	)

	h2c, _ := lru.New[uint64, h2Connection](1024 * 10)
	largeBuffers := expirable.NewLRU[largeBufferKey, *largeBuffer](1024, nil, 5*time.Minute)

	if cfg != nil {
		protocolDebug = cfg.ProtocolDebug

		if cfg.RedisDBCache.Enabled {
			redisDBCache, err = simplelru.NewLRU[BpfConnectionInfoT, int](cfg.RedisDBCache.MaxSize, nil)
			if err != nil {
				ptlog().Error("failed to create Redis DB cache", "error", err)
				redisDBCache = nil
			}
		}

		mysqlPreparedStatements, err = simplelru.NewLRU[mysqlPreparedStatementsKey, string](cfg.MySQLPreparedStatementsCacheSize, nil)
		if err != nil {
			ptlog().Error("failed to create MySQL prepared statements cache", "error", err)
		}

		postgresPreparedStatements, err = simplelru.NewLRU[postgresPreparedStatementsKey, string](cfg.PostgresPreparedStatementsCacheSize, nil)
		if err != nil {
			ptlog().Error("failed to create Postgres prepared statements cache", "error", err)
		}

		postgresPortals, err = simplelru.NewLRU[postgresPortalsKey, string](cfg.PostgresPreparedStatementsCacheSize, nil)
		if err != nil {
			ptlog().Error("failed to create Postgres portals cache", "error", err)
		}

		kafkaTopicUUIDToName, err = simplelru.NewLRU[kafkaparser.UUID, string](cfg.KafkaTopicUUIDCacheSize, nil)
		if err != nil {
			ptlog().Error("failed to create Kafka topic UUID to name cache", "error", err)
		}

		mongoRequestCache = expirable.NewLRU[MongoRequestKey, *MongoRequestValue](cfg.MongoRequestsCacheSize, nil, 0)

		payloadExtraction = cfg.PayloadExtraction

		dnsEvents = expirable.NewLRU(1024, dnsEventExpireHandler(spansChan, filter), cfg.DNSRequestTimeout)
	}

	return &EBPFParseContext{
		protocolDebug:              protocolDebug,
		h2c:                        h2c,
		redisDBCache:               redisDBCache,
		largeBuffers:               largeBuffers,
		mongoRequestCache:          mongoRequestCache,
		mysqlPreparedStatements:    mysqlPreparedStatements,
		postgresPreparedStatements: postgresPreparedStatements,
		postgresPortals:            postgresPortals,
		kafkaTopicUUIDToName:       kafkaTopicUUIDToName,
		payloadExtraction:          payloadExtraction,
		dnsEvents:                  dnsEvents,
	}
}

func NewEBPFEventContext() *EBPFEventContext {
	return &EBPFEventContext{
		EBPFMaps:    map[string]*ebpf.Map{},
		RingBufLock: sync.Mutex{},
		MapsLock:    sync.Mutex{},
		LoadLock:    sync.Mutex{},
	}
}

func ReadBPFTraceAsSpan(parseCtx *EBPFParseContext, cfg *config.EBPFTracer, record *ringbuf.Record, filter ServiceFilter) (request.Span, bool, error) {
	if len(record.RawSample) == 0 {
		return request.Span{}, true, errors.New("invalid ringbuffer record size")
	}

	eventType := record.RawSample[0]

	switch eventType {
	case EventTypeSQL:
		return ReadSQLRequestTraceAsSpan(record)
	case EventTypeKHTTP:
		return ReadHTTPInfoIntoSpan(parseCtx, record, filter)
	case EventTypeKHTTP2:
		return ReadHTTP2InfoIntoSpan(parseCtx, record, filter)
	case EventTypeTCP:
		return ReadTCPRequestIntoSpan(parseCtx, cfg, record, filter)
	case EventTypeGoSarama:
		return ReadGoSaramaRequestIntoSpan(record)
	case EventTypeGoRedis:
		return ReadGoRedisRequestIntoSpan(record)
	case EventTypeGoMongo:
		return ReadGoMongoRequestIntoSpan(record)
	case EventTypeGoKafkaGo:
		return ReadGoKafkaGoRequestIntoSpan(record)
	case EventTypeTCPLargeBuffer:
		return appendTCPLargeBuffer(parseCtx, record)
	case EventOTelSDKGo:
		return ReadGoOTelEventIntoSpan(record)
	case EventTypeFailedConnect:
		return ReadFailedConnectIntoSpan(record, filter)
	case EventTypeDNS:
		return readDNSEventIntoSpan(parseCtx, record)
	}

	event, err := ReinterpretCast[HTTPRequestTrace](record.RawSample)
	if err != nil {
		return request.Span{}, true, err
	}

	return HTTPRequestTraceToSpan(event), false, nil
}

func ReinterpretCast[T any](b []byte) (*T, error) {
	var zero T

	if len(b) < int(unsafe.Sizeof(zero)) {
		return nil, errors.New("byte slice too short")
	}

	return (*T)(unsafe.Pointer(unsafe.SliceData(b))), nil
}

func ReadSQLRequestTraceAsSpan(record *ringbuf.Record) (request.Span, bool, error) {
	event, err := ReinterpretCast[SQLRequestTrace](record.RawSample)
	if err != nil {
		return request.Span{}, true, err
	}

	return SQLRequestTraceToSpan(event), false, nil
}

type KernelLockdown uint8

const (
	KernelLockdownNone KernelLockdown = iota + 1
	KernelLockdownIntegrity
	KernelLockdownConfidentiality
	KernelLockdownOther
)

func SupportsContextPropagationWithProbe(log *slog.Logger) bool {
	kernelMajor, kernelMinor := KernelVersion()
	log.Debug("Linux kernel version", "major", kernelMajor, "minor", kernelMinor)

	if kernelMajor < 5 || (kernelMajor == 5 && kernelMinor < 10) {
		log.Debug("Found Linux kernel earlier than 5.10, Go trace context propagation at library level is supported", "major", kernelMajor, "minor", kernelMinor)
		return true
	}

	// bpf_probe_write_user(), used to inject the context, requires CAP_SYS_ADMIN

	if !hasCapSysAdmin() {
		log.Info("Go context propagation at library level disabled due to missing capability CAP_SYS_ADMIN")
		return false
	}

	lockdown := KernelLockdownMode()

	if lockdown == KernelLockdownNone {
		log.Debug("Kernel not in lockdown mode, Go trace context propagation at library level is supported.")
		return true
	}

	return false
}

func SupportsEBPFLoops(log *slog.Logger, overrideKernelVersion bool) bool {
	if overrideKernelVersion {
		log.Debug("Skipping kernel version check for bpf_loop functionality: user supplied confirmation of support")
		return true
	}
	kernelMajor, kernelMinor := KernelVersion()
	return kernelMajor > 5 || (kernelMajor == 5 && kernelMinor >= 17)
}

func FixupSpec(spec *ebpf.CollectionSpec, overrideKernelVersion bool) {
	if !SupportsEBPFLoops(ptlog(), overrideKernelVersion) {
		// Hack: instead of redefining bpf2go generated struct for mutually exclusive conditional programs,
		// use one predefined field name to store either of them.
		spec.Programs["obi_protocol_http"] = spec.Programs["obi_protocol_http_legacy"]
		spec.Programs["obi_protocol_http"].Name = "obi_protocol_http"
	}
	// Hack: insert a dummy unused program in order to be able to use bpf2go generated struct to load
	// the collection.
	spec.Programs["obi_protocol_http_legacy"] = &ebpf.ProgramSpec{
		Name: "obi_dummy",
		Type: ebpf.Kprobe,
		Instructions: asm.Instructions{
			asm.Mov.Imm(asm.R0, 0),
			asm.Return(),
		},
		License: "MIT",
	}
}

// Injectable for tests
var lockdownPath = "/sys/kernel/security/lockdown"

func KernelLockdownMode() KernelLockdown {
	plog := ptlog()
	plog.Debug("checking kernel lockdown mode, [none] allows us to propagate trace context")
	// If we can't find the file, assume no lockdown
	if _, err := os.Stat(lockdownPath); err == nil {
		f, err := os.Open(lockdownPath)
		if err != nil {
			plog.Warn("failed to open /sys/kernel/security/lockdown, assuming lockdown [integrity]", "error", err)
			return KernelLockdownIntegrity
		}

		defer f.Close()
		scanner := bufio.NewScanner(f)
		if scanner.Scan() {
			lockdown := scanner.Text()
			switch {
			case strings.Contains(lockdown, "[none]"):
				return KernelLockdownNone
			case strings.Contains(lockdown, "[integrity]"):
				return KernelLockdownIntegrity
			case strings.Contains(lockdown, "[confidentiality]"):
				return KernelLockdownConfidentiality
			default:
				return KernelLockdownOther
			}
		}

		plog.Warn("file /sys/kernel/security/lockdown is empty, assuming lockdown [integrity]")
		return KernelLockdownIntegrity
	}

	plog.Debug("can't find /sys/kernel/security/lockdown, assuming no lockdown")
	return KernelLockdownNone
}

func cstr(chars []uint8) string {
	addrLen := bytes.IndexByte(chars, 0)
	if addrLen < 0 {
		addrLen = len(chars)
	}

	return string(chars[:addrLen])
}

func (connInfo *BPFConnInfo) reqHostInfo() (source, target string) {
	src := make(net.IP, net.IPv6len)
	dst := make(net.IP, net.IPv6len)
	copy(src, connInfo.S_addr[:])
	copy(dst, connInfo.D_addr[:])

	srcStr := src.String()
	dstStr := dst.String()

	if src.IsUnspecified() {
		srcStr = ""
	}

	if dst.IsUnspecified() {
		dstStr = ""
	}

	return srcStr, dstStr
}

func isClientEvent(et uint8) bool {
	switch request.EventType(et) {
	case request.EventTypeGRPCClient, request.EventTypeHTTPClient, request.EventTypeRedisClient,
		request.EventTypeKafkaClient, request.EventTypeSQLClient, request.EventTypeMongoClient,
		request.EventTypeFailedConnect:
		return true
	}

	return false
}

func directionByPacketType(pt uint8, isClient bool) uint8 {
	if isClient {
		if pt == packetTypeRequest {
			return directionSend
		}
		return directionRecv
	}
	if pt == packetTypeRequest {
		return directionRecv
	}
	return directionSend
}
