/*
 * This file is part of GNAC - Gnome Audio Converter
 *
 * Copyright (C) 2007 - 2012 Gnac
 *
 *    - DUPASQUIER  Benoit    <bdupasqu@src.gnome.org>
 *    - JOAQUIM     David     <djoaquim@src.gnome.org>
 *    - ROUX        Alexandre <alexroux@src.gnome.org>
 *
 * GNAC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNAC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNAC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <glib/gstdio.h>

#include "../gnac-profiles-properties.h"
#include "gnac-profiles-opus.h"


typedef struct
{
  AudioProfileGeneric *generic;

  gchar *audio_type;
  gchar *bitrate_mode;
  gchar *bitrate;
}
AudioProfileOPUS;

BasicFormatInfo opus_bfi = {
  PKGDATADIR "/profiles/gnac-profiles-opus.xml",
  NULL,
  PKGDATADIR "/profiles/opus.xml",
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL
};


static const gchar *
gnac_profiles_opus_init(void)
{
  gnac_profiles_default_init(&opus_bfi);

  // Audio type
  gnac_profiles_utils_init_widget(&opus_bfi, "combo-audio-type",
      "//variable[@id='audio-type']");

  // Bitrate mode
  gnac_profiles_utils_init_widget(&opus_bfi, "combo-bitrate-mode",
      "//variable[@id='bitrate-mode']");

  // Bitrate
  gnac_profiles_utils_init_widget(&opus_bfi, "combo-bitrate",
      "//variable[@id='bitrate']");

  gnac_profiles_xml_engine_free_doc_xpath(opus_bfi.doc);
  opus_bfi.doc = NULL;

  return opus_bfi.format_id;
}


void
gnac_profiles_opus_generate_pipeline(void)
{
  gchar *pipeline = gnac_profiles_default_generate_pipeline(&opus_bfi);
  pipeline = gnac_profiles_utils_add_properties(pipeline, &opus_bfi,
      "combo-audio-type", "combo-bitrate-mode", "combo-bitrate", NULL);

  pipeline = gnac_profiles_utils_add_pipe(pipeline,
      opus_bfi.pipeline_multiplexers);

  gnac_profiles_properties_update_textbuffer(pipeline);

  g_free(opus_bfi.pipeline);

  opus_bfi.pipeline = pipeline;
}


static void
gnac_profiles_opus_reset_ui(void)
{
  gnac_profiles_default_reset_ui(&opus_bfi);
  gnac_profiles_utils_reset_values(&opus_bfi,
       "combo-audio-type", "combo-bitrate-mode", "combo-bitrate", NULL);
}


static void
gnac_profiles_opus_set_fields(gpointer data)
{
  if (!data) {
    gnac_profiles_opus_reset_ui();
    return;
  }

  AudioProfileOPUS *profile = (AudioProfileOPUS *) data;
  gnac_profiles_default_init_fields(profile->generic, &opus_bfi);
  gnac_profiles_utils_set_values(&opus_bfi,
      "combo-audio-type", profile->audio_type,
      "combo-bitrate-mode", profile->bitrate_mode,
      "combo-bitrate", profile->bitrate,
      NULL);
}


static gchar *
gnac_profiles_opus_get_combo_format_name(void)
{
  return gnac_profiles_default_get_combo_format_name(&opus_bfi);
}


static void
gnac_profiles_opus_free_audio_profile(gpointer data)
{
  if (!data) return;

  AudioProfileOPUS *profile = (AudioProfileOPUS *) data;

  gnac_profiles_utils_free_audio_profile_generic(profile->generic);

  g_free(profile->audio_type);
  g_free(profile->bitrate_mode);
  g_free(profile->bitrate);
  g_free(profile);
}


static gpointer
gnac_profiles_opus_generate_audio_profile(GError **error)
{
  AudioProfileGeneric *generic = gnac_profiles_default_generate_audio_profile(
      &opus_bfi);
  AudioProfileOPUS *profile = g_malloc(sizeof(AudioProfileOPUS));

  profile->generic = generic;

  gnac_profiles_utils_get_values_and_set(&opus_bfi,
      "combo-audio-type", &profile->audio_type,
      "combo-bitrate-mode", &profile->bitrate_mode,
      "combo-bitrate", &profile->bitrate,
      NULL);

  return profile;
}


static GtkWidget *
gnac_profiles_opus_get_widget(void)
{
  return gnac_profiles_default_get_properties_alignment(&opus_bfi);
}


static void
gnac_profiles_opus_save_profile(gpointer data)
{
  if (!data) return;

  AudioProfileOPUS *profile = (AudioProfileOPUS *) data;
  XMLDoc *doc = gnac_profiles_default_save_profile(profile->generic, &opus_bfi);
  gnac_profiles_xml_engine_add_values(doc,
      "combo-audio-type", profile->audio_type,
      "combo-bitrate-mode", profile->bitrate_mode,
      "combo-bitrate", profile->bitrate,
      NULL);
  gnac_profiles_xml_engine_save_doc(doc, profile->generic->name);

  gnac_profiles_xml_engine_free_doc_xpath(doc);
}


static gpointer
gnac_profiles_opus_load_specific_properties(XMLDoc              *doc,
                                            AudioProfileGeneric *generic)
{
  AudioProfileOPUS *profile = g_malloc(sizeof(AudioProfileOPUS));

  profile->generic = generic;
  gnac_profiles_utils_load_saved_profile(doc,
      "/audio-profile/format-specific/",
      "combo-audio-type", &profile->audio_type,
      "combo-bitrate-mode", &profile->bitrate_mode,
      "combo-bitrate", &profile->bitrate,
      NULL);

  return profile;
}


static void
gnac_profiles_opus_clean_up(void)
{
  gnac_profiles_default_clean_up(&opus_bfi);
  gnac_profiles_utils_free_values(&opus_bfi,
      "combo-audio-type", "combo-bitrate-mode", "combo-bitrate", NULL);
  gnac_profiles_utils_free_basic_format_info(&opus_bfi);
}


static const gchar *
gnac_profiles_opus_get_plugin_name(void)
{
  return opus_bfi.format_plugin_name;
}


static const gchar *
gnac_profiles_opus_get_description(void)
{
  return opus_bfi.description;
}


FormatModuleFuncs
gnac_profiles_opus_get_funcs(void)
{
  FormatModuleFuncs funcs = {
    gnac_profiles_opus_init,
    gnac_profiles_opus_get_description,
    gnac_profiles_opus_generate_pipeline,
    gnac_profiles_opus_generate_audio_profile,
    gnac_profiles_opus_free_audio_profile,
    gnac_profiles_opus_set_fields,
    gnac_profiles_opus_get_widget,
    gnac_profiles_opus_save_profile,
    gnac_profiles_opus_load_specific_properties,
    gnac_profiles_opus_clean_up,
    NULL,
    gnac_profiles_opus_get_combo_format_name,
    gnac_profiles_opus_get_plugin_name
  };

  return funcs;
}
