/*
 * SOURCE:  init.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: initialization routines
 *
 * UPDATES: 12/06/1991 - major rewrite
 *          11/03/1993 - options added
 *
 * (c)M.Schollmeyer
 */
#include "main.h"
#include "version.h"

/* The following table is defined in init.c */
extern void *profiledesc[MAXPROFILEENTRIES];

extern struct Editor ed;       // 'ed' is defined in edit.c
extern struct MemoryHeader mh; // 'mh' is defined in memory.h
extern char *IniName, *TmpPath, *MyPath; // defined in main.c
extern BOOL periodic_backup;
extern struct SharedMemory *SharedMemory;
extern struct IntuitionBase *IntuitionBase;

static BOOL ovl_loaded = FALSE;
static BOOL create_ini = FALSE;
static BOOL freezevmode = FALSE;
static BOOL foundini = FALSE;

extern BOOL ini_loaded;
extern BOOL periodic_backup;
extern BOOL ignore_palette;
extern int old_breakflag;

char *HlpName;

static void _addarg( char * );

/*
 *    Name: GetProfile
 *  Return: TRUE if successful, FALSE if not
 * Purpose: Writes the profile data to a specified file
 *
 *
 *
 * (c)M.Schollmeyer
 */
BOOL GetProfile( char *fname ) {

    int                 fptr   = -1,
                        ret    = FALSE;
    long                size;
    unsigned char _far *buf    = 0L,
                  _far *scan,
                  _far *end;
    unsigned int        slot,
                        esize;
    void               *eptr;

    /* first, try to open profile */
    fptr = Open( fname, ACCESS_READ );
    if( fptr == -1 ) {
        printf( errmsg(EM_PROFNOTFOUND|EM_PROFILE_C), fname );
        goto done;
    }

    /* profile opened successfully, get file size */
    size = Seek( fptr, 0L, SEEK_END );
    Seek( fptr, 0L, SEEK_START );

    /* the file must have at least thes size of the magic chars */
    if( size < (long)sizeof( PROFILEMAGIC ) ) {
        printf( errmsg(EM_PROFCORRUPT|EM_PROFILE_C), fname );
        goto done;
    }

    buf = AllocHeap( (unsigned)size );
    if( buf == 0L ) {
        printf( errmsg(EM_PROFERROR|EM_PROFILE_C), errmsg(EM_OUTOFMEM) );
        goto done;
    }

    /* read file, assuming max. size of 0xffff */
    if( _Read( fptr, buf, (int)size ) != (int)size ) {
        printf( errmsg(EM_PROFERROR|EM_PROFILE_C), errmsg(EM_READERROR) );
        goto done;
    }

    /* set 'scan' to top of memory, 'end' to end */
    scan = buf;
    end = scan + size;

    /* check the magic chars */
    if( *(unsigned long _far *)scan != PROFILEMAGIC ) {
        printf( errmsg(EM_PROFCORRUPT|EM_PROFILE_C), fname );
        goto done;
    }

    /* skip the magic chars in 'scan' */
    scan += sizeof( unsigned long );

    /* MAIN SCAN LOOP.
       now we scan for each entry of the format:

       offset  size  comment
       
            0     2  index
            2     2  length of entry
            4     n  entry
       

    */
    while( scan < end ) {
        /* get the slot to put the entry to */
        slot = *(unsigned int _far *)scan;
        scan += sizeof(unsigned int);
        /* get the size of the entry */
        esize = *(unsigned int _far *)scan;
        scan += sizeof(unsigned int);

        /* check slot for bounds */
        if( slot > MAXPROFILEENTRIES ) {
            printf( errmsg(EM_PROFERROR|EM_PROFILE_C), "Entry Offset out of Bounds" );
        } else if( !PutProfileData( slot, scan, esize ) ) {
            printf( errmsg(EM_PROFERROR|EM_PROFILE_C), errmsg(EM_OUTOFMEM) );
            ret = FALSE;
            goto done;
        }

        /* set 'scan' to top of next entry */
        scan += esize;
    }

    ret = TRUE;

done:
    if( fptr != -1 )
        Close( fptr );
    if( buf )
        free( buf );

    return ret;
}


/*
 *    Name: PutProfile
 *  Return: TRUE if successful, FALSE if not
 * Purpose: Writes profile entries to a specified file
 *
 *
 *
 * (c)M.Schollmeyer
 */
BOOL PutProfile( char *fname ) {

    static unsigned long magic = PROFILEMAGIC;

    int          fptr  = -1,
                 ret   = FALSE;
    unsigned int slot,
                 esize;
    void        *eptr;

    fptr = Create( fname, 0 );
    if( fptr == -1 ) {
        PutError(-1, fname );
        goto done;
    }

    _Write( fptr, (unsigned char *)&magic, sizeof( unsigned long ) );
    for( slot = 0; slot < MAXPROFILEENTRIES; ++slot ) {
        eptr = profiledesc[slot];
        if( eptr ) {
            esize = _msize( eptr );
            _Write( fptr, (char _far *)&slot, sizeof( unsigned int ) );
            _Write( fptr, (char _far *)&esize, sizeof( unsigned int ) );
            _Write( fptr, eptr, esize );
        }
    }

    ret = TRUE;
done:
    if( fptr != -1 )
        Close( fptr );

    return ret;
}


/*
 *    Name: get_init, put_init
 *  Return: TRUE if successful, FALSE if not
 * Purpose: read from and writes the ini-file to disk using
 *          PutProfile() and GetProfile().
 *
 *          if reason is set to TRUE, the current directory is set to
 *          the value contained in PR_INIPATH, else no current directroy
 *          is set.
 *
 * (c)M.Schollmeyer
 */
int get_init( int reason )
{
    char *cp, *env;
    unsigned char *pptr;
    int i;

    /* Delete all previously assigned values. */
    DeleteTmps();
    for (i=0; i<MAXPROFILEENTRIES; ++i)
        ClearProfileData( i );

    /* Load the settings from disk. */
    if( !GetProfile( IniName ) )
        return 0;

    /* Assign common variables to direct addresses for faster access. */
    ed.WordWrap = GetProfileInt( PR_WORDWRAP );
    ed.TabStops = GetProfileInt( PR_TABSTOPS );
    ed.CurrFile = GetProfileInt( PR_CURRFILE );
    ed.LastFile = GetProfileInt( PR_LASTFILE );

    ed.Flags = GetProfileLong( PR_FLAGS );
    ed.SearchFlags = GetProfileInt( PR_SEARCHFLAGS );

    if( GetProfileLong( PR_PERISAV ) )
        periodic_backup = TRUE;
    else
        periodic_backup = FALSE;

    if( reason )
        SetCurrDir( GetProfileString( PR_INIPATH, &profnull ) );

    return TRUE;
}


int put_init( void )
{
    int i, size;
    char *cp, *envstr;

    PutProfileInt( PR_WORDWRAP, ed.WordWrap );
    PutProfileInt( PR_TABSTOPS, ed.TabStops );
    PutProfileInt( PR_CURRFILE, ed.CurrFile );
    PutProfileInt( PR_LASTFILE, ed.LastFile );

    PutProfileLong( PR_FLAGS, ed.Flags );
    PutProfileInt( PR_SEARCHFLAGS, ed.SearchFlags );

    e_savekoords();

    /* put environment */
    for( i = 0, size = 1; environ[i]; ++i )
        size += strlen( environ[i] ) + 1;
    envstr = AllocHeap( size );
    if( envstr ) {
        for( i = 0, cp = envstr; environ[i]; ++i ) {
            strcpy( cp, environ[i] );
            cp += strlen( environ[i] ) + 1;
        }
        *cp = '\0';
        PutProfileData( PR_ENVIRONMENT, envstr, size );
    }

    free( envstr );

    return PutProfile( IniName );
}

BOOL maininit( char **argv, int argc ) {

    char *cp, *newini;
    int i;
    char buffer[BUFSIZ];

    if( !m_init( ) )  return FALSE;

    old_breakflag = BreakFlag( 0 );
    SetHandler( critical_error );
    SetHook( _printtime, HOOK_TIME );
    SetHook( help, HOOK_HELP );
    SetHook( m_putmsg, HOOK_MSG );

    // create MyPath
    cp = argv[0];
    while( *cp ) ++cp;

    // scan reverse until top of string
    while( cp > argv[0] ) {
        if( *cp == '\\' || *cp == ':' ) {
            *(cp+1) = '\0';
            break;
        }
        --cp;
    }
    if( cp == argv[0] ) {
        /* argv[0] contains no path */
        cp = malloc( _MAX_PATH );
        GetCurrDir( cp );
        if( cp[strlen(cp)-1] != '\\' )
            strcat( cp, "\\" );
    } else
        cp = argv[0];

    MyPath = strdup( cp );
    if( MyPath == NULL ) {
        printf( errmsg( EM_OUTOFMEM|EM_INI_C ) );
        return FALSE;
    }

    if( HlpName = malloc( strlen( MyPath ) + 16 ) ) {
        strcpy( HlpName, MyPath );
        strcat( HlpName, HLP_NAME );
    } else {
        printf( errmsg( EM_OUTOFMEM|EM_INI_C ) );
        return FALSE;
    }

    if( IniName = malloc( _MAX_PATH ) ) {
        strcpy( IniName, MyPath );
    } else {
        printf( errmsg( EM_OUTOFMEM|EM_INI_C ) );
        return FALSE;
    }

    for( i = 1; i < argc; ++i ) {
        _addarg( argv[i] );
    }

#ifndef DEBUG
    /* Look if SharedMemory is initialized. */
    if (SharedMemory->magic != SM_MAGIC) {
        printf( errmsg( EM_ILLEGALMAGIC|EM_INI_C ) );
        /* Exit immediately, this is serious. */
        ExitProc(0);
    }
#endif

    if( ! foundini ) {
        strcat( IniName, INI_NAME );
    }

    ini_loaded = get_init( !ovl_loaded );
    if( create_ini )
        ini_loaded = TRUE;

    cp = getenv( "TMP" );
    if( cp ) {
        TmpPath = malloc( strlen( cp ) + 2 );
        if( TmpPath == NULL ) {
            printf( errmsg( EM_OUTOFMEM|EM_INI_C ) );
            return FALSE;
        }
        strcpy( TmpPath, cp );
        i = strlen( TmpPath ) - 1;
        if( TmpPath[i] != '\\' ) {
            TmpPath[i+1] = '\\';
            TmpPath[i+2] = '\0';
        }
    } else
        TmpPath = MyPath;

    SetLearnKey( 0x260c );   // set learn key to Ctrl+L

    IntuitionBase = OpenIntuition( 0 );
    if( IntuitionBase == 0L )
        return FALSE;

    SetHelpKey( BK_F1 );

    i = -1;
    if( !freezevmode )
        i = *(int *)GetProfileData( PR_VIDEOMODE, &i );

    if( !setvmode(i) ) {
        printf( errmsg( EM_NOVMODE|EM_INI_C ) );
        return FALSE;
    }

    if( freezevmode )
        PutProfileData( PR_VIDEOMODE, &(VideoBase->Mode), sizeof(unsigned) );
    if( !ovl_loaded )
        PutProfileData( PR_INITVMODE, &(VideoBase->OldMode), sizeof(unsigned) );

    ShowMenu();

    if( !GetProfileString( PR_INIPATH, NULL ) )
        PutProfileString( PR_INIPATH, MyPath );

    updatemenu();

    /* the following unfortunately prevents from data discovery after
       system hang up or so, but if we would allow data discovery at
       system startup, things could get out of control by some means.
    */
    if( !ovl_loaded ) {
        for( i = 0; i < MAXFILES; ++i )
            ClearProfileData( PR_VIRTFILE(i) );

        ClearProfileData( PR_CLIPBOARDNAME );
        DeleteTmps();
    }

    cp = GetProfileString( PR_RESPONSEFILE, NULL );
    if( cp )
        Delete( cp );

    return TRUE;
}

/* This function is called once per argument passed from the command line.
   'cp' is a pointer to the argument. */
static void _addarg( char *cp ) {

    if( strcmp( cp, MAGICOVERLAY ) == 0 ) {
        ovl_loaded = TRUE;
        return;
    }

    if( strcmp( cp, "/P" ) == 0 || strcmp( cp, "/p" ) == 0 ) {
        ignore_palette = TRUE;
        return;
    }

    if( strcmp( cp, "/F" ) == 0 || strcmp( cp, "/f" ) == 0 ) {
        freezevmode = TRUE;
        return;
    }

    if( strcmp( cp, "/C" ) == 0 || strcmp( cp, "/c" ) == 0 ) {
        create_ini = TRUE;
        return;
    }

    if( strncmp( cp, "/I", 2 ) == 0 || strncmp( cp, "/i", 2 ) == 0 ) {
        if (foundini) {
            /* There can only be one */
            printf( errmsg(EM_ALREADYINI|EM_INI_C), cp );
            return;
        }

        cp += 2;
        while( (*cp==':') || (*cp=='=') ) ++cp;
        // use argument as ini-file
        if( strchr( cp, '\\' ) || strchr( cp, ':' ) )
            strcpy( IniName, cp );
        else if( strlen( cp ) > _MAX_FNAME + _MAX_EXT + 1 ) {
            /* the file name is too long for a DOS filename
               (and too long for our buffer), display error */
            printf( errmsg(EM_ININAME|EM_INI_C), cp );
            strcat( IniName, INI_NAME );
        } else
            strcat( IniName, cp );
        foundini = TRUE;

        return;
    }

    if( strncmp( cp, "/et", 3 ) == 0 ) {
        cp += 3;
        while( (*cp==':') || (*cp=='=') ) ++cp;
        sscanf( cp, "%p", &SharedMemory );
        return;
    }

    /* All options has been parsed before, this one is illegal (if it is
       an option. */
    if( cp[0] == '/' ) {
        printf( errmsg(EM_BAGARGUMENT|EM_INI_C), cp );
        return;
    }

    /* Maybe it is a filename. */

    AddToFileArgsList(cp);
}

static char *files[MAXFILES];
static int numfiles = 0;

void AddToFileArgsList(char *name) {

    if (numfiles < MAXFILES) {
        files[numfiles++] = strdup(name);
    }
}

char *GetFromFileArgsList() {

    if (numfiles) {
        return files[--numfiles];
    }
    return NULL;
}



void die( int code ) {

    int i;

    DeleteTmps();

    for( i = 0; i < MAXFILES; ++i )
        ClearProfileData( PR_VIRTFILE(i) );

    ClearProfileData( PR_CLIPBOARDNAME );

    if( ini_loaded && (code == 0) )
        put_init();

    if( code == 0 ) {
        InitVideo( *(int *)GetProfileData( PR_INITVMODE, &(VideoBase->OldMode ) ) );
        ClearProfileData( PR_INITVMODE );

        HideMouse();
        if( VideoBase->ScreenHeight != 25 ) {
            ClearScreen();
            SetCursorPos( 0, 0 );
        } else {
            SetActualPage( DOSPAGE );
            SetVisualPage( DOSPAGE );
            SetPointerPage( DOSPAGE );
        }

    }

    UnSetHandler();
    BreakFlag( old_breakflag );
    m_cleanup();

    ResetMouse();

    /* They'll never come back... */
    SharedMemory->ExitCode = 0;

    DeleteTeXTmps();
    ExitProc( code );
}


static char *CODEBASED _emsg[] = {
    "Edit buffer not allocatable",
    "Video mode not supported",
    "Out of memory",
    "Not enough core",
    "Read error",
    "Illegal name for initialization file: '%s', using default",
    "Already has initialization file.",
    "Profile not found: '%s'",
    "Profile corrupt: '%s': bad or missing file introducer",
    "Unable to load profile: '%s'",
    "Ignoring illegal option or parameter: '%s'",
    "Too many arguments: '%s'",
    "Communication corrupt: et.com <-> et.ovl",
};

char *errmsg( unsigned int num ) {

    static char buffer[100];

    if( num < 0x1000 ) {
        strcpy( buffer, _emsg[num] );
        return buffer;
    }

    sprintf( buffer, "EasyTeX Error %#x: %s\n", num, _emsg[num & 0x0fff] );

    return buffer;
}

/*
 *    Name: mem
 *  Return: void
 * Purpose: displays memory information in a window
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 45
#define WIN_HEIGHT 14
// #pragma optimize( "elg", off )
void mem( void )
{
    static char bar[] = "";

    static struct IntuiText CODEBASED it_totalfree = { NULL, NULL, NULL, 10, 2 };
    static struct IntuiText CODEBASED it_used = { &it_totalfree, NULL, NULL, 10, 3 };
    static struct IntuiText CODEBASED it_separator1 = { &it_used, NULL, bar, 10, 4 };
    static struct IntuiText CODEBASED it_total = { &it_separator1, NULL, NULL, 5, 5 };
    static struct IntuiText CODEBASED it_relfree = { &it_total, NULL, NULL, 2, 7 };
    static struct IntuiText CODEBASED it_cboard = { &it_relfree, NULL, NULL, 2, 9 };

    static struct IntuiText CODEBASED it_largest = { NULL, NULL, NULL, 7, 2 };
    static struct IntuiText CODEBASED it_lost = { &it_largest, NULL, NULL, 9, 3 };
    static struct IntuiText CODEBASED it_relunfrag = { &it_lost, NULL, NULL, 2, 5 };
    static struct IntuiText CODEBASED it_separator2 = { &it_relunfrag, NULL, bar, 10, 7 };
    static struct IntuiText CODEBASED it_heapavl = { &it_separator2, NULL, NULL, 7, 8 };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };

    static struct IntuiText CODEBASED it_more = { NULL, NULL, "&More", 1, 0 };

    static struct Gadget CODEBASED g_more = {
            &g_help, &it_more,
            WIN_WIDTH - 20,
            WIN_HEIGHT - 4,
            8, 3, BOOLGADGET | ENDGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
            &g_more, NULL,
            2,
            WIN_HEIGHT -4,
            6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct Window CODEBASED mwin = {
            -1, -1,
            WIN_WIDTH, WIN_HEIGHT,    /* width, height of window */
            0, 0,
            "Memory",
            0L,                        /* flags */
            NULL,
    };

    unsigned long freemem, largestblock;
    unsigned long used, lost;
    unsigned char _huge *buf;
    register long rl;
    struct Gadget *g;
    struct IntuiText *first[2];
    int toggle = 0;
    char buffer[100];

    extern struct Clipboard cboard;

    it_totalfree.Text = NULL;
    it_used.Text = NULL;
    it_total.Text = NULL;
    it_relfree.Text = NULL;
    it_cboard.Text = NULL;
    it_largest.Text = NULL;
    it_lost.Text = NULL;
    it_relunfrag.Text = NULL;
    it_heapavl.Text = NULL;

    sprintf( buffer, "Available Heap Memory: %lu", MemFree() );
    if( ! ( it_heapavl.Text = HeapStrDup( buffer ) ) ) goto done;

    if( cboard.flags & CLIP_EMPTY ) {
        if( ! ( it_cboard.Text = HeapStrDup( "The clipboard is empty." ) ) )
            goto done;
    } else if( cboard.flags & CLIP_STREAM ) {
        sprintf( buffer,
            "%u characters in stream clipboard", cboard.size );
        if( ! ( it_cboard.Text = HeapStrDup( buffer ) ) ) goto done;
    } else {
        sprintf( buffer,
            "%u lines in multiline clipboard", cboard.lines );
        if( ! ( it_cboard.Text = HeapStrDup( buffer ) ) ) goto done;
    }

    buf = mh.base;

    lost = used = 0L;
    for( freemem = 0L; buf < mh.endlines; buf += *(buf+1)+2 ) {
        if( *buf == SMALLFRAME|FREEFRAME )
            freemem += (unsigned long)*(buf+1) + 2L;
        else if( *buf == SMALLFRAME|ALLOCFRAME )
            used += (unsigned long)*(buf+1) + 2L;
        else
            lost += (unsigned long)*(buf+1) + 2L;
    }

    largestblock = (unsigned long)((unsigned char _huge *)mh.desc + 4 -
        (4 * mh.numlines) - mh.endlines );
    used += 4 * mh.numlines;

    sprintf( buffer,"Total free: %6ld Bytes", freemem+largestblock );
    if( ! ( it_totalfree.Text = HeapStrDup( buffer ) ) ) goto done;

    sprintf( buffer,    "Total used: %6ld Bytes", used );
    if( ! ( it_used.Text = HeapStrDup( buffer ) ) ) goto done;

    sprintf( buffer,    "Total allocated: %6ld Bytes", mh.size );
    if( ! ( it_total.Text = HeapStrDup( buffer ) ) ) goto done;

    rl = (long)( ((freemem+largestblock) * 1000L) / mh.size ),

    sprintf( buffer, "Percentage of free memory: %ld.%ld%%", rl/10L, rl%10L );
    if( ! ( it_relfree.Text = HeapStrDup( buffer ) ) ) goto done;

    sprintf( buffer, "Largest frame: %6ld Bytes", largestblock );
    if( ! ( it_largest.Text = HeapStrDup( buffer ) ) ) goto done;
    sprintf( buffer, "Lost frames: %6ld Bytes", lost );
    if( ! ( it_lost.Text = HeapStrDup( buffer ) ) ) goto done;

    if(largestblock+freemem+lost)
        rl = (long)( (largestblock * 1000L) / (largestblock+freemem+lost) );
    else
        rl = 0L;

    sprintf( buffer, "Percentage of unfragmented memory: %ld.%ld%%", rl/10L, rl%10L );
    if( ! ( it_relunfrag.Text = HeapStrDup( buffer ) ) ) goto done;

    if( OpenWindow( &mwin, HLP_MEMORY ) ) {

        first[0] = &it_cboard;
        first[1] = &it_heapavl;
        do {
            PrintIText( first[toggle], mwin.Left, mwin.Top );
            do {
                g = OpenGadget( &g_ok, mwin.Left, mwin.Top, 0L );
                if( g == &g_help ) help( -1 );
            } while( g == &g_help );

            if( g == &g_more ) {
                toggle ^= 1;

                VideoFill( CR_WINDOW, mwin.Left+1, mwin.Top+1,
                     mwin.Left+WIN_WIDTH-2, mwin.Top+WIN_HEIGHT-5 );

            }
        } while( g == &g_more );

        CloseWindow( &mwin );
    }

done:
    if( it_totalfree.Text ) free( it_totalfree.Text );
    if( it_used.Text )      free( it_used.Text );
    if( it_total.Text )     free( it_total.Text );
    if( it_relfree.Text )   free( it_relfree.Text );
    if( it_cboard.Text )    free( it_cboard.Text );
    if( it_largest.Text )   free( it_largest.Text );
    if( it_lost.Text )      free( it_lost.Text );
    if( it_relunfrag.Text ) free( it_relunfrag.Text );
    if( it_heapavl.Text )   free( it_heapavl.Text );

}
#undef WIN_WIDTH
#undef WIN_HEIGHT
// #pragma optimize( "", on )

/*
 *    Name: about
 *  Return: void
 * Purpose: displays logostring in a window
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 40
#define WIN_HEIGHT 10
void about( void )
{
        static int disp_time( void );

        static struct IntuiText CODEBASED it_in1 = { NULL, NULL, NULL, 0, 1 };
        static struct IntuiText CODEBASED it_in2 = { &it_in1, NULL, NULL, 0, 2 };
        static struct IntuiText CODEBASED it_in3 = { &it_in2, NULL, NULL, 0, 3 };
        static struct IntuiText CODEBASED it_fine = { NULL, NULL, "Fine!", 0, 0 };

        static struct Gadget CODEBASED g_ok = {
                NULL, NULL,
                2,
                WIN_HEIGHT - 4,
                6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
        };
        static struct Gadget CODEBASED g_fine = {
                &g_ok, &it_fine,
                WIN_WIDTH - 9,
                WIN_HEIGHT - 4,
                7, 3, BOOLGADGET | ENDGADGET,
        };

        static struct Window CODEBASED iwin = {
                -1, -1,
                WIN_WIDTH, WIN_HEIGHT,    /* width, height of window */
                0, 0,
                NULL,
                0L,                        /* flags */
                NULL,
        };

        static char *CODEBASED banner0 = "EasyTeX Version " VERSIONSTRING;
        static char *CODEBASED banner1 = "(c)Copyright ITLR 1990-1994";
        static char *CODEBASED banner2 = "written by M.Schollmeyer";

        it_in1.Text = banner0;
        CenterText( &it_in1, WIN_WIDTH );
        it_in2.Text = banner1;
        CenterText( &it_in2, WIN_WIDTH );
        it_in3.Text = banner2;
        CenterText( &it_in3, WIN_WIDTH );

        SetHook( disp_time, HOOK_WAIT );
        OpenWindow( &iwin, HLP_ABOUT );
        PrintIText( &it_in3, iwin.Left, iwin.Top );

        OpenGadget( &g_fine, iwin.Left, iwin.Top, 0L );
        CloseWindow( &iwin );
        SetHook( NULL, HOOK_WAIT );

}

static int disp_time( void )
{
    static unsigned int secs = 100;
    struct RealTimeClock rtc;

    char buffer[50];
    static struct IntuiText CODEBASED itex0 = { NULL, NULL, NULL, 2, 1 };

    static char *CODEBASED days[] = {
        "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat",
    };

    GetRTC( &rtc );
    if( secs != rtc.Seconds )
    {
        itex0.Text = buffer;
        sprintf( buffer, "%s, %2d.%02d.%04d %d:%02d.%02d ",
                    days[rtc.DayOfWeek],rtc.Day, rtc.Month,rtc.Year
                    ,rtc.Hours, rtc.Minutes, rtc.Seconds );

        PrintIText( &itex0, ( VideoBase->ScreenWidth - WIN_WIDTH ) / 2 + 6
                      , ( VideoBase->ScreenHeight - WIN_HEIGHT ) / 2 + 4 );
        secs = rtc.Seconds;
    }

    return( 0 );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

#define WIN_WIDTH 52
#define WIN_HEIGHT 21
int settingdlg( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0,
        "Editor Settings", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_color = {
        NULL, NULL, "Color &Setup...", 1, 0
    };
    static struct Gadget CODEBASED g_color = {
        &g_ok, &it_color, 2, WIN_HEIGHT-8,
        22, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText CODEBASED it_ini = {
        NULL, NULL, "&Change ini-file...", 1, 0
    };
    static struct Gadget CODEBASED g_ini = {
        &g_color, &it_ini, 2, WIN_HEIGHT-11,
        22, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText CODEBASED it_min = {
        NULL, NULL, "min", 5, 0
    };
    static struct IntuiText CODEBASED it_peri = {
        &it_min, NULL, "&Periodic Backup:", -16, 0
    };
    static struct StringInfo CODEBASED si_peri = {
        NULL, 4, 0L
    };
    static struct Gadget CODEBASED g_peri = {
        &g_ini, &it_peri, 42, 13, 5, 1,
        TEXTGADGET | LONGINT | BORDERLESS,
        (APTR)&si_peri,
    };

    static struct IntuiText CODEBASED it_err = {
        NULL, NULL, "&Error Level:", -12, 0
    };
    static struct StringInfo CODEBASED si_err = {
        NULL, 5, 0L
    };
    static struct Gadget CODEBASED g_err = {
        &g_peri, &it_err, 42, 11, 5, 1,
        TEXTGADGET | LONGINT | BORDERLESS,
        (APTR)&si_err,
    };

    static struct IntuiText CODEBASED it_dispmem = {
        NULL, NULL, "Display Mem&ory", 4, 0
    };
    static struct Gadget CODEBASED g_dispmem = {
        &g_err, &it_dispmem, 32, 9, 16, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_tracklines = {
        NULL, NULL, "Track &Lines", 4, 0
    };
    static struct Gadget CODEBASED g_tracklines = {
        &g_dispmem, &it_tracklines, 32, 8, 16, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_trackfiles = {
        NULL, NULL, "T&rack Files", 4, 0
    };
    static struct Gadget CODEBASED g_trackfiles = {
        &g_tracklines, &it_trackfiles, 32, 7, 16, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_expand = {
        NULL, NULL, "E&xpand Tabs", 4, 0
    };
    static struct Gadget CODEBASED g_expand = {
        &g_trackfiles, &it_expand, 32, 6, 16, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_backup = {
        NULL, NULL, "&Backup", 4, 0
    };
    static struct Gadget CODEBASED g_backup = {
        &g_expand, &it_backup, 32, 5, 10, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_overtype = {
        NULL, NULL, "Insert &Mode", 4, 0
    };
    static struct Gadget CODEBASED g_overtype = {
        &g_backup, &it_overtype, 32, 4, 15, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_autosave = {
        NULL, NULL, "&Auto Save", 4, 0
    };
    static struct Gadget CODEBASED g_autosave = {
        &g_overtype, &it_autosave, 32, 3, 13, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_indent = {
        NULL, NULL, "&Indent", 4, 0
    };
    static struct Gadget CODEBASED g_indent = {
        &g_autosave, &it_indent, 32, 2, 10, 1,
        CHECKGADGET | TOGGLESELECT
    };

    static struct IntuiText CODEBASED it_tmps = {
        NULL, NULL, "Temporary &Files:", 0, -1
    };
    static struct StringInfo CODEBASED si_tmps = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_tmps = {
        &g_indent, &it_tmps, 2, 9, 22, 1,
        TEXTGADGET | BORDERLESS,
        (APTR)&si_tmps,
    };

    static struct IntuiText CODEBASED it_inipath = {
        NULL, NULL, "Initial &Directory:", 0, -1
    };
    static struct StringInfo CODEBASED si_inipath = {
        NULL, BUFSIZE, 0L
    };
    static struct Gadget CODEBASED g_inipath = {
        &g_tmps, &it_inipath, 2, 7, 22, 1,
        TEXTGADGET | BORDERLESS,
        (APTR)&si_inipath,
    };

    static struct IntuiText CODEBASED it_tab = {
        NULL, NULL, "&Tabstops:", 0, -1
    };
    static struct StringInfo CODEBASED si_tab = {
        NULL, 17, 0L
    };
    static struct Gadget CODEBASED g_tab = {
        &g_inipath, &it_tab, 2, 5, 17, 1,
        TEXTGADGET | LONGINT | BORDERLESS,
        (APTR)&si_tab,
    };

    static struct IntuiText CODEBASED it_wrap = {
        NULL, NULL, "&Word Wrap Column:", 0, -1
    };
    static struct StringInfo CODEBASED si_wrap = {
        NULL, 17, 0L
    };
    static struct Gadget CODEBASED g_wrap = {
        &g_tab, &it_wrap, 2, 3, 17, 1,
        TEXTGADGET | LONGINT | BORDERLESS | ACTIVATED,
        (APTR)&si_wrap,
    };

    struct Gadget *g;
    char          *inipath = NULL, *textmps = NULL,
                  *cp;
    long           perisav;
    int            ret = 0;

    inipath = AllocHeap(BUFSIZE);
    if( inipath == NULL )
        return 0;
    textmps = AllocHeap(BUFSIZE);
    if( textmps == NULL )
        goto done;

    si_tmps.Buffer = textmps;
    strcpy( textmps, GetProfileString( PR_TMPTEXFILES, &profnull ) );
    si_inipath.Buffer = inipath;
    strcpy( inipath, GetProfileString( PR_INIPATH, &profnull ) );
    si_err.LongInt = (long)GetProfileInt( PR_ERRORLEVEL );
    si_peri.LongInt = TICKS_TO_MINS(GetProfileLong( PR_PERISAV ));

    if( ed.Flags & INDENT )
        SETBIT( g_indent.Flags, SELECTED );
    else
        CLEARBIT( g_indent.Flags, SELECTED );

    if( ed.Flags & AUTOSAVE )
        SETBIT( g_autosave.Flags, SELECTED );
    else
        CLEARBIT( g_autosave.Flags, SELECTED );

    if( ed.Flags & OVERTYPE )
        CLEARBIT( g_overtype.Flags, SELECTED );
    else
        SETBIT( g_overtype.Flags, SELECTED );

    if( ed.Flags & BACKUP )
        SETBIT( g_backup.Flags, SELECTED );
    else
        CLEARBIT( g_backup.Flags, SELECTED );

    if( ed.Flags & EXPANDTABS )
        SETBIT( g_expand.Flags, SELECTED );
    else
        CLEARBIT( g_expand.Flags, SELECTED );

    if( ed.Flags & TRACKFILES )
        SETBIT( g_trackfiles.Flags, SELECTED );
    else
        CLEARBIT( g_trackfiles.Flags, SELECTED );

    if( ed.Flags & TRACKLINES )
        SETBIT( g_tracklines.Flags, SELECTED );
    else
        CLEARBIT( g_tracklines.Flags, SELECTED );

    if( ed.Flags & DISPLAY_MEM )
        SETBIT( g_dispmem.Flags, SELECTED );
    else
        CLEARBIT( g_dispmem.Flags, SELECTED );

    si_wrap.LongInt = ed.WordWrap;
    si_tab.LongInt = ed.TabStops;

    OpenWindow( &win, HLP_SETTINGS );
    do {
        g = OpenGadget( &g_wrap, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g && g != &g_cancel ) {
        packstr( inipath );
        if( *inipath )
            PutProfileString( PR_INIPATH, inipath );
        else
            ClearProfileData( PR_INIPATH );

        packstr( textmps );
        if( *textmps )
            PutProfileString( PR_TMPTEXFILES, textmps );
        else
            ClearProfileData( PR_TMPTEXFILES );

        PutProfileInt( PR_ERRORLEVEL, (int)(si_err.LongInt) );

        perisav = MINS_TO_TICKS(si_peri.LongInt);
        if( perisav ) {
            PutProfileLong( PR_PERISAV, perisav );
            periodic_backup = TRUE;
        } else {
            ClearProfileData( PR_PERISAV );
            periodic_backup = FALSE;
        }

        if( g_indent.Flags & SELECTED )
            SETBIT( ed.Flags, INDENT );
        else
            CLEARBIT( ed.Flags, INDENT );

        if( g_autosave.Flags & SELECTED )
            SETBIT( ed.Flags, AUTOSAVE );
        else
            CLEARBIT( ed.Flags, AUTOSAVE );

        if( g_overtype.Flags & SELECTED )
            CLEARBIT( ed.Flags, OVERTYPE );
        else
            SETBIT( ed.Flags, OVERTYPE );

        if( g_backup.Flags & SELECTED )
            SETBIT( ed.Flags, BACKUP );
        else
            CLEARBIT( ed.Flags, BACKUP );

        if( g_expand.Flags & SELECTED )
            SETBIT( ed.Flags, EXPANDTABS );
        else
            CLEARBIT( ed.Flags, EXPANDTABS );

        if( g_trackfiles.Flags & SELECTED )
            SETBIT( ed.Flags, TRACKFILES );
        else
            CLEARBIT( ed.Flags, TRACKFILES );

        if( g_tracklines.Flags & SELECTED )
            SETBIT( ed.Flags, TRACKLINES );
        else
            CLEARBIT( ed.Flags, TRACKLINES );

        if( g_dispmem.Flags & SELECTED )
            SETBIT( ed.Flags, DISPLAY_MEM );
        else
            CLEARBIT( ed.Flags, DISPLAY_MEM );

        ed.WordWrap = (int)si_wrap.LongInt;
        ed.TabStops = si_tab.LongInt;

        if( g == &g_ini )
            ret = 1;
        else if( g == &g_color )
            color_setup();
    }

done:
    if( inipath )  free( inipath );
    if( textmps )  free( textmps );

    return ret;
}
#undef WIN_HEIGHT
#undef WIN_WIDTH

#define WIN_HEIGHT 18
#define WIN_WIDTH  60
static int printiblock( char *, int, int, int );

void disp_info( int slot ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0,
        "File/Buffer Info", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_help, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_6 = {
        NULL, NULL, "Last Update:", 4, 11
    };
    static struct IntuiText CODEBASED it_5 = {
        &it_6, NULL, "Size:", 11, 10
    };
    static struct IntuiText CODEBASED it_4 = {
        &it_5, NULL,               "[?] Hidden        [?] Archive", 19, 8 };
    static struct IntuiText CODEBASED it_3 = {
        &it_4, NULL, "Attributes:   [?] Read Only     [?] System", 5, 7 };
    static struct IntuiText CODEBASED it_2 = {
        &it_3, NULL, "Virtual File:", 3, 5 };
    static struct IntuiText CODEBASED it_1 = {
        &it_2, NULL, "File Name:", 6, 2 };

    static struct IntuiText CODEBASED it_back = {
        NULL, NULL, "", 17, 2 };

    static char *CODEBASED question = "???";

    unsigned int attributes;
    char *vname, *rname;
    int i;
    char buffer[SCREEN_WIDTH];
    int fptr;

    vname = GetProfileString( PR_VIRTFILE( slot ), NULL );
    rname = GetProfileString( PR_FILENAME( slot ), NULL );

    if( rname ) {

        attributes = GetFileAttr( rname );
        if( attributes & ATTR_READONY ) it_3.Text[15] = '';
        else                            it_3.Text[15] = ' ';
        if( attributes & ATTR_SYSTEM )  it_3.Text[33] = '';
        else                            it_3.Text[33] = ' ';
        if( attributes & ATTR_HIDDEN )  it_4.Text[ 1] = '';
        else                            it_4.Text[ 1] = ' ';
        if( attributes & ATTR_ARCHIVE ) it_4.Text[19] = '';
        else                            it_4.Text[19] = ' ';

    } else {

        it_3.Text[15] = '?';
        it_3.Text[33] = '?';
        it_4.Text[ 1] = '?';
        it_4.Text[19] = '?';

    }

    OpenWindow( &win, HLP_FILEINFO );

    PrintIText( &it_1, win.Left, win.Top );
    for( i = 0; i < 4; ++i )
        PrintIText( &it_back, win.Left, win.Top + i );

    if( rname )
        printiblock( rname, 17 + win.Left, 2 + win.Top, 40 );
    else
        printiblock( question, 17 + win.Left, 2 + win.Top, 40 );

    if( vname )
        printiblock( vname, 17 + win.Left, 5 + win.Top, 40 );
    else
        printiblock( "<no virtual file>", 17 + win.Left, 5 + win.Top, 40 );

    fptr = Open( rname, ACCESS_READ );
    if( fptr == -1 ) {
        strcpy( buffer, question );
        printiblock( buffer, 17 + win.Left, 10 + win.Top, 30 );
        printiblock( buffer, 17 + win.Left, 11 + win.Top, 30 );
    } else {
        sprintf( buffer, "%ld Bytes", Seek( fptr, 0L, SEEK_END ) );
        printiblock( buffer, 17 + win.Left, 10 + win.Top, 30 );
        dta_datetoa( GetFileDate(fptr), buffer );
        printiblock( buffer, 17 + win.Left, 11 + win.Top, 30 );

        Close( fptr );
    }

    while( OpenGadget( &g_ok, win.Left, win.Top, 0L ) == &g_help )
        help( -1 );

    CloseWindow( &win );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT


static int printiblock( char *text, int left, int top, int width ) {

    struct IntuiText it;

    it.Next = NULL;
    it.TextBorder = NULL;
    it.Text = text;
    it.Left = left;
    it.Top = top;
    it.Width = width;
    it.Flags = IT_LEFT|IT_MULTILINE;

    return PrintIText( &it, 0, 0 );
}



/* end of file init.c */
