/*
 * SOURCE:  filereq.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: file requester
 *
 * UPDATES: 03/29/1992 - excluded from main.c
 *
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

        // static function prototypes
static int _getdir( char * );
static struct Gadget *reqgadhandler( struct Gadget *, struct Gadget *,
    unsigned, unsigned, unsigned long );
static void fileopts( void );
static BOOL makedirentry( struct DiskTransferArea _far *, struct Directory * );
static int dircomp( const void *, const void * );

#define MAX_ENTRIES 300
#define ENTRY_WIDTH 42

    // sort order
#define SO_NAME         0
#define SO_EXTENSION    1
#define SO_SIZE         2
#define SO_DATE         3
#define SOF_REVERSE     8
#define SOF_FILESFIRST 16

struct Directory {
    char Entry[13];
    char Info[ENTRY_WIDTH];
    unsigned long Size;
    int Directory;      /* 0: file  1: dir  2:drive */
    unsigned long Date;
};

#define WIN_HEIGHT 20
#define WIN_WIDTH  48

#define MAKEDIRENTRY  1
#define MAKEFILEENTRY 0

#define FILENAMELEN 12

static struct Gadget *filegad;
static struct Directory *dir;
static char filemask[FILENAMELEN+1];
unsigned int flags;

static char *lpszall = "*.*";

/*
 *    Name: DOS_Request
 *  Return: void
 * Purpose: Displays a file requester for path and filename settings
 *
 *
 *
 * (c)M.Schollmeyer
 */
// #pragma optimize( "egl", off )
void DOS_Request( char *title, char *buffer, char *filename, int hlptopic )
{
    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, NULL, 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct IntuiText CODEBASED it_opt = {
        NULL, NULL, "&Options...", 1, 0
    };
    static struct Gadget CODEBASED g_opt = {
        &g_cancel, &it_opt, 10, WIN_HEIGHT-4,
        14, 3, BOOLGADGET | ENDGADGET,
    };

    static struct Gadget CODEBASED g_ok = {
        &g_opt, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText *it, **ita;
    static struct Request CODEBASED req = {
        NULL, NULL, 0L };

    static struct Gadget CODEBASED gad_req = {
        &g_ok, NULL, 1, 5, 45, 10, REQGADGET,
        (APTR)&req,
    };

    static struct IntuiText CODEBASED it_file = {
        NULL, NULL, "&File:", -5, 0
    };

    static struct StringInfo CODEBASED si_file = {
        NULL, _MAX_PATH, 0L
    };
    static struct Gadget CODEBASED g_file = {
        &gad_req, &it_file, 7, 2, 39, 1,
        TEXTGADGET | BORDERLESS,
        (APTR)&si_file,
    };

    static struct IntuiText CODEBASED it_dir = {
        NULL, NULL, NULL, 2, 4
    };

    int                           i,
                                  j;
    struct Gadget                *g;
    char                          buf[_MAX_PATH],
                                  path[_MAX_PATH],
                                  suffix[10];
    char                         *diskpath;
    struct DiskTransferArea _far *dta;

    dir = NULL;
    it = NULL;
    ita = NULL;

    buffer[0] = '\0';

    if( ! (si_file.Buffer = AllocHeap( _MAX_PATH+1) ))
        goto done;

    filegad = &g_file;

    dir = AllocHeap( sizeof(struct Directory) * MAX_ENTRIES );
    if( dir == NULL )
        goto done;

    it = AllocHeap( sizeof(struct IntuiText) * MAX_ENTRIES );
    if( it == NULL )
        goto done;

    ita = AllocHeap( sizeof(struct IntuiText *) * MAX_ENTRIES );
    if( ita == NULL )
        goto done;

    req.Items = ita;

    SETBIT( g_file.Flags, ACTIVATED );
    CLEARBIT( gad_req.Flags, ACTIVATED );

    splitpath( filename, path, si_file.Buffer, suffix );

    if( (suffix[0] == '\0') && (filename[0] == '\0') ) {
        strncpy( filemask, GetProfileString( PR_FILEMASK, lpszall ),
            FILENAMELEN+1 );
        filemask[FILENAMELEN] = '\0';
        strcpy( si_file.Buffer, filemask );
    } else {
        strcat( si_file.Buffer, "." );
        strcat( si_file.Buffer, suffix );
        sprintf( filemask, "*.%s", suffix );
    }


    win.Title = title;
    if( !OpenWindow( &win, hlptopic == -1 ? HLP_DOSREQUEST : hlptopic ) )
        goto done;

    dta = GetDTAAddress();
    if( path[0] ) {
        SetCurrDir( path );
    } else
        GetCurrDir( path );

    for ever {

        req.CurrIt = req.WinTop = 0;

        i = _getdir(path);

        // terminate list by NULL
        ita[i] = NULL;

        /* print the current directory */
        MakeShortName( buf, path, win.Width-4 );
        it_dir.Text = buf;
        VideoFill( CR_WINDOW, win.Left+1, win.Top+4,
            win.Left+win.Width-2, win.Top+4 );
        PrintIText( &it_dir, win.Left, win.Top );

        for( --i; i >= 0; --i )
        {
            ita[i] = &it[i];
            it[i].Flags = 0;
            it[i].Next = NULL;
            it[i].TextBorder = NULL;
            it[i].Text = dir[i].Info;
            it[i].Left = 0;
            it[i].Top = 0;
            if( dir[i].Directory )
                SETBIT( it[i].Flags, IT_ITALIC );
            else {
                // file
                if (IsFileInFileList(dir[i].Entry, path, FALSE))
                    SETBIT( it[i].Flags, IT_HILITE );
            }
        }

        do {
            g = OpenGadget( &g_file, win.Left, win.Top, reqgadhandler );
            if( g == &g_help ) {
                help( -1 );
            }
        } while( g == &g_help );

        if( g == &g_opt ) {

            fileopts();

        } else if( g == &gad_req || g == &g_ok || g == &g_file ) {

            /* set activation flag on the appropriate gadget */
            if( g == &gad_req ) {
                CLEARBIT( g_file.Flags, ACTIVATED );
                SETBIT( gad_req.Flags, ACTIVATED );
            } else if( g == &g_file ) {
                CLEARBIT( gad_req.Flags, ACTIVATED );
                SETBIT( g_file.Flags, ACTIVATED );
            }

            if( si_file.Buffer[0] == 0 ) {
                strcpy( si_file.Buffer, lpszall );
            }

            splitpath( si_file.Buffer, path, buf, NULL );

            if( path[0] == '\0' || SetCurrDir( path ) ) {
                /* Now the path which was given (if any) was correct
                   or there was no path at all. */

                /* Look if 'filemask' contains a wildchar */
                if( wildchar(buf) ) {
                    GetCurrDir( path );
                    strcpy( si_file.Buffer, buf );
                    strcpy( filemask, buf );
                    PutProfileString( PR_FILEMASK, filemask );
                    continue;
                }

                if( path[1] == ':' && path[2] == '\0' )
                    continue;

                if( buf[0] == '\0' ) {
                    strcpy( filemask, lpszall );
                    strcpy( si_file.Buffer, lpszall );
                    continue;
                }


                if( buf[0] == '.' ||
                    ( FindFirstFile( buf, DTA_DIRECTORY )
                    && dta->Attributes & DTA_DIRECTORY ) ) {
                    if( SetCurrDir( buf ) )
                        GetCurrDir( path );
                    else
                        path[0] = '\0';

                    strcpy( si_file.Buffer, filemask );

                    continue;
                }

                GetCurrDir( path );
                strcpy( buffer, path );
                if( buffer[ strlen( buffer ) - 1 ] != '\\' && buffer[0] )
                    strcat( buffer, "\\" );

                /* buf contains only the base name: strcat default extension
                   to avoid strcat'ing the default extension, the user
                   can cat a '.' wich will be removed below */
                if( strchr( buf, '.' ) == NULL && !_Access(buf, MODE_COMP) )
                    strcat( buf, DEF_FILE_EXT );

                /* remove cat'ed '.' (if any) */
                i = strlen(buf)-1; /* This is save, because 'buf' is not
                                      empty in any case */
                if( buf[i] == '.' )
                    buf[i] = '\0';

                strcat( buffer, buf );
                strlow( buffer );

                break;

            } else
                DoErrorBox( HLP_DEFAULT, "Invalid Directory:'%s'", path );

        } else if( g == &g_cancel || g == NULL ) {

            buffer[0] = '\0';
            break;
        }

    } /* for ever */

    CloseWindow( &win );
done:
    if( dir ) free( dir );
    if( it ) free( it );
    if( ita ) free( ita );
    if( si_file.Buffer ) free( si_file.Buffer );
}
#undef WIN_HEIGHT
#undef WIN_WIDTH
// #pragma optimize( "", on )


/* Returns 0 if the file is not in the file list, offset+1 if it is. If
   setup is set to TRUE, the full filespec is copied to file. */
int IsFileInFileList(char *file, char *path, BOOL setup) {

    char buffer[_MAX_PATH];
    char base[_MAX_PATH];
    char suffix[4];

    int j;

    /* Look if the filename has a path. */
    if ( strchr( file, '\\' ) || strchr( file, ':' ) ) {
        /* It has, get the correct expession */
        splitpath(file, buffer, base, suffix);
        /* Convert path. */
        SetCurrDir(buffer);
        GetCurrDir(buffer);
        sprintf (buffer+strlen(buffer), "%c%s%c%s",
            '\\', base, '.', suffix );
    } else {
        /* Filename has no path */
        /* Look if the path exists. */
        if (!path) {
            GetCurrDir(buffer);
        } else {
            strcpy( buffer, path );
        }
        if( buffer[strlen(buffer)-1] != '\\' )
            strcat( buffer, "\\" );
        strcat( buffer, file );
    }

    /* Now buffer contains the filespec. */
    if (setup) {
        strcpy( file, buffer);
        strlow( file );
    }

    for( j = 0; j < MAXFILES; ++j ) {
        if( stricmp( buffer, GetProfileString( PR_FILENAME(j),&profnull ) ) == 0 )
            return j+1;
    }
    return 0;
}


static int _getdir( char *path ) {

    int i,
        currdrive,
        savdrive,
        numdrives = 0,
        maxdrives;

    struct DiskTransferArea _far *dta;

    dta = GetDTAAddress();
    maxdrives = GetNumDrives();

    /* setup drives */
    savdrive = GetCurrDisk();
    for( i = 0, numdrives = 0; i < maxdrives; ++i )
    {
        SelectDisk( i );
        currdrive = GetCurrDisk();
        if( i == currdrive )
        {
            strcpy( dir[ numdrives ].Entry, "?:" );
            dir[ numdrives ].Entry[0] = (char)(i + 'A');
            strcpy( dir[ numdrives ].Info, "[?:]           <drive>");
            dir[ numdrives ].Info[1] = (char)(i + 'A');
            dir[ numdrives ].Directory = 2;
            dir[ numdrives ].Size = 0L;
            dir[ numdrives ].Date = 0L;
            ++numdrives;
        }
    }
    SelectDisk( savdrive );
    i = numdrives;

    /* setup directory */
    if( path[0] && GetCurrDir(path) )
    {
        /* We could get directories and files simultainiously, the
           reason for first loading directories and then files is
           that we want to load ALL directories, but files only
           according to the current suffix.
         */
        if( FindFirstFile( lpszall, DTA_DIRECTORY ) )
        {
            do
            {
                if( dta->Attributes & DTA_DIRECTORY )
                {
                    if( makedirentry( dta, &dir[ i ] ) )
                        ++i;
                }
            }while( FindNextFile() );
        }
        if( FindFirstFile( filemask, 0 ) )
        {
            do
            {
                if( makedirentry( dta, &dir[ i ] ) )
                    ++i;
                if( i >= MAX_ENTRIES ) {
                    DoErrorBox( HLP_DEFAULT,
                "Directory entries exeeds %d\nDirectory list truncated, sorry.", MAX_ENTRIES );
                    break;
                }
            }while( FindNextFile() );
        }
        flags = GetProfileInt( PR_DOSREQOPTS );
	qsort( (void *)dir, i, sizeof(struct Directory), dircomp );
    }

    return i;
}


/*
 *    Name: gadhandler
 *  Return: nothing from interest
 * Purpose: this handler is called from INTUITION to permanently
 *          update the file gadget
 *
 *
 * (c)M.Schollmeyer
 */
static struct Gadget *reqgadhandler( struct Gadget *first, struct Gadget *g,
        unsigned left_off, unsigned top_off, unsigned long flags ) {

    struct StringInfo *si;
    struct Request *req;

    if( g->Flags & REQGADGET ) {
        si = (struct StringInfo *)filegad->SpecialInfo;
        req = (struct Request *)g->SpecialInfo;
        if( flags & CG_CHANGE ) {
            strcpy( si->Buffer, dir[ req->CurrIt ].Entry );
            DrawGadget( filegad, left_off, top_off );
        }
    }
}


/*
 *    Name: wildchar
 *  Return: BOOLean variable indicating if the string contains a
 *          wildchar character or not.
 * Purpose: Scans a string for a wildchar character
 *
 *
 *
 * (c)M.Schollmeyer
 */
BOOL wildchar(char *cp) {

    return( strrchr(cp, '*') || strrchr(cp, '?') );
}


/*
 *    Name: splitpath
 *  Return: void
 * Purpose: slits up a path name into various strings
 *
 *
 *
 * (c)M.Schollmeyer
 */
void splitpath( file, path, base, suffix )
char *file, *path, *base, *suffix;
{

    char *cp;
    char donesuffix = 0;    /* flag if the suffix is already copied.
                               This is necessary because directories
                               followed in 'file' could also have a '.'
                             */
    if( file == NULL )
        file = (char *)&profnull;

    if( suffix )
        *suffix = '\0';
    if (base)
        *base = '\0';

    /* first, copy filename to destination. */
    strcpy( path, file );

    /* scan string reverse */
    cp = path + strlen( path );
    while( cp >= path ) {
        if( *cp == '.' && !donesuffix ) {
            donesuffix = 1;
            if( suffix ) {
                strcpy( suffix, cp+1 );
                *cp = '\0';
            }
        }
        if( *cp == '\\' || *cp == ':' ) {
            break;
        }
        --cp;
    }
    ++cp;

    if (base)
        strcpy( base, cp );
    *cp = '\0';
}


static BOOL makedirentry( dta, dir )
struct DiskTransferArea _far *dta;
struct Directory *dir;
{
    char _far *cp, _far *c, _far *ccp, _far *base;
    int i;
    unsigned long size;

    cp = dir->Info;
    c = dta->Name;
    base = cp;

    dir->Date = 0L;
    dir->Size = 0L;

    _fstrcpy( dir->Entry, c );

    if( _fstrcmp( c, "." ) == 0 )
    {
        /* ignore current directory */
        return FALSE;
    }
    else if( _fstrcmp( c, ".." ) == 0 )
    {
        _fstrcpy( dir->Info, "[..]          <parent>" );
        dir->Directory = 1;
        return TRUE;
    }

    /* fill string */
    for( i = 40, ccp = base; i ; *ccp++ = ' ', --i );
    *ccp = '\0';

    while( *c && *c != '.' )
    {
        *cp = tolower( *c );
        ++cp;
        ++c;
    }
    cp = base + 8;
    while( *c )
    {
        *cp++ = tolower( *c );
        ++c;
    }

    if( dta->Attributes & DTA_DIRECTORY )
    {
        dir->Directory = 1;
        strncpy( base + 13, " <drawer>", 9 );
    }
    else
    {
        dir->Directory = 0;
        dir->Size = dta->Size;
        /* remove seconds from date info */
        dir->Date = dta->Date >> 5;
        cp = base + 21;
        size = dir->Size = dta->Size;
        do
        {
            *cp-- = _i2asc( size % 10 );
            size /= 10;
        }while( size );
    }

    dta_datetoa( dta->Date, base + 24 );

    return TRUE;
}


/*
    Options...Ŀ
                               
     Sort by... [ ] Name       
                [ ] Extension  
                [ ] Size       
                [ ] Date       
                               
     [ ] Put Files First       
     [ ] Reverse Order         
                               
     Ŀ         Ŀ 
      OK           Cancel  
               
    
*/

#define WIN_HEIGHT 16
#define WIN_WIDTH  33
static void fileopts( void ) {

    static struct Window CODEBASED win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Options...", 0L
    };

    static struct Gadget CODEBASED g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget CODEBASED g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct Gadget CODEBASED g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText CODEBASED it_sortby = {
        NULL, NULL, "Sort by...", 2, 2
    };

    static struct IntuiText CODEBASED it_order = {
        NULL, NULL, "&Reverse Order", 4, 0
    };
    static struct Gadget CODEBASED g_order = {
        &g_ok, &it_order, 2, 8,
        17, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_ffirst = {
        NULL, NULL, "&Put Files First", 4, 0
    };
    static struct Gadget CODEBASED g_ffirst = {
        &g_order, &it_ffirst, 2, 7,
        19, 1, CHECKGADGET | TOGGLESELECT,
    };

    static struct IntuiText CODEBASED it_date = {
        NULL, NULL, "&Date", 4, 0
    };
    static struct Gadget CODEBASED g_date = {
        &g_ffirst, &it_date, 13, 5,
        8, 1, OPTIONGADGET | TOGGLESELECT | NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_size = {
        NULL, NULL, "&Size", 4, 0
    };
    static struct Gadget CODEBASED g_size = {
        &g_date, &it_size, 13, 4,
        8, 1, OPTIONGADGET | TOGGLESELECT | NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_ext = {
        NULL, NULL, "&Extension", 4, 0
    };
    static struct Gadget CODEBASED g_ext = {
        &g_size, &it_ext, 13, 3,
        13, 1, OPTIONGADGET | TOGGLESELECT | NODESELECT,
        NULL, 1L
    };

    static struct IntuiText CODEBASED it_name = {
        NULL, NULL, "&Name", 4, 0
    };
    static struct Gadget CODEBASED g_name = {
        &g_ext, &it_name, 13, 2,
        8, 1, OPTIONGADGET | TOGGLESELECT | NODESELECT,
        NULL, 1L
    };

    struct Gadget *g;
    unsigned int flags;

    CLEARBIT( g_name.Flags, SELECTED );
    CLEARBIT( g_ext.Flags, SELECTED );
    CLEARBIT( g_size.Flags, SELECTED );
    CLEARBIT( g_date.Flags, SELECTED );

    flags = GetProfileInt( PR_DOSREQOPTS );

    if( flags & SOF_REVERSE )
        SETBIT( g_order.Flags, SELECTED );
    if( flags & SOF_FILESFIRST )
        SETBIT( g_ffirst.Flags, SELECTED );
    switch( flags & 0x07 ) {
        case SO_NAME: SETBIT( g_name.Flags, SELECTED );
                      break;
        case SO_EXTENSION: SETBIT( g_ext.Flags, SELECTED );
                      break;
        case SO_SIZE: SETBIT( g_size.Flags, SELECTED );
                      break;
        case SO_DATE: SETBIT( g_date.Flags, SELECTED );
                      break;
    }

    OpenWindow( &win, HLP_DOSREQOPTS );
    PrintIText( &it_sortby, win.Left, win.Top );
    do {
        g = OpenGadget( &g_name, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );

    if( g && g != &g_cancel ) {
        if( g_name.Flags & SELECTED )
            flags = SO_NAME;
        else if( g_ext.Flags & SELECTED )
            flags = SO_EXTENSION;
        else if( g_size.Flags & SELECTED )
            flags = SO_SIZE;
        else if( g_date.Flags & SELECTED )
            flags = SO_DATE;
        if( g_ffirst.Flags & SELECTED )
            SETBIT( flags, SOF_FILESFIRST );
        if( g_order.Flags & SELECTED )
            SETBIT( flags, SOF_REVERSE );

        PutProfileInt( PR_DOSREQOPTS, flags );
    }

    CloseWindow( &win );

}
#undef WIN_HEIGHT
#undef WIN_WIDTH



/* this routine is passed to qsort() */
static int dircomp( const void *arg1, const void *arg2 ) {

    const struct Directory *d1 = (const struct Directory *)arg1;
    const struct Directory *d2 = (const struct Directory *)arg2;

    unsigned int ret = 0;

    if( (d1->Directory != d2->Directory) || d1->Directory ) {
        if( flags & SOF_FILESFIRST && ( !d1->Directory || !d2->Directory ) )
            ret = d1->Directory - d2->Directory;
        else
            ret = d2->Directory - d1->Directory;

        if( ret ) return ret;
        return strncmp( d1->Info, d2->Info, 12 );
    }

    if( (flags & 0x07) == SO_SIZE ) {
        /* since dx->Size is an unsigned long, we cannot just return
           the difference like above, and casting to long and then
           casting the difference to int don't work because the difference
           can easily be >32768.
         */
        if( d1->Size > d2->Size )
            ret = 1;
        else if( d1->Size < d2->Size )
            ret = -1;
        if( ret ) goto done;
    }

    else if( (flags & 0x07) == SO_DATE ) {
        if( d1->Date > d2->Date )
            ret = 1;
        else if( d1->Date < d2->Date )
            ret = -1;
        if( ret ) goto done;
    }

    else if( (flags & 0x07) == SO_EXTENSION ) {
        ret = strncmp( d1->Info + 9, d2->Info + 9, 3 );
        if( ret ) goto done;
    }

    ret = strncmp( d1->Info, d2->Info, 12 );

done:
    return ( ret * ( flags & SOF_REVERSE ? -1 : 1 ) );

}

/* end of file filereq.c */
