#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;
use lib 'lib';

use Hypersonic::Future;
use Hypersonic::Future::Pool;

# Compile Future (includes Pool)
Hypersonic::Future->compile();

Hypersonic::Future::Pool->init;

# Test simple submit and process
{
    my $f = Hypersonic::Future->new;
    ok(!$f->is_ready, 'Future starts pending');

    Hypersonic::Future::Pool->submit($f, sub {
        my ($a, $b) = @_;
        return $a + $b;
    }, [10, 20]);

    # Give worker thread time to move it to completed queue
    select(undef, undef, undef, 0.1);

    my $processed = Hypersonic::Future::Pool->process_ready;
    ok($processed >= 1, "Processed $processed operations");

    ok($f->is_done, 'Future is done after processing');
    my @result = $f->result;
    is($result[0], 30, 'Future has correct result (10 + 20 = 30)');
}

# Test submit with no args
{
    my $f = Hypersonic::Future->new;

    Hypersonic::Future::Pool->submit($f, sub {
        return 'no args';
    });

    select(undef, undef, undef, 0.1);
    Hypersonic::Future::Pool->process_ready;

    ok($f->is_done, 'Future done with no args');
    is($f->result, 'no args', 'Result correct');
}

# Test submit returning multiple values
{
    my $f = Hypersonic::Future->new;

    Hypersonic::Future::Pool->submit($f, sub {
        return ('a', 'b', 'c');
    }, []);

    select(undef, undef, undef, 0.1);
    Hypersonic::Future::Pool->process_ready;

    ok($f->is_done, 'Future done');
    my @r = $f->result;
    is_deeply(\@r, ['a', 'b', 'c'], 'Multiple return values preserved');
}

Hypersonic::Future::Pool->shutdown;

done_testing;
