/* -*- Mode: C++; tab-width: 4; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 *     Copyright 2010-2013 Couchbase, Inc.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
#ifndef TESTS_MOCK_ENVIRONMENT_H
#define TESTS_MOCK_ENVIRONMENT_H 1

#include "config.h"
#include <gtest/gtest.h>
#include <libcouchbase/couchbase.h>
#include "serverparams.h"


class HandleWrap
{

    friend class MockEnvironment;

public:
    lcb_t getLcb() const {
        return instance;
    }

    void destroy();

    // Don't ever allow copying. C++0x allows = 0, though
    HandleWrap operator= (const HandleWrap &other) {
        fprintf(stderr, "Can't copy this object around!\n");
        abort();
        return HandleWrap();
    }

    HandleWrap() : instance(NULL), iops(NULL) { }
    virtual ~HandleWrap();


private:
    lcb_t instance;
    lcb_io_opt_t iops;
};

class MockEnvironment : public ::testing::Environment
{
public:
    enum ServerVersion {
        VERSION_UNKNOWN = 0,
        VERSION_10 = 1,
        VERSION_20 = 2
    };

    virtual void SetUp();
    virtual void TearDown();

    static MockEnvironment *getInstance(void);
    static void Reset();

    /**
     * Make a connect structure you may utilize to connect to
     * the backend we're running the tests towards.
     *
     * @param crst the create structure to fill in
     * @param io the io ops to use (pass NULL if you don't have a
     *           special io ops you want to use
     */
    void makeConnectParams(lcb_create_st &crst, lcb_io_opt_t io) {
        serverParams.makeConnectParams(crst, io);
    }

    /**
     * Get the number of nodes used in the backend
     */
    int getNumNodes(void) const {
        return numNodes;
    }

    /**
     * Are we currently using a real cluster as the backend, or
     * are we using the mock server.
     *
     * You should try your very best to avoid using this variable, and
     * rather extend the mock server to support the requested feature.
     */
    bool isRealCluster(void) const {
        return realCluster;
    }

    /**
     * Simulate node failover. In this case mock will disable server
     * corresponding given index an push new configuration. No data
     * rebalancing implemented on the mock.
     *
     * @param index the index of the node on the mock
     * @param bucket the name of the bucket
     */
    void failoverNode(int index, std::string bucket = "default");

    /**
     * Simulate node reconvering. In this case mock will enable server
     * corresponding given index an push new configuration. No data
     * rebalancing implemented on the mock.
     *
     * @param index the index of the node on the mock
     * @param bucket the name of the bucket
     */
    void respawnNode(int index, std::string bucket = "default");

    /**
     * Create a connection to the mock/real server.
     *
     * The instance will be initialized with the the connect parameters
     * to either the mock or a real server (just like makeConnectParams),
     * and call lcb_create. The io instance will be stored in the instance
     * cookie so you may grab it from there.
     *
     * You should call lcb_destroy on the instance when you're done
     * using it.
     *
     * @param instance the instane to create
     */
    void createConnection(lcb_t &instance);

    void createConnection(HandleWrap &handle);

    /**
     * Setup mock to split response in two parts: send first "offset" bytes
     * immediately and send the rest after "msecs" milliseconds.
     *
     * @param msecs the number of milliseconds to wait before sending the
     *              rest of the packet.
     * @param offset the number of the bytes to send in first before delay
     */
    void hiccupNodes(int msecs, int offset);

    ServerVersion getServerVersion(void) const {
        return serverVersion;
    }

    void setServerVersion(ServerVersion ver)  {
        serverVersion = ver;
    }

protected:
    /**
     * Protected destructor to make it to a singleton
     */
    MockEnvironment();
    /**
     * Handle to the one and only instance of the mock environment
     */
    static MockEnvironment *instance;

    void bootstrapRealCluster();
    const struct test_server_info *mock;
    ServerParams serverParams;
    int numNodes;
    bool realCluster;
    ServerVersion serverVersion;
    const char *http;
    lcb_io_opt_st *iops;
};

#define LCB_TEST_REQUIRE_CLUSTER_VERSION(v) \
    if (!MockEnvironment::getInstance()->isRealCluster()) {             \
        std::cerr << "Skipping " << __FILE__ << ":" << std::dec << __LINE__; \
        std::cerr << " (need real cluster) " << std::endl; \
        return; \
    } \
    if (MockEnvironment::getInstance()->getServerVersion() < v) {      \
        std::cerr << "Skipping " << __FILE__ << ":" << std::dec << __LINE__; \
        std::cerr << " (test needs higher cluster version)" << std::endl; \
        return; \
    }



#endif
