package JSON::Lines;
use 5.006; use strict; use warnings; our $VERSION = '0.01';
use JSON; use base 'Import::Export';

our ($JSON, $LINES, %EX);
BEGIN {
	$JSON = JSON->new;
	$LINES = qr{ ([\[\{] (?: (?> [^\[\]\{\}]+ ) | (??{ $LINES }) )* [\]\}]) }x;
	%EX = (
		jsonl => [qw/all/]
	);
}

sub jsonl {
	my %args = (scalar @_ == 1 ? %{$_[0]} : @_);
	my $self = __PACKAGE__->new(%args);
	return $self->encode($args{data}) if ($args{encode});
	return $self->decode($args{data}) if ($args{decode});
}

sub new {
	my ($pkg, %args) = (shift, scalar @_ == 1 ? %{$_[0]} : @_);
	my $self = bless {}, $pkg;
	exists $args{$_} && $JSON->$_($args{$_}) for qw/json pretty canonical/;
	$self->{$_} = $args{$_} for qw/error_cb success_cb/;
	$self;
}

sub encode {
	my ($self, @data) = (shift, scalar @_ == 1 ? @{$_[0]} : @_);

	if ($self->{parse_headers}) {
		@data = $self->parse_headers(@data);
	}
	
	my $string;
	for (@data) {
		my $json = eval { $JSON->encode($_) };
		if ($@) {
			if ($self->{error_cb}) {
				$self->{error_cb}->($@);
			} else {
				die $@;
			}
		} else {
			$self->{success_cb}->($@) if $self->{success_cb};
			$string .= ($string && $string !~ m/\n$/ ? "\n" : "") .  $json;
		}
	}
	$self->{string} = $string;

	return $string;
}

sub add_line {
	my $string = $_[0]->{string};
	my $add = $_[0]->encode([$_[1]]);
	$_[0]->{string} = ($string ? "$string\n" : "") . $add;
	$_[0]->{string};
}

sub clear_stream {
	$_[0]->{string} = '';
}

sub decode {
	my ($self, $string) = @_;

	my @lines;
	for ($string =~ m/$LINES/g) {
		my $struct = eval { $JSON->decode($_) };
		if ($@) {
			if ($self->{error_cb}) {
				$self->{error_cb}->($@);
			} else {
				die $@;
			}
		} else {
			$self->{success_cb}->($struct) if $self->{success_cb};
			push @lines, $struct;
		}
	}

	return wantarray ? @lines : \@lines;
}

=head1 NAME

JSON::Lines - Parse JSONLines with perl.

=head1 VERSION

Version 0.01

=cut

=head1 SYNOPSIS

Quick summary of what the module does.

	use JSON::Lines;

	my $jsonl = JSON::Lines->new();

	my @data = (
		["Name", "Session", "Score", "Completed"],
		["Gilbert", "2013", 24, true],
		["Alexa", "2013", 29, true],
		["May", "2012B", 14, false],
		["Deloise", "2012A", 19, true]
	);

	my $string = $jsonl->encode(@data);

	# ["Name", "Session", "Score", "Completed"]
	# ["Gilbert", "2013", 24, true]
	# ["Alexa", "2013", 29, true]
	# ["May", "2012B", 14, false]
	# ["Deloise", "2012A", 19, true]  

	...

	use JSON::Lines qw/jsonl/;

	my $data = [
		{
			"name" => "Gilbert", 
			"wins" => [
				["straight", "7♣"], 
				["one pair", "10♥"]
			]
		},
		{
			"name" => "Alexa", 
			"wins" => [
				["two pair", "4♠"], 
				["two pair", "9♠"]
			]
		},
		{
			"name" => "May", 
			"wins": []
		},
		{
			"name" => "Deloise", 
			"wins" => [
				["three of a kind", "5♣"]
			]
		}
	];

	my $string = jsonl( canonical => 1, encode => 1, data => $data );

	# {"name": "Gilbert", "wins": [["straight", "7♣"], ["one pair", "10♥"]]}
	# {"name": "Alexa", "wins": [["two pair", "4♠"], ["two pair", "9♠"]]}
	# {"name": "May", "wins": []}
	# {"name": "Deloise", "wins": [["three of a kind", "5♣"]]}
	
=head1 DESCRIPTION

JSON Lines is a convenient format for storing structured data that may be processed one record at a time. It works well with unix-style text processing tools and shell pipelines. It's a great format for log files. It's also a flexible format for passing messages between cooperating processes.

L<https://jsonlines.org/>

=head1 EXPORT

=head2 jsonl

	my $string = jsonl( canonical => 1, encode => 1, data => $data );

	my $aoh = jsonl( decode => 1, data => $string )

=head1 SUBROUTINES/METHODS

=head2 new

Instantiate a new JSON::Lines object.

	my $jsonl = JSON::Lines->new

=head2 encode

Encode a perl struct into a json lines string.

	$jsonl->encode( %args );

=head2 decode

Decode a json lines string into a perl struct.

	$jsonl->decode( %args );

=head2 add_line

Add a new line to the current JSON::Lines stream.

	$jsonl->add_line($line);

=head2 clear_stream

Clear the current JSON::Lines stream.

	$jsonl->clear_string;

=head1 AUTHOR

LNATION, C<< <thisusedtobeanemail at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-json-lines at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=JSON-Lines>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

	perldoc JSON::Lines

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<https://rt.cpan.org/NoAuth/Bugs.html?Dist=JSON-Lines>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/JSON-Lines>

=item * Search CPAN

L<https://metacpan.org/release/JSON-Lines>

=back

=head1 ACKNOWLEDGEMENTS

=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2020 by LNATION.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut

1; # End of JSON::Lines
